<?php
namespace ModernPress\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Text_Shadow;

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Image Box widget class.
 */
class Image_Box_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * @return string
     */
    public function get_name()
    {
        return 'Image_Box_Widget';
    }

    /**
     * Get widget title.
     *
     * @return string
     */
    public function get_title()
    {
        return esc_html__('ModernPress Image Box', 'modernpress-addons');
    }

    /**
     * Get widget icon.
     *
     * @return string
     */
    public function get_icon()
    {
        return 'eicon-image-box';
    }

    /**
     * Get widget categories.
     *
     * @return array
     */
    public function get_categories()
    {
        return ['modernpress'];
    }

    /**
     * Register widget controls.
     *
     * Written in PHP and used to generate the widget settings form in the Elementor editor.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        /**
         * Content Section
         */
        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        // Add control to choose between Image or Icon
        $this->add_control(
            'media_type',
            [
                'label' => __('Media Type', 'modernpress-addons'),
                'type' => Controls_Manager::SELECT,
                'default' => 'image',
                'options' => [
                    'image' => __('Image', 'modernpress-addons'),
                    'icon' => __('Icon', 'modernpress-addons'),
                ],
            ]
        );

        // Image control (conditional)
        $this->add_control(
            'image',
            [
                'label' => __('Choose Image', 'modernpress-addons'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => plugin_dir_url(dirname(__DIR__)) . 'assets/images/placeholder.png', // Adjust path based on location
                ],
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );

        // Icon control (conditional)
        $this->add_control(
            'icon',
            [
                'label' => __('Choose Icon', 'modernpress-addons'),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'solid',
                ],
                'condition' => [
                    'media_type' => 'icon',
                ],
            ]
        );


        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'image',
                'default' => 'large',
            ]
        );

        $this->add_control(
            'imagebox_title',
            [
                'label' => esc_html__('Title', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__('Top Figma plugins to use for mobile responsive design', 'modernpress-addons'),
                'placeholder' => esc_html__('Enter your title', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'imagebox_description',
            [
                'label' => esc_html__('Description', 'modernpress-addons'),
                'type' => Controls_Manager::TEXTAREA,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__('Laoreet nunc tellus quis luctus Feugiat accumsan sit pretium.', 'modernpress-addons'),
                'placeholder' => esc_html__('Enter your description', 'modernpress-addons'),
                'rows' => 5,
            ]
        );

        $this->add_control(
            'imagebox_link_mode',
            [
                'label' => esc_html__('Link Mode', 'modernpress-addons'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Full', 'modernpress-addons'),
                'label_off' => esc_html__('Below', 'modernpress-addons'),
                'return_value' => 'yes',
                'default' => 'below',
            ]
        );

        $this->add_control(
            'imagebox_link',
            [
                'label' => esc_html__('Card Link', 'modernpress-addons'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'modernpress-addons'),
                'default' => [
                    'url' => '',
                ],
                'condition' => [
                    'imagebox_link_mode' => 'yes', // Show this control when linking the entire card
                ],
            ]
        );

        $this->add_control(
            'imagebox_link_text',
            [
                'label' => esc_html__('Link Text', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Read More', 'modernpress-addons'),
                'placeholder' => esc_html__('Learn More', 'modernpress-addons'),
                'condition' => [
                    'imagebox_link_mode!' => 'yes', // Show this control when linking below the description
                ],
            ]
        );

        $this->add_control(
            'imagebox_below_link',
            [
                'label' => esc_html__('Below Description Link', 'modernpress-addons'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'modernpress-addons'),
                'default' => [
                    'url' => '#',
                ],
                'condition' => [
                    'imagebox_link_mode!' => 'yes', // Show this control when linking below the description
                ],
            ]
        );




        $this->end_controls_section();

        /**
         * Style Section - Box
         */
        $this->start_controls_section(
            'section_style_box',
            [
                'label' => esc_html__('Box', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        // Image Position
        $this->add_control(
            'image_position',
            [
                'label' => esc_html__('Image Position', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'top',
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'modernpress-addons'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'top' => [
                        'title' => esc_html__('Top', 'modernpress-addons'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'modernpress-addons'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'prefix_class' => 'modernpress-image-position-',
            ]
        );


        // Content Alignment
        $this->add_responsive_control(
            'imagebox_content_align',
            [
                'label' => esc_html__('Content Alignment', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('start', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('end', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'start',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box' => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-image-box-icon' => 'justify-content: {{VALUE}}; display: flex;',
                    '{{WRAPPER}} .modernpress-image-box-content' => 'text-align: {{VALUE}};',
                ],
            ]
        );
        // Vertical Alignment (conditionally shown)
        $this->add_responsive_control(
            'imagebox_vertical_align',
            [
                'label' => esc_html__('Vertical Alignment', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__('Top', 'modernpress-addons'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'center' => [
                        'title' => esc_html__('Middle', 'modernpress-addons'),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'flex-end' => [
                        'title' => esc_html__('Bottom', 'modernpress-addons'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-icon' => 'align-items: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-image-box-content' => 'display: flex; flex-direction: column; justify-content: {{VALUE}};',
                ],
                'condition' => [
                    'image_position' => ['left', 'right'], // Show only for left or right position
                ],
            ]
        );


        // Icon Spacing
        $this->add_responsive_control(
            'image_spacing',
            [
                'label' => esc_html__('Image Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-icon img' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'image_position' => 'top', // Only show for center alignment
                ],
            ]
        );
        // Icon Spacing for Left/Right alignment
        $this->add_responsive_control(
            'image_spacing_left',
            [
                'label' => esc_html__('Image Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    // Horizontal spacing when content alignment is left or right
                    '{{WRAPPER}} .modernpress-image-box-icon' => 'margin-right: {{SIZE}}{{UNIT}};'
                ],
                'condition' => [
                    'image_position' => 'left', // Only show when content is not aligned to center
                ],
            ]
        );
        $this->add_responsive_control(
            'image_spacing_right',
            [
                'label' => esc_html__('Image Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    // Horizontal spacing when content alignment is left or right
                    '{{WRAPPER}} .modernpress-image-box-icon' => 'margin-left: {{SIZE}}{{UNIT}};'
                ],
                'condition' => [
                    'image_position' => 'right', // Only show when content is not aligned to center
                ],
            ]
        );

        // Content Spacing
        $this->add_responsive_control(
            'imagebox_content_spacing',
            [
                'label' => esc_html__('Content Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-description' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .modernpress-image-box',
            ]
        );
        $this->add_responsive_control(
            'box_border_radius',
            [
                'label' => esc_html__('Border Radius', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'default' => [
                    'top' => '8',
                    'right' => '8',
                    'bottom' => '8',
                    'left' => '8',
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();

        /**
         * Style Section - Icon
         */
        $this->start_controls_section(
            'section_style_image',
            [
                'label' => esc_html__('Image', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('icon_colors');

        $this->start_controls_tab(
            'imagebox_filters_normal',
            [
                'label' => esc_html__('Normal', 'modernpress-addons'),
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Css_Filter::get_type(),
            [
                'name' => 'custom_css_filters',
                'selector' => '{{WRAPPER}} .modernpress-image-box-icon img',
                'condition' => [
                    'media_type' => 'image',
                ],
            ]

        );

        $this->add_responsive_control(
            'image_opacity',
            [
                'label' => esc_html__('Opacity', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%'],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-icon img' => 'opacity: {{SIZE}}%;',
                ],
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );


        $this->end_controls_tab();

        $this->start_controls_tab(
            'imagebox_filters_hover',
            [
                'label' => esc_html__('Hover', 'modernpress-addons'),
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Css_Filter::get_type(),
            [
                'name' => 'custom_css_filters_hover',
                'selector' => '{{WRAPPER}}:hover .modernpress-image-box-icon img',
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_opacity_hover',
            [
                'label' => esc_html__('Opacity', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%'],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}}:hover .modernpress-image-box-icon img' => 'opacity: {{SIZE}}%;',
                ],
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'image_width',
            [
                'label' => esc_html__('Image Width', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'separator' => 'before',
                'size_units' => ['px', '%', 'vw'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 1000,
                    ],
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 300,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box img' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => esc_html__('Size', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'separator' => 'before',
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 6,
                        'max' => 300,
                    ],
                    'em' => [
                        'min' => 0.1,
                        'max' => 20,
                    ],
                    'rem' => [
                        'min' => 0.1,
                        'max' => 20,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'media_type' => 'icon', // Only show for center alignment
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .modernpress-image-box-icon img',
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'default' => [
                    'top' => '0',
                    'right' => '0',
                    'bottom' => '0',
                    'left' => '0',
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-icon img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'media_type' => 'image',
                ],
            ]
        );


        $this->end_controls_section();
        /**
         * Style Section - Content
         */
        $this->start_controls_section(
            'section_style_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'imagebox_title_color',
            [
                'label' => esc_html__('Title Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'imagebox_title_typography',
                'selector' => '{{WRAPPER}} .modernpress-image-box-title',
            ]
        );
        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'imagebox_title_text_shadow',
                'label' => esc_html__('Text Shadow', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-image-box-title',
            ]
        );


        $this->add_control(
            'imagebox_description_color',
            [
                'label' => esc_html__('Description Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-image-box-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'imagebox_description_typography',
                'selector' => '{{WRAPPER}} .modernpress-image-box-description',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'imagebox_description_text_shadow',
                'label' => esc_html__('Text Shadow', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-image-box-description',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Add render attributes
        $this->add_render_attribute('wrapper', 'class', 'modernpress-image-box');
        $this->add_render_attribute('image-wrapper', 'class', 'modernpress-image-box-icon');
        $this->add_render_attribute('content', 'class', 'modernpress-image-box-content');
        $this->add_render_attribute('title', 'class', 'modernpress-image-box-title');
        $this->add_render_attribute('description', 'class', 'modernpress-image-box-description');

        // Add position class for icon
        $this->add_render_attribute('wrapper', 'class', 'modernpress-position-' . $settings['image_position']);  // Add the position class here

        // Link attributes
        $has_link = !empty($settings['imagebox_link']['url']);
        if ($has_link) {
            $this->add_link_attributes('link', $settings['imagebox_link']);
            $this->add_render_attribute('link', 'class', 'modernpress-image-box-link');
        }
        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <?php if ($has_link): ?>
                <a <?php $this->print_render_attribute_string('link'); ?>>
                <?php endif; ?>

                <div <?php $this->print_render_attribute_string('image-wrapper'); ?>>
                    <?php
                    // Check media type and render appropriately
                    if ($settings['media_type'] === 'image' && !empty($settings['image']['url'])) {
                        // Escape the output of image HTML tag
                        echo wp_kses_post(\Elementor\Group_Control_Image_Size::get_attachment_image_html($settings, 'image'));
                    } elseif ($settings['media_type'] === 'icon' && !empty($settings['icon']['value'])) {
                        \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']);
                    }
                    ?>
                </div>

                <div <?php $this->print_render_attribute_string('content'); ?>>
                    <?php if (!empty($settings['imagebox_title'])): ?>
                        <h3 <?php $this->print_render_attribute_string('title'); ?>>
                            <?php echo wp_kses($settings['imagebox_title'], []); ?>
                        </h3>
                    <?php endif; ?>

                    <?php if (!empty($settings['imagebox_description'])): ?>
                        <p <?php $this->print_render_attribute_string('description'); ?>>
                            <?php echo wp_kses_post($settings['imagebox_description']); ?>
                        </p>
                    <?php endif; ?>
                    <?php

                    // Card content here
                    if (!empty($settings['imagebox_below_link']['url'])) {
                        echo '<a href="' . esc_url($settings['imagebox_below_link']['url']) . '" class="modernpress-link">';
                        echo esc_html($settings['imagebox_link_text']);
                        ?>
                        <svg width="7" height="12" viewBox="0 0 7 12" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M0.750035 1.5L5.25 6.00004L0.75 10.5" stroke="#0D3FB0" stroke-width="1.5"
                                stroke-miterlimit="16" />
                        </svg>
                        <?php

                        echo '</a>';
                    }
                    ?>
                </div>
                <?php if ($has_link): ?>
                </a>
            <?php endif; ?>
        </div>
        <?php
    }
}

