<?php
namespace ModernPress\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Icon Box widget class.
 */
class Icon_Box_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * @return string
     */
    public function get_name()
    {
        return 'Icon_Box_Widget';
    }

    /**
     * Get widget title.
     *
     * @return string
     */
    public function get_title()
    {
        return esc_html__('ModernPress Icon Box', 'modernpress-addons');
    }

    /**
     * Get widget icon.
     *
     * @return string
     */
    public function get_icon()
    {
        return 'eicon-image-box';
    }

    /**
     * Get widget categories.
     *
     * @return array
     */
    public function get_categories()
    {
        return ['modernpress'];
    }

    /**
     * Register widget controls.
     *
     * Written in PHP and used to generate the widget settings form in the Elementor editor.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        /**
         * Content Section
         */
        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'selected_icon',
            [
                'label' => esc_html__('Icon', 'modernpress-addons'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__('Icon Box Title', 'modernpress-addons'),
                'placeholder' => esc_html__('Enter your title', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'description',
            [
                'label' => esc_html__('Description', 'modernpress-addons'),
                'type' => Controls_Manager::TEXTAREA,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__('This is a description text.', 'modernpress-addons'),
                'placeholder' => esc_html__('Enter your description', 'modernpress-addons'),
                'rows' => 5,
            ]
        );

        $this->add_control(
            'link',
            [
                'label' => esc_html__('Link', 'modernpress-addons'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'modernpress-addons'),
                'default' => [
                    'url' => '',
                ],
            ]
        );

        $this->add_control(
            'link_text',
            [
                'label' => esc_html__('Link Text', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'modernpress-addons'),
                'placeholder' => esc_html__('Learn More', 'modernpress-addons'),
                'condition' => [
                    'link[url]!' => '',
                ],
            ]
        );



        $this->end_controls_section();

        /**
         * Style Section - Box
         */
        $this->start_controls_section(
            'section_style_box',
            [
                'label' => esc_html__('Box', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        // Icon Position
        $this->add_control(
            'position',
            [
                'label' => esc_html__('Icon Position', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'top',
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'modernpress-addons'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'top' => [
                        'title' => esc_html__('Top', 'modernpress-addons'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'modernpress-addons'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'prefix_class' => 'modernpress-position-',
            ]
        );

        // Content Alignment
        $this->add_responsive_control(
            'content_align',
            [
                'label' => esc_html__('Content Alignment', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        // Icon Spacing
        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => esc_html__('Icon Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-icon' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        // Content Spacing
        $this->add_responsive_control(
            'content_spacing',
            [
                'label' => esc_html__('Content Spacing', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-description' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        /**
         * Style Section - Icon
         */
        $this->start_controls_section(
            'section_style_icon',
            [
                'label' => esc_html__('Icon', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('icon_colors');

        $this->start_controls_tab(
            'icon_colors_normal',
            [
                'label' => esc_html__('Normal', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__('Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-icon-box-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_colors_hover',
            [
                'label' => esc_html__('Hover', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'icon_hover_color',
            [
                'label' => esc_html__('Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover .modernpress-icon-box-icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}}:hover .modernpress-icon-box-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => esc_html__('Size', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'separator' => 'before',
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 6,
                        'max' => 300,
                    ],
                    'em' => [
                        'min' => 0.1,
                        'max' => 20,
                    ],
                    'rem' => [
                        'min' => 0.1,
                        'max' => 20,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .modernpress-icon-box-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();
        /**
         * Style Section - Content
         */
        $this->start_controls_section(
            'section_style_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .modernpress-icon-box-title',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__('Description Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .modernpress-icon-box-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .modernpress-icon-box-description',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Add render attributes
        $this->add_render_attribute('wrapper', 'class', 'modernpress-icon-box');
        $this->add_render_attribute('icon-wrapper', 'class', 'modernpress-icon-box-icon');
        $this->add_render_attribute('content', 'class', 'modernpress-icon-box-content');
        $this->add_render_attribute('title', 'class', 'modernpress-icon-box-title');
        $this->add_render_attribute('description', 'class', 'modernpress-icon-box-description');

        // Add position class for icon
        $this->add_render_attribute('wrapper', 'class', 'modernpress-position-' . $settings['position']);  // Add the position class here

        // Link attributes
        $has_link = !empty($settings['link']['url']);
        if ($has_link) {
            $this->add_link_attributes('link', $settings['link']);
            $this->add_render_attribute('link', 'class', 'modernpress-icon-box-link');
        }
        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <?php if ($has_link): ?>
                <a <?php $this->print_render_attribute_string('link'); ?>>
                <?php endif; ?>

                <div <?php $this->print_render_attribute_string('icon-wrapper'); ?>>
                    <?php \Elementor\Icons_Manager::render_icon($settings['selected_icon'], ['aria-hidden' => 'true']); ?>
                </div>

                <div <?php $this->print_render_attribute_string('content'); ?>>
                    <?php if (!empty($settings['title'])): ?>
                        <h3 <?php $this->print_render_attribute_string('title'); ?>>
                            <?php echo wp_kses($settings['title'], []); ?>
                        </h3>
                    <?php endif; ?>

                    <?php if (!empty($settings['description'])): ?>
                        <p <?php $this->print_render_attribute_string('description'); ?>>
                            <?php echo wp_kses_post($settings['description']); ?>
                        </p>
                    <?php endif; ?>

                    <?php if ($has_link && !empty($settings['link_text'])): ?>
                        <span class="modernpress-icon-box-link-text">
                            <?php echo esc_html($settings['link_text']); ?>
                        </span>
                    <?php endif; ?>
                </div>

                <?php if ($has_link): ?>
                </a>
            <?php endif; ?>
        </div>
        <?php
    }
}

