<?php

namespace ModernPress\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Dual Heading widget class.
 */
class ModernPress_Dual_Heading extends Widget_Base
{

    /**
     * Get widget name.
     *
     * @return string
     */
    public function get_name()
    {
        return 'Modernpress_Dual_Heading';
    }

    /**
     * Get widget title.
     *
     * @return string
     */
    public function get_title()
    {
        return __('ModernPress Dual Heading', 'modernpress-addons');
    }

    /**
     * Get widget icon.
     *
     * @return string
     */
    public function get_icon()
    {
        return 'eicon-t-letter';
    }

    /**
     * Get widget categories.
     *
     * @return array
     */
    public function get_categories()
    {
        return ['modernpress'];
    }

    public function get_keywords()
    {
        return ['dual heading', 'heading', 'modernpress-addons'];
    }

    /**
     * Register widget controls.
     *
     * Written in PHP and used to generate the widget settings form in the Elementor editor.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        // First Heading
        $this->start_controls_section(
            'section_first_heading',
            [
                'label' => __('First Heading', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'first_heading_text',
            [
                'label' => __('Text', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Modern', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'first_heading_rotation',
            [
                'label' => __('Rotation (degrees)', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -360,
                        'max' => 360,
                    ],
                ],

            ]
        );


        $this->end_controls_section();

        // Second Heading
        $this->start_controls_section(
            'section_second_heading',
            [
                'label' => __('Second Heading', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'second_heading_text',
            [
                'label' => __('Text', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Press', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'second_heading_rotation',
            [
                'label' => __('Rotation (degrees)', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'unit' => 'deg',  // Default unit for degrees
                    'size' => 0,  // Default rotation value (0 degrees)
                ],
                'range' => [
                    'deg' => [
                        'min' => -360,
                        'max' => 360,
                        'step' => 1,  // Step value for the slider
                    ],
                ],
            ]
        );


        $this->end_controls_section();

        // Display and Alignment
        $this->start_controls_section(
            'section_display',
            [
                'label' => __('Display and Alignment', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'html_tag',
            [
                'label' => __('HTML Tag', 'modernpress-addons'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'DIV',
                    'span' => 'SPAN',
                ],
                'default' => 'h2',
            ]
        );

        $this->add_control(
            'display_type',
            [
                'label' => __('Display', 'modernpress-addons'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'inline' => __('Inline', 'modernpress-addons'),
                    'block' => __('Block', 'modernpress-addons'),
                ],
                'default' => 'inline',
            ]
        );

        $this->add_responsive_control(
            'alignment',
            [
                'label' => __('Alignment', 'modernpress-addons'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'modernpress-addons'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->end_controls_section();

        // Link
        $this->start_controls_section(
            'section_link',
            [
                'label' => __('Link', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'enable_link',
            [
                'label' => __('Enable Link', 'modernpress-addons'),
                'type' => Controls_Manager::SWITCHER,
                'default' => '',
            ]
        );

        $this->add_control(
            'link_url',
            [
                'label' => __('Link URL', 'modernpress-addons'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'modernpress-addons'),
                'condition' => [
                    'enable_link' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Background Text
        $this->start_controls_section(
            'section_background_text',
            [
                'label' => __('Background Text', 'modernpress-addons'),
            ]
        );

        $this->add_control(
            'enable_background_text',
            [
                'label' => __('Enable Background Text', 'modernpress-addons'),
                'type' => Controls_Manager::SWITCHER,
                'default' => '',
            ]
        );

        $this->add_control(
            'background_text_content',
            [
                'label' => __('Text', 'modernpress-addons'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Background', 'modernpress-addons'),
                'condition' => [
                    'enable_background_text' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'background_text_horizontal_offset',
            [
                'label' => __('Horizontal Offset (px)', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -1000,
                        'max' => 1000,
                    ],
                ],
                'condition' => [
                    'enable_background_text' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'background_text_vertical_offset',
            [
                'label' => __('Vertical Offset (px)', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -1000,
                        'max' => 1000,
                    ],
                ],
                'condition' => [
                    'enable_background_text' => 'yes',
                ],
            ]
        );


        $this->add_control(
            'background_text_rotation',
            [
                'label' => __('Rotation (degrees)', 'modernpress-addons'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => 0,
                ],
                'range' => [
                    'px' => [
                        'min' => -360,
                        'max' => 360,
                        'step' => 1,
                    ],
                ],
                'condition' => [
                    'enable_background_text' => 'yes',
                ],
            ]
        );


        $this->end_controls_section();
        // First Heading Style
        $this->start_controls_section(
            'style_first_heading',
            [
                'label' => __('First Heading Style', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        // Typography
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'first_heading_typography',
                'label' => __('Typography', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-first-heading',
            ]
        );

        // Text Color
        $this->add_control(
            'first_heading_text_color',
            [
                'label' => __('Text Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-first-heading' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'background',
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .modernpress-first-heading span',
            ]
        );

        // Border Type
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .modernpress-first-heading span',
            ]
        );

        $this->add_responsive_control(
            'first_heading_border_radius',
            [
                'label' => __('Border Radius', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-first-heading span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );




        // Margin
        $this->add_responsive_control(
            'first_heading_margin',
            [
                'label' => __('Margin', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-first-heading span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Padding
        $this->add_responsive_control(
            'first_heading_padding',
            [
                'label' => __('Padding', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-first-heading span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
        // First Heading Style
        $this->start_controls_section(
            'style_second_heading',
            [
                'label' => __('Second Heading Style', 'modernpress-addons'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        // Typography
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'second_heading_typography',
                'label' => __('Typography', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-second-heading',
            ]
        );

        // Text Color
        $this->add_control(
            'second_heading_text_color',
            [
                'label' => __('Text Color', 'modernpress-addons'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-second-heading' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'background_second_heading',
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .modernpress-second-heading span',
            ]
        );

        // Border Type
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'border_second_heading',
                'selector' => '{{WRAPPER}} .modernpress-second-heading span',
            ]
        );

        $this->add_responsive_control(
            'second_heading_border_radius',
            [
                'label' => __('Border Radius', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-second-heading span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );




        // Margin
        $this->add_responsive_control(
            'second_heading_margin',
            [
                'label' => __('Margin', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-second-heading span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Padding
        $this->add_responsive_control(
            'second_heading_padding',
            [
                'label' => __('Padding', 'modernpress-addons'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-second-heading span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Prepare alignment class for the container
        $alignment_class = 'align-' . esc_attr($settings['alignment']);

        // Adjust rotation for both headings
        $this->add_render_attribute('first_heading_text', 'style', sprintf('transform: rotate(%ddeg);', floatval($settings['first_heading_rotation']['size'])));
        $this->add_render_attribute('second_heading_text', 'style', sprintf('transform: rotate(%ddeg);', floatval($settings['second_heading_rotation']['size'])));

        // HTML Tag - Ensure it's a valid tag
        $allowed_tags = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'span'];
        $tag = in_array($settings['html_tag'], $allowed_tags) ? $settings['html_tag'] : 'h2';

        // Determine the display class based on the display_type setting
        $display_class = $settings['display_type'] === 'inline' ? 'inline' : 'block';

        echo '<div class="modernpress-dual-heading-wrapper ' . esc_attr($alignment_class) . '">';

        // If link is enabled, wrap the headings in an <a> tag
        if (!empty($settings['enable_link']) && !empty($settings['link_url']['url'])) {
            printf(
                '<a href="%1$s" target="%2$s" rel="%3$s" class="%4$s" style="width:100%%;">',
                esc_url($settings['link_url']['url']),
                esc_attr($settings['link_url']['is_external'] ? '_blank' : '_self'),
                esc_attr($settings['link_url']['nofollow'] ? 'nofollow' : ''),
                esc_attr($display_class)
            );
        } else {
            printf(
                '<div class="modernpress-dual-heading" style="display: %s;">',
                esc_attr($settings['display_type'])
            );
        }

        // Background Text
        if (!empty($settings['enable_background_text'])) {
            $horizontal_offset = isset($settings['background_text_horizontal_offset']['size']) ? floatval($settings['background_text_horizontal_offset']['size']) : 0;
            $vertical_offset = isset($settings['background_text_vertical_offset']['size']) ? floatval($settings['background_text_vertical_offset']['size']) : 0;
            $rotation = isset($settings['background_text_rotation']['size']) ? floatval($settings['background_text_rotation']['size']) : 0;

            printf(
                '<div class="modernpress-background-text" style="position: absolute; left: %1$spx; top: %2$spx; transform: rotate(%3$sdeg);">%4$s</div>',
                esc_attr($horizontal_offset),
                esc_attr($vertical_offset),
                esc_attr($rotation),
                esc_html($settings['background_text_content'])
            );
        }

        // First Heading
        printf(
            '<%1$s class="modernpress-first-heading" style="display: %2$s;"><span %3$s>%4$s</span></%1$s>',
            esc_html($tag),
            esc_attr($settings['display_type']),
            esc_attr($this->get_render_attribute_string('first_heading_text')), // Escape here
            esc_html($settings['first_heading_text'])
        );

        // Second Heading
        printf(
            '<%1$s class="modernpress-second-heading" style="display: %2$s;"><span %3$s>%4$s</span></%1$s>',
            esc_html($tag),
            esc_attr($settings['display_type']),
            esc_attr($this->get_render_attribute_string('second_heading_text')), // Escape here
            esc_html($settings['second_heading_text'])
        );

        // Close link if enabled
        if (!empty($settings['enable_link']) && !empty($settings['link_url']['url'])) {
            echo '</a>';
        } else {
            echo '</div>';
        }
        echo '</div>';
    }

}

