<?php
namespace ModernPress\Includes;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class ModernPress_Ajax
{
    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->init_hooks();
    }

    private function init_hooks()
    {
        // Ajax for logged in users
        add_action('wp_ajax_modernpress_live_search', array($this, 'handle_live_search'));
        add_action('wp_ajax_modernpress_get_categories', array($this, 'get_categories'));

        // Ajax for non-logged in users
        add_action('wp_ajax_nopriv_modernpress_live_search', array($this, 'handle_live_search'));
        add_action('wp_ajax_nopriv_modernpress_get_categories', array($this, 'get_categories'));
    }

    public function get_categories()
    {
        try {
            // Verify nonce
            if (!check_ajax_referer('modernpress_search_nonce', 'nonce', false)) {
                throw new \Exception('Invalid security token.');
            }

            $categories = get_categories(array(
                'orderby' => 'name',
                'order' => 'ASC',
                'hide_empty' => true
            ));

            $category_list = array();
            foreach ($categories as $category) {
                $category_list[] = array(
                    'id' => $category->term_id,
                    'name' => $category->name,
                    'count' => $category->count
                );
            }

            wp_send_json_success(array(
                'categories' => $category_list
            ));

        } catch (\Exception $e) {
            wp_send_json_error(array(
                'message' => $e->getMessage()
            ));
        }
    }

    public function handle_live_search()
    {
        try {
            // Verify nonce
            if (!check_ajax_referer('modernpress_search_nonce', 'nonce', false)) {
                throw new \Exception('Invalid security token.');
            }

            // Get and sanitize inputs
            $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';
            $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 5;

            // Get and sanitize filters
            $filters = array();
            if (isset($_POST['filters']) && is_array($_POST['filters'])) {
                // Sanitize the entire filters array first
                $raw_filters = map_deep(wp_unslash($_POST['filters']), 'sanitize_text_field');
                // Additional type checking for security
                $filters = array(
                    'postType' => isset($raw_filters['postType']) ? sanitize_text_field($raw_filters['postType']) : 'any',
                    'category' => isset($raw_filters['category']) ? sanitize_text_field($raw_filters['category']) : 'all',
                    'date' => isset($raw_filters['date']) ? sanitize_text_field($raw_filters['date']) : 'all'
                );
            }


            // Build search query args
            $args = array(
                'post_status' => 'publish',
                'posts_per_page' => $limit,
                's' => $query,
                'orderby' => 'relevance',
                'order' => 'DESC'
            );

            // Apply post type filter
            if (!empty($filters['postType']) && $filters['postType'] !== 'any') {
                $args['post_type'] = $filters['postType'];
            } else {
                $args['post_type'] = ['post', 'page'];
            }

            // Apply category filter
            if (!empty($filters['category']) && $filters['category'] !== 'all') {
                $args['cat'] = intval($filters['category']);
            }

            // Apply date filter
            if (!empty($filters['date']) && $filters['date'] !== 'all') {
                $date_query = [];

                switch ($filters['date']) {
                    case 'today':
                        $date_query = [
                            'after' => '1 day ago'
                        ];
                        break;
                    case 'week':
                        $date_query = [
                            'after' => '1 week ago'
                        ];
                        break;
                    case 'month':
                        $date_query = [
                            'after' => '1 month ago'
                        ];
                        break;
                    case 'year':
                        $date_query = [
                            'after' => '1 year ago'
                        ];
                        break;
                }

                if (!empty($date_query)) {
                    $args['date_query'] = $date_query;
                }
            }

            // Allow filtering of search query args
            $args = apply_filters('modernpress_search_query_args', $args);

            // Perform search
            $search_query = new \WP_Query($args);
            $results = array();

            if ($search_query->have_posts()) {
                while ($search_query->have_posts()) {
                    $search_query->the_post();

                    // Get featured image
                    $thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'thumbnail');
                    if (!$thumbnail_url) {
                        $thumbnail_url = ''; // You can set a default image URL here
                    }

                    // Get post categories
                    $categories = get_the_category();
                    $category_list = array();
                    if (!empty($categories)) {
                        foreach ($categories as $category) {
                            $category_list[] = array(
                                'name' => $category->name,
                                'url' => get_category_link($category->term_id)
                            );
                        }
                    }

                    // Build result item
                    $results[] = array(
                        'id' => get_the_ID(),
                        'title' => get_the_title(),
                        'url' => get_permalink(),
                        'excerpt' => wp_trim_words(get_the_excerpt(), 20),
                        'thumbnail' => $thumbnail_url,
                        'date' => get_the_date(),
                        'categories' => $category_list,
                        'author' => array(
                            'name' => get_the_author(),
                            'url' => get_author_posts_url(get_the_author_meta('ID')),
                            'avatar' => get_avatar_url(get_the_author_meta('ID'), array('size' => 32))
                        )
                    );
                }
                wp_reset_postdata();
            }

            // Allow filtering of search results
            $results = apply_filters('modernpress_search_results', $results);

            // Send success response
            wp_send_json_success(array(
                'results' => $results,
                'found_posts' => $search_query->found_posts,
                'max_pages' => $search_query->max_num_pages
            ));

        } catch (\Exception $e) {
            wp_send_json_error(array(
                'message' => $e->getMessage()
            ));
        }
    }
}

// Initialize the class
ModernPress_Ajax::get_instance();