jQuery(document).ready(function($) {
    let searchTimeout;
    let activeFilters = {
        postType: 'post',
        category: 'all',
        date: 'all'
    };

    function initModernPressSearch() {
        const searchContainers = $('.modernpress-search-container');

        searchContainers.each(function() {
            const container = $(this);
            const searchInput = container.find('.modernpress-search-input');
            const resultsContainer = container.find('.modernpress-search-results');
            const filterIcon = container.find('.filter-icon');
            
            // Create and append filter dropdown
            const filterDropdown = $(`
                <div class="modernpress-filter-dropdown">
                    <div class="filter-section">
                        <h4>Post Type</h4>
                        <select class="filter-post-type">
                            <option value="post">Posts</option>
                            <option value="page">Pages</option>
                            <option value="any">All Types</option>
                        </select>
                    </div>
                    <div class="filter-section">
                        <h4>Category</h4>
                        <select class="filter-category">
                            <option value="all">All Categories</option>
                        </select>
                    </div>
                    <div class="filter-section">
                        <h4>Date</h4>
                        <select class="filter-date">
                            <option value="all">All Time</option>
                            <option value="today">Today</option>
                            <option value="week">This Week</option>
                            <option value="month">This Month</option>
                            <option value="year">This Year</option>
                        </select>
                    </div>
                    <div class="filter-actions">
                        <button class="apply-filters">Apply Filters</button>
                        <button class="reset-filters">Reset</button>
                    </div>
                </div>
            `).insertAfter(filterIcon);

            // Load categories dynamically
            $.ajax({
                url: modernpressSearch.ajaxurl,
                type: 'POST',
                data: {
                    action: 'modernpress_get_categories',
                    nonce: modernpressSearch.nonce
                },
                success: function(response) {
                    if (response.success && response.data.categories) {
                        const categorySelect = container.find('.filter-category');
                        response.data.categories.forEach(category => {
                            categorySelect.append(`<option value="${category.id}">${category.name}</option>`);
                        });
                    }
                }
            });

            // Toggle filter dropdown
            filterIcon.on('click', function(e) {
                e.stopPropagation();
                filterDropdown.toggleClass('active');
            });

            // Handle filter changes
            container.find('.apply-filters').on('click', function() {
                activeFilters = {
                    postType: container.find('.filter-post-type').val(),
                    category: container.find('.filter-category').val(),
                    date: container.find('.filter-date').val()
                };
                filterDropdown.removeClass('active');
                if (searchInput.val().length >= 2) {
                    performSearch(searchInput.val(), container);
                }
            });

            // Reset filters
            container.find('.reset-filters').on('click', function() {
                container.find('.filter-post-type').val('post');
                container.find('.filter-category').val('all');
                container.find('.filter-date').val('all');
                activeFilters = {
                    postType: 'post',
                    category: 'all',
                    date: 'all'
                };
                if (searchInput.val().length >= 2) {
                    performSearch(searchInput.val(), container);
                }
            });

            // Prevent propagation for filter dropdown interactions
            container.on('click', '.modernpress-filter-dropdown, .filter-section select, .filter-actions button', function(e) {
                e.stopPropagation();
            });

            // Handle filter icon click
            container.on('click', '.filter-icon', function(e) {
                e.preventDefault();
                e.stopPropagation();
                filterDropdown.toggleClass('active');
                $(this).toggleClass('active');
            });

            // Handle clicks outside search container
            $(document).on('click', function(e) {
                const $target = $(e.target);
                const isSearchInput = $target.closest('.modernpress-search-input').length;
                const isSearchResults = $target.closest('.modernpress-search-results').length;
                const isFilterDropdown = $target.closest('.modernpress-filter-dropdown').length;
                const isFilterIcon = $target.closest('.filter-icon').length;

                // Close search results if clicking outside search area
                if (!isSearchInput && !isSearchResults && !isFilterDropdown && !isFilterIcon) {
                    resultsContainer.removeClass('active').empty();
                    filterDropdown.removeClass('active');
                    container.find('.filter-icon').removeClass('active');
                }
            });

            // Prevent search results from closing when clicking inside them
            resultsContainer.on('click', function(e) {
                e.stopPropagation();
            });

            // Handle escape key to close both dropdowns
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    resultsContainer.removeClass('active').empty();
                    filterDropdown.removeClass('active');
                    container.find('.filter-icon').removeClass('active');
                }
            });

            // Prevent search results from closing when clicking inside them
            resultsContainer.on('click', function(e) {
                e.stopPropagation();
            });

            // Handle escape key to close both dropdowns
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    resultsContainer.removeClass('active').empty();
                    filterDropdown.removeClass('active');
                }
            });

            // Handle search input
            searchInput.on('input', function() {
                const query = $(this).val();
                
                clearTimeout(searchTimeout);

                if (query.length < 2) {
                    resultsContainer.removeClass('active').empty();
                    return;
                }

                searchTimeout = setTimeout(() => {
                    performSearch(query, container);
                }, 300);
            });

            // Handle result item click
            container.on('click', '.search-result-item', function() {
                window.location.href = $(this).data('url');
            });
        });
    }

    function performSearch(query, container) {
        const searchInput = container.find('.modernpress-search-input');
        const resultsContainer = container.find('.modernpress-search-results');
        const resultsLimit = searchInput.data('results-limit');

        $.ajax({
            url: modernpressSearch.ajaxurl,
            type: 'POST',
            data: {
                action: 'modernpress_live_search',
                nonce: modernpressSearch.nonce,
                query: query,
                limit: resultsLimit,
                filters: activeFilters
            },
            beforeSend: function() {
                searchInput.addClass('loading');
                resultsContainer.html('<div class="search-loading">Searching...</div>');
                resultsContainer.addClass('active');
            },
            success: function(response) {
                if (response.success && response.data) {
                    resultsContainer.empty();

                    function highlightText(text, searchTerm) {
                        const searchTerms = searchTerm.split(' ').filter(term => term.length > 0);
                        let highlightedText = text;
                        
                        searchTerms.forEach(term => {
                            const regex = new RegExp(`(${term})`, 'gi');
                            highlightedText = highlightedText.replace(regex, '<mark>$1</mark>');
                        });
                        
                        return highlightedText;
                    }

                    if (response.data.results && response.data.results.length > 0) {
                        response.data.results.forEach(item => {
                            const highlightedTitle = highlightText(item.title, query);
                            const highlightedExcerpt = highlightText(item.excerpt, query);
                            
                            const resultHtml = `
                                <div class="search-result-item" data-url="${item.url}">
                                    ${item.thumbnail ? `
                                        <div class="search-result-thumbnail">
                                            <img src="${item.thumbnail}" alt="${item.title}">
                                        </div>
                                    ` : ''}
                                    <div class="search-result-content">
                                        <div class="search-result-title">${highlightedTitle}</div>
                                        <div class="search-result-excerpt">${highlightedExcerpt}</div>
                                        ${item.categories.length ? `
                                            <div class="search-result-categories">
                                                ${item.categories.map(cat => 
                                                    `<span class="category-tag">${cat.name}</span>`
                                                ).join('')}
                                            </div>
                                        ` : ''}
                                    </div>
                                </div>
                            `;
                            resultsContainer.append(resultHtml);
                        });

                        if (response.data.found_posts > response.data.results.length) {
                            // Create search URL with current filters
                            const searchParams = new URLSearchParams();
                            searchParams.append('s', query);
                            
                            if (activeFilters.postType !== 'any') {
                                searchParams.append('post_type', activeFilters.postType);
                            }
                            if (activeFilters.category !== 'all') {
                                searchParams.append('cat', activeFilters.category);
                            }
                            if (activeFilters.date !== 'all') {
                                searchParams.append('date_filter', activeFilters.date);
                            }
                            
                            const searchUrl = `${modernpressSearch.siteUrl}/?${searchParams.toString()}`;
                            
                            resultsContainer.append(`
                                <div class="search-results-footer">
                                    <a href="${searchUrl}" class="view-all-results">
                                        Showing ${response.data.results.length} of ${response.data.found_posts} results
                                        <span class="view-all-text">View all results</span>
                                    </a>
                                </div>
                            `);
                        }
                    } else {
                        resultsContainer.html('<div class="no-results">No results found</div>');
                    }
                    
                    resultsContainer.addClass('active');
                }
            },
            error: function() {
                resultsContainer.html('<div class="search-error">Error occurred while searching</div>');
            },
            complete: function() {
                searchInput.removeClass('loading');
            }
        });
    }

    // Initialize search functionality
    initModernPressSearch();

    // Re-initialize when new content is added (e.g., in Elementor editor)
    if (window.elementor) {
        elementor.hooks.addAction('panel/open_editor/widget', function() {
            initModernPressSearch();
        });
    }
});