
    /**
 * ModernPress Slider Script
 * Handles initialization of Swiper sliders for all layouts
 */

(function($) {
    "use strict";
    
    // Collection to keep track of initialized sliders
    var initializedSliders = {};
    
    /**
     * Initialize a slider based on layout type and settings
     */
    function initSlider(sliderWrapper) {
        var $wrapper = $(sliderWrapper);
        var sliderId = $wrapper.data('slider-id');
        
        // Don't reinitialize if already done
        if (initializedSliders[sliderId]) {
            return;
        }
        
        var settings;
        try {
            settings = $wrapper.data('settings');
        } catch (e) {
            console.error('Error parsing slider settings', e);
            return;
        }
        
        var layout = settings.layout || 'layout-1';
        var $container = $wrapper.find('.swiper');
        
        if (!$container.length) {
            return;
        }
        
        // Common config for all layouts
        var swiperConfig = {
            loop: settings.loop === 'yes',
            speed: parseInt(settings.speed) || 500,
            effect: settings.effect || 'slide',
            autoplay: settings.autoplay === 'yes' ? {
                delay: parseInt(settings.autoplay_speed) || 5000,
                disableOnInteraction: false
            } : false,
            navigation: settings.navigation === 'yes' ? {
                nextEl: $wrapper.find('.swiper-button-next')[0],
                prevEl: $wrapper.find('.swiper-button-prev')[0]
            } : false,
            pagination: settings.pagination === 'yes' ? {
                el: $wrapper.find('.swiper-pagination')[0],
                clickable: true,
                type: settings.pagination_type || 'bullets'
            } : false
        };
        
        // Layout specific configuration
        switch(layout) {
            case 'layout-1':
                // Classic slider - Full width, full height
                // Default config is fine
                break;
                
            case 'layout-2':
                // Modern slider - Multiple slides
                swiperConfig.slidesPerView = 3;
                swiperConfig.spaceBetween = 20;
                swiperConfig.breakpoints = {
                    320: {
                        slidesPerView: 1,
                        spaceBetween: 10
                    },
                    480: {
                        slidesPerView: 2,
                        spaceBetween: 15
                    },
                    768: {
                        slidesPerView: 2,
                        spaceBetween: 20
                    },
                    1024: {
                        slidesPerView: 3,
                        spaceBetween: 20
                    }
                };
                break;
                
            case 'layout-3':
                // Creative slider - Partial slides
                swiperConfig.slidesPerView = 2.3;
                swiperConfig.spaceBetween = 0;
                swiperConfig.breakpoints = {
                    320: {
                        slidesPerView: 1.3,
                        spaceBetween: 10
                    },
                    768: {
                        slidesPerView: 2.3,
                        spaceBetween: 20
                    }
                };
                break;
        }
        
        // Initialize Swiper
        try {
            var swiper = new Swiper($container[0], swiperConfig);
            initializedSliders[sliderId] = swiper;
            
            // Update slider on window resize
            $(window).on('resize', function() {
                if (swiper) {
                    swiper.update();
                }
            });
        } catch (e) {
            console.error('Error initializing Swiper', e);
        }
    }
    
    /**
     * Initialize all sliders on the page
     */
    function initAllSliders() {
        $('.modernpress-slider-wrapper').each(function() {
            initSlider(this);
        });
    }
    
    // Initialize sliders on document ready
    $(document).ready(function() {
        initAllSliders();
    });
    
    // Handle Elementor frontend init for editor preview
    $(window).on('elementor/frontend/init', function() {
        if (typeof elementorFrontend !== 'undefined') {
            elementorFrontend.hooks.addAction('frontend/element_ready/modernpress_slider_widget.default', function($element) {
                // Wait a moment for DOM to be fully updated
                setTimeout(function() {
                    var $slider = $element.find('.modernpress-slider-wrapper');
                    if ($slider.length) {
                        initSlider($slider[0]);
                    }
                }, 100);
            });
        }
    });
    
    // Reinitialize on changes in Elementor editor
    if (typeof elementor !== 'undefined') {
        elementor.channels.editor.on('change', function() {
            // Delay to ensure DOM is updated
            setTimeout(initAllSliders, 500);
        });
    }
    
})(jQuery);