<?php
/**
 * Direct Export-Import functions for ModernPress
 *
 * This file contains alternative export-import functionality that bypasses the normal form handling
 * to address persistent issues with file uploads and JSON handling.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Direct export settings to a JSON file
 */
function modernpress_direct_export_settings()
{
    // Verify user capabilities
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'modernpress-addons'));
    }

    // Get export data
    $active_widgets = get_option('modernpress_activated_widgets', []);
    $settings = get_option('modernpress_settings', []);

    $data = array(
        'widgets' => $active_widgets,
        'settings' => $settings
    );

    // Convert to JSON
    $json = wp_json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    if ($json === false) {
        wp_die(esc_html__('Error encoding settings to JSON.', 'modernpress-addons'));
    }

    // Force download
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename=modernpress-settings-' . gmdate('Y-m-d') . '.json');
    header('Cache-Control: no-cache, no-store, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    header('Content-Length: ' . strlen($json));

    // Output file contents and exit
    wp_send_json($data);

}

/**
 * Process direct import of settings from uploaded file
 */
function modernpress_process_direct_import()
{
    // Verify user capabilities
    if (!current_user_can('manage_options')) {
        return false;
    }

    // Verify nonce
    if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'modernpress_import_nonce')) {
        modernpress_add_admin_notice(esc_html__('Nonce verification failed.', 'modernpress-addons'), 'error');
        return false;
    }

    // Check if we have a file
    if (empty($_FILES['modernpress_import_file']['name'])) {
        modernpress_add_admin_notice(esc_html__('No file uploaded.', 'modernpress-addons'), 'error');
        return false;
    }

    // Check for upload errors
    if (!isset($_FILES['modernpress_import_file']['error']) || $_FILES['modernpress_import_file']['error'] !== UPLOAD_ERR_OK) {
        modernpress_add_admin_notice(esc_html__('Upload error. Please try again.', 'modernpress-addons'), 'error');
        return false;
    }

    // Validate file extension
    $file_info = pathinfo(sanitize_file_name($_FILES['modernpress_import_file']['name']));
    $extension = strtolower($file_info['extension'] ?? '');
    if ($extension !== 'json') {
        modernpress_add_admin_notice(esc_html__('Invalid file type. Please upload a JSON file.', 'modernpress-addons'), 'error');
        return false;
    }

    // Read file content
    if (isset($_FILES['modernpress_import_file']['tmp_name']) && !empty($_FILES['modernpress_import_file']['tmp_name'])) {
        $file_path = sanitize_text_field(wp_unslash($_FILES['modernpress_import_file']['tmp_name']));

        if (file_exists($file_path) && is_readable($file_path)) {
            $file_content = implode('', file($file_path)); // Read file safely
        } else {
            wp_die(esc_html__('Uploaded file cannot be read.', 'modernpress-addons'));
        }
    } else {
        wp_die(esc_html__('No file uploaded.', 'modernpress-addons'));
    }

    if ($file_content === false) {
        modernpress_add_admin_notice(esc_html__('Failed to read the uploaded file.', 'modernpress-addons'), 'error');
        return false;
    }

    // Remove BOM if present
    $file_content = preg_replace('/^\xEF\xBB\xBF/', '', $file_content);

    // Parse JSON
    $imported_data = json_decode($file_content, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        modernpress_add_admin_notice(esc_html__('Error parsing the JSON file: ', 'modernpress-addons') . esc_html(json_last_error_msg()), 'error');
        return false;
    }

    // Validate data structure
    if (!isset($imported_data['widgets']) || !isset($imported_data['settings'])) {
        modernpress_add_admin_notice(esc_html__('Invalid file structure. The file must contain widgets and settings keys.', 'modernpress-addons'), 'error');
        return false;
    }

    // Sanitize and update settings
    if (is_array($imported_data['widgets'])) {
        update_option('modernpress_activated_widgets', array_map('sanitize_text_field', $imported_data['widgets']));
    }

    if (is_array($imported_data['settings'])) {
        update_option('modernpress_settings', array_map('sanitize_text_field', $imported_data['settings']));
    }

    modernpress_add_admin_notice(esc_html__('Settings imported successfully.', 'modernpress-addons'), 'success');
    return true;
}

/**
 * Helper function to add admin notices
 */
function modernpress_add_admin_notice($message, $type = 'info')
{
    add_action('admin_notices', function () use ($message, $type) {
        echo '<div class="notice notice-' . esc_attr($type) . ' is-dismissible"><p>' . wp_kses_post($message) . '</p></div>';
    });
}
