<?php

namespace Modern_Pricing_Table\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class PricingTable extends Widget_Base
{
    /**
     * Get widget name.
     *
     * Retrieve Pricing Table widget name.
     *
     * @since 0.1.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'modren_pricing_table';
    }

    /**
     * Get widget title.
     *
     * Retrieve Pricing Table widget title.
     *
     * @since 0.1.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Modern Pricing Table', 'modern-pricing-table-for-elementor');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Pricing Table widget icon.
     *
     * @since 0.1.0
     * @access public
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-price-table';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Pricing Table widget belongs to.
     *
     * @since 0.1.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['pay-addons'];
    }

    /**
     * Get widget style dependencies.
     * 
     * Retrieve the list of style dependencies for the Pricing Table widget.
     * 
     * @since 0.1.0
     * @access public
     * @return array Widget styles.
     */
    public function get_style_depends()
    {
        return ['modern-pricing-table-style'];
    }

    /**
     * Register Pricing Table widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 0.1.0
     * @access protected
     */
    protected function register_controls()
    {
        $this->register_content_controls();
        $this->register_style_controls();
		parent::register_controls();
    }

    private function register_content_controls() {

        // Header Section
        $this->start_controls_section(
            'section_header',
            [
                'label' => esc_html__('Settings', 'modern-pricing-table-for-elementor'),
            ]
        );
        $this->add_control(
            'plan_title',
            [
                'label' => esc_html__('Title', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Starter', 'modern-pricing-table-for-elementor'),
                'dynamic' => ['active' => true],
            ]
        );
        $this->add_control(
            'plan_subtitle',
            [
                'label' => esc_html__('Subtitle', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Best for individuals', 'modern-pricing-table-for-elementor'),
                'dynamic' => ['active' => true],
            ]
        );
        $this->add_control(
            'heading_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                ],
                'default' => 'h3',
            ]
        );
        $this->add_control(
            'show_highlight',
            [
                'label' => esc_html__('Highlight', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );
        $this->end_controls_section();

        // Pricing Section
        $this->start_controls_section(
            'section_pricing',
            [
                'label' => esc_html__('Pricing', 'modern-pricing-table-for-elementor'),
            ]
        );
        $this->add_control(
            'currency_symbol',
            [
                'label' => esc_html__('Currency Symbol', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => esc_html__('None', 'modern-pricing-table-for-elementor'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency', 'modern-pricing-table-for-elementor'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency', 'modern-pricing-table-for-elementor'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency', 'modern-pricing-table-for-elementor'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency', 'modern-pricing-table-for-elementor'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency', 'modern-pricing-table-for-elementor'),
                    'krona' => 'kr ' . _x('Krona', 'Currency', 'modern-pricing-table-for-elementor'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency', 'modern-pricing-table-for-elementor'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency', 'modern-pricing-table-for-elementor'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency', 'modern-pricing-table-for-elementor'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency', 'modern-pricing-table-for-elementor'),
                    'real' => 'R$ ' . _x('Real', 'Currency', 'modern-pricing-table-for-elementor'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency', 'modern-pricing-table-for-elementor'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency', 'modern-pricing-table-for-elementor'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency', 'modern-pricing-table-for-elementor'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency', 'modern-pricing-table-for-elementor'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency', 'modern-pricing-table-for-elementor'),
                    'won' => '&#8361; ' . _x('Won', 'Currency', 'modern-pricing-table-for-elementor'),
                    'custom' => esc_html__('Custom', 'modern-pricing-table-for-elementor'),
                ],
                'default' => 'dollar',
            ]
        );
        $this->add_control(
            'currency_symbol_custom',
            [
                'label' => esc_html__('Custom Symbol', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'condition' => ['currency_symbol' => 'custom'],
            ]
        );
        $this->add_control(
            'currency_position',
            [
                'label' => esc_html__('Position', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'before',
                'options' => [
                    'before' => [
                        'title' => esc_html__('Before', 'modern-pricing-table-for-elementor'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'after' => [
                        'title' => esc_html__('After', 'modern-pricing-table-for-elementor'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
            ]
        );
        $this->add_control(
            'price_amount',
            [
                'label' => esc_html__('Price', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => '29',
                'dynamic' => ['active' => true],
            ]
        );
        $this->add_control(
            'sale',
            [
                'label' => esc_html__('Sale', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('On', 'modern-pricing-table-for-elementor'),
                'label_off' => esc_html__('Off', 'modern-pricing-table-for-elementor'),
                'default' => '',
            ]
        );
        $this->add_control(
            'original_price',
            [
                'label' => esc_html__('Original Price', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::NUMBER,
                'default' => '39',
                'condition' => ['sale' => 'yes'],
                'dynamic' => ['active' => true],
            ]
        );
        $this->add_control(
            'period',
            [
                'label' => esc_html__('Period', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('per month', 'modern-pricing-table-for-elementor'),
                'dynamic' => ['active' => true],
            ]
        );

        $this->add_control(
            'period_position',
            [
                'label' => esc_html__('Position', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    'below' => esc_html__('Below', 'modern-pricing-table-for-elementor'),
                    'beside' => esc_html__('Beside', 'modern-pricing-table-for-elementor'),
                ],
                'default' => 'below',
                'condition' => [
                    'period!' => '',
                ],
            ]
        );
        $this->end_controls_section();

        // Features Section
        $this->start_controls_section(
            'section_features',
            [
                'label' => esc_html__('Features', 'modern-pricing-table-for-elementor'),
            ]
        );
        $repeater = new Repeater();
        $repeater->add_control(
            'feature_text',
            [
                'label' => esc_html__('Text', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('List Item', 'modern-pricing-table-for-elementor'),
                'dynamic' => ['active' => true],
            ]
        );
        $repeater->add_control(
            'feature_icon',
            [
                'label' => esc_html__('Icon', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::ICONS,
                'default' => ['value' => 'fas fa-check', 'library' => 'fa-solid'],
            ]
        );
        $repeater->add_control(
            'feature_strikethrough',
            [
                'label' => esc_html__('Show strikethrough', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no', 
            ]
        );
        $this->add_control(
            'features_list',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    ['feature_text' => esc_html__('All tools you need to manage payments', 'modern-pricing-table-for-elementor')],
                    ['feature_text' => esc_html__('No setup, monthly, or hidden fees', 'modern-pricing-table-for-elementor')],
                    ['feature_text' => esc_html__('Comprehensive security', 'modern-pricing-table-for-elementor')],
                ],
                'title_field' => '{{{ feature_text }}}',
            ]
        );
        $this->end_controls_section();

        // Button Section
        $this->start_controls_section(
            'section_button',
            [
                'label' => esc_html__('Button', 'modern-pricing-table-for-elementor'),
            ]
        );
        $this->add_control(
            'button_text',
            [
                'label' => esc_html__('Button Text', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Get started', 'modern-pricing-table-for-elementor'),
                'dynamic' => ['active' => true],
            ]
        );
        $this->add_control(
            'button_url',
            [
                'label' => esc_html__('Button URL', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::URL,
                'placeholder' => esc_html__('https://your-link.com', 'modern-pricing-table-for-elementor'),
                'default' => ['url' => '#'],
                'dynamic' => ['active' => true],
            ]
        );
        $this->end_controls_section();

        // Ribbon Section
        $this->start_controls_section(
            'section_ribbon',
            [
                'label' => esc_html__('Ribbon', 'modern-pricing-table-for-elementor'),
            ]
        );
        $this->add_control(
            'show_ribbon',
            [
                'label' => esc_html__('Show Ribbon', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );
        $this->add_control(
            'ribbon_title',
            [
                'label' => esc_html__('Title', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Most popular', 'modern-pricing-table-for-elementor'),
                'condition' => ['show_ribbon' => 'yes'],
                'dynamic' => ['active' => true],
            ]
        );
        $this->end_controls_section();
    }

    private function register_style_controls()
    {
        $this->start_controls_section(
            'section_header_style',
            [
                'label' => esc_html__('Header', 'modern-pricing-table-for-elementor'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'heading_heading_style',
            [
                'label' => esc_html__('Title', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'heading_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__plan-name' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'heading_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__plan-name',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
            ]
        );

        $this->add_control(
            'heading_sub_heading_style',
            [
                'label' => esc_html__('Sub Title', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'sub_heading_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__plan-subtitle' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'sub_heading_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__plan-subtitle',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
                ],
            ]
        );

        $this->add_control(
            'heading_lighlight_style',
            [
                'label' => esc_html__('Highlight', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'highlight_color',
            [
                'label' => esc_html__('Highlight Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'default' => '#2563eb',
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table--highlight' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pricing_element_style',
            [
                'label' => esc_html__('Pricing', 'modern-pricing-table-for-elementor'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'price_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__price-amount' => 'color: {{VALUE}}',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__price',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
            ]
        );

        $this->add_control(
            'heading_original_price_style',
            [
                'label' => esc_html__('Original Price', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'sale' => 'yes',
                    'original_price!' => '',
                ],
            ]
        );

        $this->add_control(
            'original_price_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'global' => [
                    'default' => Global_Colors::COLOR_SECONDARY,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__original-price' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'sale' => 'yes',
                    'original_price!' => '',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'original_price_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__original-price',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
                'condition' => [
                    'sale' => 'yes',
                    'original_price!' => '',
                ],
            ]
        );

        $this->add_control(
            'heading_period_style',
            [
                'label' => esc_html__('Period', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'period!' => '',
                ],
            ]
        );

        $this->add_control(
            'period_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'global' => [
                    'default' => Global_Colors::COLOR_SECONDARY,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__price-period' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'period!' => '',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'period_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__price-period',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_SECONDARY,
                ],
                'condition' => [
                    'period!' => '',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_features_list_style',
            [
                'label' => esc_html__('Features', 'modern-pricing-table-for-elementor'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'features_list_color',
            [
                'label' => esc_html__('Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__feature-text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'features_list_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__feature-text',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_TEXT,
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_button_style',
            [
                'label' => esc_html__('Button', 'modern-pricing-table-for-elementor'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'button_size',
            [
                'label' => esc_html__('Size', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::SELECT,
                'default' => 'md',
                'options' => [
                    'xs' => esc_html__('Extra Small', 'modern-pricing-table-for-elementor'),
                    'sm' => esc_html__('Small', 'modern-pricing-table-for-elementor'),
                    'md' => esc_html__('Medium', 'modern-pricing-table-for-elementor'),
                    'lg' => esc_html__('Large', 'modern-pricing-table-for-elementor'),
                    'xl' => esc_html__('Extra Large', 'modern-pricing-table-for-elementor'),
                ],
                'condition' => [
                    'button_text!' => '',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_button_style');

        $this->start_controls_tab(
            'tab_button_normal',
            [
                'label' => esc_html__('Normal', 'modern-pricing-table-for-elementor'),
                'condition' => [
                    'button_text!' => '',
                ],
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label' => esc_html__('Text Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__cta-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_ACCENT,
                ],
                'selector' => '{{WRAPPER}} .modern-pricing-table__cta-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'button_background',
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .modern-pricing-table__cta-button',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'button_border',
                'selector' => '{{WRAPPER}} .modern-pricing-table__cta-button',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'button_border_radius',
            [
                'label' => esc_html__('Border Radius', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__cta-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_text_padding',
            [
                'label' => esc_html__('Text Padding', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__cta-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_button_hover',
            [
                'label' => esc_html__('Hover', 'modern-pricing-table-for-elementor'),
                'condition' => [
                    'button_text!' => '',
                ],
            ]
        );

        $this->add_control(
            'button_hover_color',
            [
                'label' => esc_html__('Text Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__cta-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'button_background_hover',
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .modern-pricing-table__cta-button:hover',
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                ],
            ]
        );

        $this->add_control(
            'button_hover_border_color',
            [
                'label' => esc_html__('Border Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__cta-button:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_hover_animation',
            [
                'label' => esc_html__('Animation', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::HOVER_ANIMATION,
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'section_ribbon_style',
            [
                'label' => esc_html__('Ribbon', 'modern-pricing-table-for-elementor'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
                'condition' => [
                    'show_ribbon' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'ribbon_bg_color',
            [
                'label' => esc_html__('Background Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'default' => '#dbeafe',
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__badge' => 'background-color: {{VALUE}}',
                ],
            ]
        );


        $this->add_control(
            'ribbon_text_color',
            [
                'label' => esc_html__('Text Color', 'modern-pricing-table-for-elementor'),
                'type' => Controls_Manager::COLOR,
                'default' => '#2563eb',
                'selectors' => [
                    '{{WRAPPER}} .modern-pricing-table__badge' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'ribbon_typography',
                'selector' => '{{WRAPPER}} .modern-pricing-table__badge',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_ACCENT,
                ],
            ]
        );

        $this->end_controls_section();
    }

    private function render_currency_symbol($symbol, $location)
    {
        $currency_position = $this->get_settings('currency_position');
        $location_setting = ! empty($currency_position) ? $currency_position : 'before';
        if (! empty($symbol) && $location === $location_setting) {
            echo '<span class="epa-price-table__currency">' . esc_html($symbol) . '</span>';
        }
    }

    private function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'pound' => '&#163;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'baht' => '&#3647;',
            'yen' => '&#165;',
            'won' => '&#8361;',
            'guilder' => '&fnof;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'rupee' => '&#8360;',
            'indian_rupee' => '&#8377;',
            'real' => 'R$',
            'krona' => 'kr',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }

    /**
     * Render Pricing Table widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 0.1.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $symbol = '';

        if (! empty($settings['currency_symbol'])) {
            if ('custom' !== $settings['currency_symbol']) {
                $symbol = $this->get_currency_symbol($settings['currency_symbol']);
            } else {
                $symbol = $settings['currency_symbol_custom'];
            }
        }

		$period_position = $settings['period_position'];
        $heading_tag = !empty($settings['heading_tag']) ? $settings['heading_tag'] : 'h3';
        $main_classes = 'modern-pricing-table';
        if (!empty($settings['show_highlight']) && $settings['show_highlight'] === 'yes') {
            $main_classes .= ' modern-pricing-table--highlight';
        }
?>
        <div class="<?php echo esc_attr($main_classes); ?>">
            <?php if ('yes' === $settings['show_ribbon'] && !empty($settings['ribbon_title'])) : ?>
                <div class="modern-pricing-table__badge">
                    <?php echo esc_html($settings['ribbon_title']); ?>
                </div>
            <?php endif; ?>
            <div class="modern-pricing-table__header">
                <<?php echo esc_attr($heading_tag); ?> class="modern-pricing-table__plan-name"><?php echo esc_html($settings['plan_title']); ?></<?php echo esc_attr($heading_tag); ?>>
                <?php if (!empty($settings['plan_subtitle'])) : ?>
                    <div class="modern-pricing-table__plan-subtitle"><?php echo esc_html($settings['plan_subtitle']); ?></div>
                <?php endif; ?>
                <div class="modern-pricing-table__price">
                    <?php if ('yes' === $settings['sale'] && !empty($settings['original_price'])) : ?>
                        <span class="modern-pricing-table__original-price">
                            <?php
                            $this->render_currency_symbol($symbol, 'before');
                            $this->print_unescaped_setting('original_price');
                            $this->render_currency_symbol($symbol, 'after');
                            ?>
                        </span>
                    <?php endif; ?>
                    <span class="modern-pricing-table__price-amount">
                            <?php
                            $this->render_currency_symbol($symbol, 'before');
                            $this->print_unescaped_setting('price_amount');
                            $this->render_currency_symbol($symbol, 'after');
                            ?>
                    </span>
                    <?php if (!empty($settings['period']) && 'beside' === $period_position ) : ?>
                    <span class="modern-pricing-table__price-period"><?php echo esc_html($settings['period']); ?></span>
                    <?php endif; ?>
                </div>
                <?php if (!empty($settings['period']) && 'below' === $period_position ) : ?>
                    <div class="modern-pricing-table__price-period"><?php echo esc_html($settings['period']); ?></div>
                <?php endif; ?>
            </div>
            <?php if (!empty($settings['button_text'])) : ?>
                <a href="<?php echo esc_url($settings['button_url']['url']); ?>" class="modern-pricing-table__cta-button">
                    <?php echo esc_html($settings['button_text']); ?>
                </a>
            <?php endif; ?>
            <?php if (!empty($settings['features_list'])) : ?>
                <ul class="modern-pricing-table__features">
                    <?php foreach ($settings['features_list'] as $item) : ?>
                        <li class="modern-pricing-table__feature <?php echo esc_attr($item['feature_strikethrough'] === 'yes' ? 'modern-pricing-table__feature--disabled' : ''); ?>">
                            <?php if (!empty($item['feature_icon']['value'])) : ?>
                                <i class="<?php echo esc_attr($item['feature_icon']['value']); ?> modern-pricing-table__feature-icon"></i>
                            <?php endif; ?>
                            <span class="modern-pricing-table__feature-text <?php echo esc_attr($item['feature_strikethrough'] === 'yes' ? 'modern-pricing-table__feature-text--disabled' : ''); ?>"><?php echo esc_html($item['feature_text']); ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    <?php
    }

    /**
     * Render list widget output in the editor.
     *
     * Written as a Backbone JavaScript template and used to generate the live preview.
     *
     * @since 0.1.0
     * @access protected
     */
    protected function content_template()
    {
    ?>
        <#

			var symbols = {
				dollar: '&#36;',
				euro: '&#128;',
				franc: '&#8355;',
				pound: '&#163;',
				ruble: '&#8381;',
				shekel: '&#8362;',
				baht: '&#3647;',
				yen: '&#165;',
				won: '&#8361;',
				guilder: '&fnof;',
				peso: '&#8369;',
				peseta: '&#8359;',
				lira: '&#8356;',
				rupee: '&#8360;',
				indian_rupee: '&#8377;',
				real: 'R$',
				krona: 'kr'
			};

			var symbol = '',
				iconsHTML = {};

			if ( settings.currency_symbol ) {
				if ( 'custom' !== settings.currency_symbol ) {
					symbol = symbols[ settings.currency_symbol ] || '';
				} else {
					symbol = settings.currency_symbol_custom;
				}
			}
            var headingTag=settings.heading_tag || 'h3' ;
            #>
            <div class="modern-pricing-table <# if ('yes' === settings.show_highlight) { #>modern-pricing-table--highlight<# } #>">
                <# if ('yes'===settings.show_ribbon && settings.ribbon_title) { #>
                    <div class="modern-pricing-table__badge">{{{ settings.ribbon_title }}}</div>
                    <# } #>
                        <div class="modern-pricing-table__header">
                            <{{ headingTag }} class="modern-pricing-table__plan-name">{{{ settings.plan_title }}}</{{ headingTag }}>
                            <# if (settings.plan_subtitle) { #>
                                <div class="modern-pricing-table__plan-subtitle">{{{ settings.plan_subtitle }}}</div>
                                <# } #>
                                    <div class="modern-pricing-table__price">
                                        <# if ('yes'===settings.sale && settings.original_price) { #>
                                            <span class="modern-pricing-table__original-price">{{{ 'before' === settings.currency_position ? symbol + settings.original_price : settings.original_price + symbol }}}</span>
                                        <# } #>
                                        <span class="modern-pricing-table__price-amount">{{{ 'before' === settings.currency_position ? symbol + settings.price_amount : settings.original_price + symbol }}}</span>
                                        <# if ( settings.period && 'beside' === settings.period_position ) { #>
						                    <span class="modern-pricing-table__price-period">{{{ settings.period }}}</span>
					                    <# } #>
                                    </div>
                                    <# if (settings.period && 'below' === settings.period_position) { #>
                                        <div class="modern-pricing-table__price-period">{{{ settings.period }}}</div>
                                        <# } #>
                        </div>
                        <# if (settings.button_text) { #>
                            <a href="{{ settings.button_url.url }}" class="modern-pricing-table__cta-button">{{{ settings.button_text }}}</a>
                            <# } #>
                                <# if (settings.features_list && settings.features_list.length) { #>
                                    <ul class="modern-pricing-table__features">
                                        <# _.each(settings.features_list, function(item) { #>
                                            <li class="modern-pricing-table__feature <# if (item.feature_strikethrough === 'yes') { #>modern-pricing-table__feature--disabled<# } #>">
                                                <# if (item.feature_icon && item.feature_icon.value) { #>
                                                    <i class="{{ item.feature_icon.value }} modern-pricing-table__feature-icon"></i>
                                                    <# } #>
                                                        <span class="modern-pricing-table__feature-text <# if (item.feature_strikethrough === 'yes') { #>modern-pricing-table__feature-text--disabled<# } #>">{{{ item.feature_text }}}</span>
                                            </li>
                                            <# }); #>
                                    </ul>
                                    <# } #>
            </div>
    <?php
    }
}
