<?php
/**
 * File: class-game-log-post-type.php
 *
 * @package Game_Log
 * @since 1.0.0
 */

declare(strict_types=1);

/**
 * Game Log Post Type class
 *
 * @package Game_Log
 */
class Game_Log_Post_Type {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register_post_type' ) );
		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
		add_action( 'save_post', array( $this, 'save_meta_boxes' ) );
	}

	/**
	 * Register the game post type
	 */
	public function register_post_type(): void {
		$labels = array(
			'name'                  => _x( 'Games', 'Post type general name', 'mode7-game-log' ),
			'singular_name'         => _x( 'Game', 'Post type singular name', 'mode7-game-log' ),
			'menu_name'             => _x( 'Games', 'Admin Menu text', 'mode7-game-log' ),
			'name_admin_bar'        => _x( 'Game', 'Add New on Toolbar', 'mode7-game-log' ),
			'add_new'               => __( 'Add New', 'mode7-game-log' ),
			'add_new_item'          => __( 'Add New Game', 'mode7-game-log' ),
			'new_item'              => __( 'New Game', 'mode7-game-log' ),
			'edit_item'             => __( 'Edit Game', 'mode7-game-log' ),
			'view_item'             => __( 'View Game', 'mode7-game-log' ),
			'all_items'             => __( 'All Games', 'mode7-game-log' ),
			'search_items'          => __( 'Search Games', 'mode7-game-log' ),
			'parent_item_colon'     => __( 'Parent Games:', 'mode7-game-log' ),
			'not_found'             => __( 'No games found.', 'mode7-game-log' ),
			'not_found_in_trash'    => __( 'No games found in Trash.', 'mode7-game-log' ),
			'featured_image'        => _x( 'Game Cover', 'Overrides the "Featured Image" phrase', 'mode7-game-log' ),
			'set_featured_image'    => _x( 'Set game cover', 'Overrides the "Set featured image" phrase', 'mode7-game-log' ),
			'remove_featured_image' => _x( 'Remove game cover', 'Overrides the "Remove featured image" phrase', 'mode7-game-log' ),
			'use_featured_image'    => _x( 'Use as game cover', 'Overrides the "Use as featured image" phrase', 'mode7-game-log' ),
			'archives'              => _x( 'Game archives', 'The post type archive label', 'mode7-game-log' ),
			'insert_into_item'      => _x( 'Insert into game', 'Overrides the "Insert into post"/"Insert into page" phrase', 'mode7-game-log' ),
			'uploaded_to_this_item' => _x( 'Uploaded to this game', 'Overrides the "Uploaded to this post"/"Uploaded to this page" phrase', 'mode7-game-log' ),
			'filter_items_list'     => _x( 'Filter games list', 'Screen reader text for the filter links', 'mode7-game-log' ),
			'items_list_navigation' => _x( 'Games list navigation', 'Screen reader text for the pagination', 'mode7-game-log' ),
			'items_list'            => _x( 'Games list', 'Screen reader text for the items list', 'mode7-game-log' ),
		);

		$args = array(
			'labels'             => $labels,
			'public'             => true,
			'publicly_queryable' => true,
			'show_ui'            => true,
			'show_in_menu'       => false, // We'll add it to our custom menu.
			'query_var'          => true,
			'rewrite'            => array( 'slug' => 'game' ),
			'capability_type'    => 'post',
			'has_archive'        => true,
			'hierarchical'       => false,
			'menu_position'      => null,
			'menu_icon'          => 'dashicons-games',
			'supports'           => array( 'title', 'editor', 'thumbnail', 'excerpt', 'custom-fields', 'tags' ),
			'show_in_rest'       => true,
			'rest_base'          => 'games',
		);

		register_post_type( 'game', $args );
	}

	/**
	 * Add meta boxes for game custom fields
	 */
	public function add_meta_boxes(): void {
		add_meta_box(
			'game_details',
			__( 'Game Details', 'mode7-game-log' ),
			array( $this, 'game_details_meta_box' ),
			'game',
			'normal',
			'high'
		);
	}

	/**
	 * Game details meta box callback
	 *
	 * @param WP_Post $post The post object.
	 */
	public function game_details_meta_box( $post ): void {
		wp_nonce_field( 'game_details_meta_box', 'game_details_meta_box_nonce' );

		$rating       = get_post_meta( $post->ID, '_game_rating', true );
		$release_date = get_post_meta( $post->ID, '_game_release_date', true );
		$igdb_id      = get_post_meta( $post->ID, '_game_igdb_id', true );

		// Get current game status terms.
		$current_status_terms = wp_get_object_terms( $post->ID, 'game_status', array( 'fields' => 'ids' ) );
		$current_status_id    = ! empty( $current_status_terms ) ? $current_status_terms[0] : 0;

		// Get all game status terms.
		$status_terms = get_terms(
			array(
				'taxonomy'   => 'game_status',
				'hide_empty' => false,
			)
		);

		?>
		<table class="form-table">
			<tr>
				<th scope="row">
					<label for="game_status"><?php esc_html_e( 'Status', 'mode7-game-log' ); ?></label>
				</th>
				<td>
					<select id="game_status" name="game_status">
						<option value=""><?php esc_html_e( 'Select a status...', 'mode7-game-log' ); ?></option>
						<?php
						foreach ( $status_terms as $term ) {
							$selected = selected( $current_status_id, $term->term_id, false );
							printf(
								'<option value="%s" %s>%s</option>',
								esc_attr( $term->term_id ),
								esc_attr( $selected ),
								esc_html( $term->name )
							);
						}
						?>
					</select>
					<p class="description"><?php esc_html_e( 'Current status of this game', 'mode7-game-log' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="game_rating"><?php esc_html_e( 'Rating', 'mode7-game-log' ); ?></label>
				</th>
				<td>
					<input type="number" id="game_rating" name="game_rating" value="<?php echo esc_attr( $rating ); ?>" min="1" max="10" step="0.1" />
					<p class="description"><?php esc_html_e( 'Rate the game from 1 to 10', 'mode7-game-log' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="game_release_date"><?php esc_html_e( 'Release Date', 'mode7-game-log' ); ?></label>
				</th>
				<td>
					<input type="date" id="game_release_date" name="game_release_date" value="<?php echo esc_attr( $release_date ); ?>" />
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="game_igdb_id"><?php esc_html_e( 'IGDB ID', 'mode7-game-log' ); ?></label>
				</th>
				<td>
					<input type="text" id="game_igdb_id" name="game_igdb_id" value="<?php echo esc_attr( $igdb_id ); ?>" readonly />
					<p class="description"><?php esc_html_e( 'Internal IGDB identifier', 'mode7-game-log' ); ?></p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Save meta box data
	 *
	 * @param int $post_id The post ID.
	 */
	public function save_meta_boxes( $post_id ): void {
		// Check nonce.
		if ( ! isset( $_POST['game_details_meta_box_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['game_details_meta_box_nonce'] ) ), 'game_details_meta_box' ) ) {
			return;
		}

		// Check if autosave.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save meta fields.
		$fields = array(
			'game_rating'       => 'sanitize_text_field',
			'game_release_date' => 'sanitize_text_field',
			'game_igdb_id'      => 'sanitize_text_field',
		);

		foreach ( $fields as $field => $sanitize_callback ) {
			if ( isset( $_POST[ $field ] ) ) {
				$value = call_user_func( $sanitize_callback, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
				update_post_meta( $post_id, '_' . $field, $value );
			}
		}

		// Save game status.
		if ( isset( $_POST['game_status'] ) ) {
			$status_id = intval( $_POST['game_status'] );

			if ( $status_id > 0 ) {
				// Verify the term exists and belongs to game_status taxonomy.
				$term = get_term( $status_id, 'game_status' );
				if ( $term && ! is_wp_error( $term ) ) {
					wp_set_object_terms( $post_id, array( $status_id ), 'game_status' );
				}
			} else {
				// If no status selected, remove all status terms.
				wp_set_object_terms( $post_id, array(), 'game_status' );
			}
		}
	}
}
