<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

// REGISTRO DE RUTAS REST API
add_action( 'rest_api_init', function () {
    // Namespace actualizado a 'mnpn/v1' para cumplir con el prefijo de 4 letras
    register_rest_route( 'mnpn/v1', '/datos', array(
        'methods'             => 'GET',
        'callback'            => 'mnpn_api_obtener_populares', // Callback actualizado
        'permission_callback' => '__return_true',
    ));

    register_rest_route( 'mnpn/v1', '/contar', array(
        'methods'             => 'POST',
        'callback'            => 'mnpn_api_registrar_visita', // Callback actualizado
        'permission_callback' => '__return_true',
    ));
});

// 1. OBTENER DATOS (GET)
function mnpn_api_obtener_populares( $request ) { // Función renombrada
    global $wpdb;
    // IMPORTANTE: Nombre de tabla actualizado a 'mnpn_log'
    $tabla = $wpdb->prefix . 'mnpn_log';

    // Sanitizamos inmediatamente la entrada
    $rango_input = sanitize_text_field( $request->get_param( 'rango' ) );
    $rangos_permitidos = array( 'hoy', 'trending', 'semana' );
    $rango = in_array( $rango_input, $rangos_permitidos ) ? $rango_input : 'trending';

    $cantidad       = absint( $request->get_param( 'cantidad' ) );
    $excluir_id     = absint( $request->get_param( 'excluir' ) );
    $antiguedad_max = absint( $request->get_param( 'antiguedad' ) );
    
    if ( $cantidad < 1 ) $cantidad = 5;
    if ( $cantidad > 50 ) $cantidad = 50; 

    $ahora_ts = current_time( 'timestamp' ); 
    $fecha_limite = '';
    
    switch ( $rango ) {
        case 'hoy':
            $fecha_limite = gmdate( 'Y-m-d 00:00:00', $ahora_ts );
            break;
        case 'trending':
            $fecha_limite = gmdate( 'Y-m-d H:i:s', $ahora_ts - ( 48 * 3600 ) );
            break;
        case 'semana':
            $fecha_limite = gmdate( 'Y-m-d H:i:s', $ahora_ts - ( 7 * 24 * 3600 ) );
            break;
    }

    // Cache Key actualizado con el nuevo prefijo
    $cache_key = 'mnpn_query_v73_' . md5( $rango . $fecha_limite . $cantidad . $excluir_id );
    
    if ( current_user_can( 'manage_options' ) ) {
        delete_transient( $cache_key );
    }
    
    $raw_results = get_transient( $cache_key );

    if ( false === $raw_results ) {
        // Ignoramos la regla de interpolación aquí porque definimos el string
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "SELECT post_id, COUNT(id) as visitas FROM {$tabla} WHERE fecha >= %s GROUP BY post_id ORDER BY visitas DESC LIMIT %d";

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.NotPrepared
        $raw_results = $wpdb->get_results( $wpdb->prepare( $sql, $fecha_limite, $cantidad + 5 ) );
        
        set_transient( $cache_key, $raw_results, 15 * MINUTE_IN_SECONDS );
    }

    $final_posts = array();
    $count = 0;

    if ( $raw_results ) {
        foreach ( $raw_results as $row ) {
            if ( $count >= $cantidad ) break;
            
            $pid = intval( $row->post_id );

            if ( $pid === $excluir_id ) continue;
            if ( get_post_status( $pid ) !== 'publish' ) continue;
            if ( get_post_type( $pid ) !== 'post' ) continue;

            if ( $antiguedad_max > 0 ) {
                $post_date = get_the_date( 'Y-m-d', $pid );
                $dias_diff = ( current_time( 'timestamp' ) - strtotime( $post_date ) ) / 86400;
                if ( $dias_diff > $antiguedad_max ) continue;
            }

            $thumb_url = get_the_post_thumbnail_url( $pid, 'medium' );
            if ( ! $thumb_url ) {
                $thumb_url = get_the_post_thumbnail_url( $pid, 'full' );
            }
            if ( ! $thumb_url ) {
                $thumb_url = '';
            }

            $final_posts[] = array(
                // Decodificamos entidades HTML para que JS las reciba limpias y las escape después
                'titulo'  => html_entity_decode( get_the_title( $pid ) ),
                'enlace'  => get_permalink( $pid ),
                'imagen'  => $thumb_url,
                'visitas' => $row->visitas
            );

            $count++;
        }
    }

    return new WP_REST_Response( $final_posts, 200 );
}

// 2. REGISTRAR VISITA (POST)
function mnpn_api_registrar_visita( $request ) { // Función renombrada
    global $wpdb;
    
    $nonce = $request->get_header( 'x-wp-nonce' );
    if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
        return new WP_REST_Response( array( 'status' => 'ignored', 'reason' => 'invalid_nonce' ), 200 );
    }

    $user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
    
    $es_bot = preg_match( '/bot|crawl|slurp|spider|mediapartners|python|curl/i', $user_agent );

    if ( $es_bot ) {
        return new WP_REST_Response( array( 'status' => 'ignored', 'reason' => 'is_bot' ), 200 );
    }

    $post_id = absint( $request->get_param( 'id' ) );
    
    if ( $post_id > 0 ) {
        // Tabla actualizada
        $tabla = $wpdb->prefix . 'mnpn_log';
        $ahora = gmdate( 'Y-m-d H:i:s' );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $wpdb->insert( $tabla, array( 'post_id' => $post_id, 'fecha' => $ahora ), array( '%d', '%s' ) );
        
        if ( wp_rand( 1, 100 ) === 1 ) {
            // Opción actualizada a 'mnpn_dias_limpieza'
            $dias_limpieza = get_option( 'mnpn_dias_limpieza', 30 );
            $fecha_corte = gmdate( 'Y-m-d H:i:s', strtotime( "-$dias_limpieza days" ) );
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $sql_delete = "DELETE FROM {$tabla} WHERE fecha < %s";

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.NotPrepared
            $wpdb->query( $wpdb->prepare( $sql_delete, $fecha_corte ) );
        }

        return new WP_REST_Response( array( 'status' => 'ok' ), 200 );
    }

    return new WP_REST_Response( array( 'status' => 'error' ), 400 );
}