<?php
/**
 * Plugin Name:       Minutemailer
 * Description:       Easily add Minutemailer subscription forms to your WordPress site with a customizable Gutenberg block.
 * Version:           0.1.6
 * Requires at least: 6.7
 * Requires PHP:      8.1
 * Author:            Minutemailer
 * Author URI:        https://minutemailer.com
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       minutemailer
 *
 * @package CreateBlock
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Include the API helper class
require_once plugin_dir_path( __FILE__ ) . 'includes/class-minutemailer-api.php';
/**
 * Registers the block using a `blocks-manifest.php` file, which improves the performance of block type registration.
 * Behind the scenes, it also registers all assets so they can be enqueued
 * through the block editor in the corresponding context.
 *
 * @see https://make.wordpress.org/core/2025/03/13/more-efficient-block-type-registration-in-6-8/
 * @see https://make.wordpress.org/core/2024/10/17/new-block-type-registration-apis-to-improve-performance-in-wordpress-6-7/
 */
function minutemailer_block_init() {
	/**
	 * Registers the block(s) metadata from the `blocks-manifest.php` and registers the block type(s)
	 * based on the registered block metadata.
	 * Added in WordPress 6.8 to simplify the block metadata registration process added in WordPress 6.7.
	 *
	 * @see https://make.wordpress.org/core/2025/03/13/more-efficient-block-type-registration-in-6-8/
	 */
	if ( function_exists( 'wp_register_block_types_from_metadata_collection' ) ) {
		wp_register_block_types_from_metadata_collection( __DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php' );
		return;
	}

	/**
	 * Registers the block(s) metadata from the `blocks-manifest.php` file.
	 * Added to WordPress 6.7 to improve the performance of block type registration.
	 *
	 * @see https://make.wordpress.org/core/2024/10/17/new-block-type-registration-apis-to-improve-performance-in-wordpress-6-7/
	 */
	if ( function_exists( 'wp_register_block_metadata_collection' ) ) {
		wp_register_block_metadata_collection( __DIR__ . '/build', __DIR__ . '/build/blocks-manifest.php' );
	}
	/**
	 * Registers the block type(s) in the `blocks-manifest.php` file.
	 *
	 * @see https://developer.wordpress.org/reference/functions/register_block_type/
	 */
	$manifest_data = require __DIR__ . '/build/blocks-manifest.php';
	foreach ( array_keys( $manifest_data ) as $block_type ) {
		register_block_type( __DIR__ . "/build/{$block_type}" );
	}
}
add_action( 'init', 'minutemailer_block_init' );

/**
 * Add Minutemailer settings page to the WordPress admin menu.
 */
function minutemailer_wp_add_admin_menu() {
	add_menu_page(
		__( 'Minutemailer', 'minutemailer' ),        // Page title
		__( 'Minutemailer', 'minutemailer' ),        // Menu title
		'manage_options',      // Capability required
		'minutemailer',        // Menu slug
		'minutemailer_wp_settings_page', // Callback function
		'dashicons-email',     // Icon (email icon)
		100                    // Position in menu
	);
}
add_action( 'admin_menu', 'minutemailer_wp_add_admin_menu' );

/**
 * Register Minutemailer settings.
 */
function minutemailer_wp_register_settings() {
	register_setting(
		'minutemailer_settings',
		'minutemailer_api_token',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => '',
		)
	);
}
add_action( 'admin_init', 'minutemailer_wp_register_settings' );

/**
 * Register REST API endpoints for Minutemailer.
 */
function minutemailer_wp_register_rest_routes() {
	register_rest_route(
		'minutemailer/v1',
		'/lists',
		array(
			'methods'             => 'GET',
			'callback'            => 'minutemailer_wp_get_lists',
			'permission_callback' => function() {
				return current_user_can( 'edit_posts' );
			},
		)
	);

	register_rest_route(
		'minutemailer/v1',
		'/contacts',
		array(
			'methods'             => 'POST',
			'callback'            => 'minutemailer_wp_add_contact',
			'permission_callback' => '__return_true', // Allow public access for form submissions
		)
	);
}
add_action( 'rest_api_init', 'minutemailer_wp_register_rest_routes' );

/**
 * Localize the view script with the REST API URL.
 */
function minutemailer_wp_localize_view_script() {
    // Get the script handle from the block.json viewScript
    $script_handle = 'minutemailer-subscribe-view-script';

    // Localize the script with the REST API URL
    wp_localize_script(
        $script_handle,
        'minutemailerWpSettings',
        array(
            'apiUrl' => rest_url('minutemailer/v1/contacts'),
            'nonce' => wp_create_nonce('wp_rest')
        )
    );
}

/**
 * Set script translations for the block editor and frontend.
 */
function minutemailer_wp_set_script_translations() {
    wp_set_script_translations( 'minutemailer-subscribe-editor-script', 'minutemailer', plugin_dir_path( __FILE__ ) . 'languages' );
    wp_set_script_translations( 'minutemailer-subscribe-view-script', 'minutemailer', plugin_dir_path( __FILE__ ) . 'languages' );
}
add_action( 'wp_enqueue_scripts', 'minutemailer_wp_localize_view_script' );
add_action( 'init', 'minutemailer_wp_set_script_translations', 20 );

/**
 * Add a contact to a Minutemailer list.
 *
 * @param WP_REST_Request $request The request object.
 * @return WP_REST_Response|WP_Error Response object or error.
 */
function minutemailer_wp_add_contact( $request ) {
	$api_token = get_option( 'minutemailer_api_token', '' );

	if ( empty( $api_token ) ) {
		return new WP_Error(
			'minutemailer_no_token',
			__( 'No API token provided', 'minutemailer' ),
			array( 'status' => 400 )
		);
	}

	// Get parameters from request
	$params = $request->get_params();

	// Validate required fields
	if ( empty( $params['email'] ) ) {
		return new WP_Error(
			'minutemailer_missing_email',
			__( 'Email is required', 'minutemailer' ),
			array( 'status' => 400 )
		);
	}

	if ( empty( $params['list_id'] ) ) {
		return new WP_Error(
			'minutemailer_missing_list',
			__( 'List ID is required', 'minutemailer' ),
			array( 'status' => 400 )
		);
	}

	$api = new Minutemailer_API( $api_token );
	$email = sanitize_email( $params['email'] );
	$response = $api->request( 'contacts/' . $email );

	if ( ! is_wp_error( $response ) ) {
		$response = $api->request( 'contact-lists/' . $params['list_id'] . '/add-contact/' . $response['id'], 'PUT' );

		if ( is_wp_error( $response ) ) {
			return new WP_Error(
				'minutemailer_api_error',
				$response->get_error_message(),
				array( 'status' => 400, 'data' => $response->get_error_data(), 'list_id' => $params['list_id'] )
			);
		}
	}

	// Contact does not exist, create a new one
	$data = array(
		'email' => $email,
		'contact_lists' => array(
			$params['list_id'],
		),
	);

	// Add name if provided
	if ( ! empty( $params['name'] ) ) {
		$name       = sanitize_text_field( $params['name'] );
		$name_parts = array_map( 'trim', explode( ' ', $name ) );
		$first_name = $name_parts[0];
		$last_name  = implode( ' ', array_slice( $name_parts, 1 ) );

		if ( $first_name ) {
			$data['first_name'] = $first_name;
		}

		if ( $last_name ) {
			$data['last_name'] = $last_name;
		}
	}

	// Make API request
	$response = $api->request( 'contacts?send_confirmation=true', 'POST', $data );

	if ( is_wp_error( $response ) ) {
		return new WP_Error(
			'minutemailer_api_error',
			$response->get_error_message(),
			array( 'status' => 400, 'data' => $response->get_error_data(), 'list_id' => $params['list_id'] )
		);
	}

	return rest_ensure_response( array(
		'success' => true,
		'message' => __( 'Contact added successfully', 'minutemailer' ),
	) );
}

/**
 * Get Minutemailer lists.
 *
 * @return WP_REST_Response|WP_Error Response object or error.
 */
function minutemailer_wp_get_lists() {
	$api_token = get_option( 'minutemailer_api_token', '' );

	if ( empty( $api_token ) ) {
		return new WP_Error(
			'minutemailer_no_token',
			__( 'No API token provided', 'minutemailer' ),
			array( 'status' => 400 )
		);
	}

	$api = new Minutemailer_API( $api_token );
	$response = $api->request( 'contact-lists' );

	if ( is_wp_error( $response ) ) {
		return new WP_Error(
			'minutemailer_api_error',
			$response->get_error_message(),
			array( 'status' => 400 )
		);
	}

	// Check if the response has the expected format
	if ( ! isset( $response['items'] ) || ! is_array( $response['items'] ) ) {
		return new WP_Error(
			'minutemailer_api_error',
			__( 'Invalid API response format', 'minutemailer' ),
			array( 'status' => 400 )
		);
	}

	return rest_ensure_response( $response['items'] );
}

/**
 * Enqueue admin scripts and styles.
 */
function minutemailer_wp_admin_enqueue_scripts() {
	// Ensure dashicons are loaded
	wp_enqueue_style( 'dashicons' );
}
add_action( 'admin_enqueue_scripts', 'minutemailer_wp_admin_enqueue_scripts' );

/**
 * Render the Minutemailer settings page.
 */
function minutemailer_wp_settings_page() {
	// Check user capabilities
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Check if settings were updated
	$settings_updated = false;
	if ( isset( $_GET['settings-updated'] ) &&
	     isset( $_GET['_wpnonce'] ) &&
	     wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'minutemailer_settings' ) ) {
		$settings_updated = (bool) $_GET['settings-updated'];
	}

	// Get the current API token value
	$api_token = get_option( 'minutemailer_api_token', '' );

	// Check connection status if API token exists
	$connection_status = array(
		'status'  => 'unknown',
		'message' => __( 'No API token provided', 'minutemailer' ),
	);

	if ( ! empty( $api_token ) ) {
		$api = new Minutemailer_API( $api_token );
		$connection_status = $api->get_connection_status();
	}
	?>
	<div class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

		<?php if ( $settings_updated ) : ?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'Settings saved.', 'minutemailer' ); ?></p>
			</div>
		<?php endif; ?>

		<form method="post" action="options.php">
			<?php
			// Output security fields
			settings_fields( 'minutemailer_settings' );
			?>

			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">
						<label for="minutemailer_api_token"><?php esc_html_e( 'API Token', 'minutemailer' ); ?></label>
					</th>
					<td>
						<input type="text"
							   id="minutemailer_api_token"
							   name="minutemailer_api_token"
							   value="<?php echo esc_attr( $api_token ); ?>"
							   class="regular-text"
						/>

						<?php if ( ! empty( $api_token ) ) : ?>
							<div class="minutemailer-connection-status" style="margin-top: 10px;">
								<strong><?php esc_html_e( 'Connection status:', 'minutemailer' ); ?></strong>
								<?php if ( 'success' === $connection_status['status'] ) : ?>
									<span style="color: green; margin-left: 5px;">
										<span class="dashicons dashicons-yes"></span>
										<?php echo esc_html( $connection_status['message'] ); ?>
									</span>
								<?php else : ?>
									<span style="color: red; margin-left: 5px;">
										<span class="dashicons dashicons-no"></span>
										<?php echo esc_html( $connection_status['message'] ); ?>
									</span>
								<?php endif; ?>
							</div>
						<?php endif; ?>

						<p class="description">
							<?php
							printf(
								/* translators: %s: URL to Minutemailer API settings */
								esc_html__( 'Enter your Minutemailer API token. You can find or generate your API token in your %s.', 'minutemailer' ),
								'<a href="https://app.minutemailer.com/u/settings/api" target="_blank">' . esc_html__( 'Minutemailer account settings', 'minutemailer' ) . '</a>'
							);
							?>
						</p>
					</td>
				</tr>
			</table>

			<?php submit_button(); ?>
		</form>
	</div>
	<?php
}
