<?php
/**
 * Minutemailer API Helper Class
 *
 * @package MinutemailerWP
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Minutemailer_API
 *
 * Helper class for communicating with the Minutemailer API.
 */
class Minutemailer_API {

	/**
	 * API base endpoint
	 *
	 * @var string
	 */
	private $api_base = 'https://api-gateway.minutemailer.com';

	/**
	 * API token
	 *
	 * @var string
	 */
	private $api_token;

	/**
	 * Constructor
	 *
	 * @param string $api_token The API token for authentication.
	 */
	public function __construct( $api_token = '' ) {
		$this->api_token = $api_token ?: get_option( 'minutemailer_api_token', '' );
	}

	/**
	 * Make a request to the Minutemailer API
	 *
	 * @param string $endpoint The API endpoint to call.
	 * @param string $method   The HTTP method to use (GET, POST, etc.).
	 * @param array  $data     The data to send with the request.
	 * @return array|WP_Error  The response or WP_Error on failure.
	 */
	public function request( $endpoint, $method = 'GET', $data = array() ) {
		$url = trailingslashit( $this->api_base ) . ltrim( $endpoint, '/' );

		$args = array(
			'method'  => $method,
			'headers' => array(
				'Accept'        => 'application/json',
				'Content-Type'  => 'application/json',
				'User-Agent'    => 'minutemailer-gateway-api/v1',
				'X-Auth-Token'  => $this->api_token,
			),
			'timeout' => 30,
		);

		// Add request body for non-GET requests
		if ( 'GET' !== $method && ! empty( $data ) ) {
			$args['body'] = wp_json_encode( $data );
		}

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		// Handle API errors
		if ( $response_code >= 400 ) {
			$error_message = isset( $response_data['message'] ) ? $response_data['message'] : __( 'Unknown API error', 'minutemailer' );
			return new WP_Error( 'api_error', $error_message, array( 'status' => $response_code ) );
		}

		return $response_data;
	}

	/**
	 * Check connection to the API
	 *
	 * @return array|WP_Error The response or WP_Error on failure.
	 */
	public function check_connection() {
		return $this->request( 'contact-lists' );
	}

	/**
	 * Get connection status
	 *
	 * @return array Connection status information.
	 */
	public function get_connection_status() {
		if ( empty( $this->api_token ) ) {
			return array(
				'status'  => 'error',
				'message' => __( 'API token is missing', 'minutemailer' ),
			);
		}

		$response = $this->check_connection();

		if ( is_wp_error( $response ) ) {
			return array(
				'status'  => 'error',
				'message' => $response->get_error_message(),
			);
		}

		return array(
			'status'  => 'success',
			'message' => __( 'Connected successfully', 'minutemailer' ),
			'data'    => $response,
		);
	}
}
