<?php
/**
 * Plugin Name: MessengerOS for WooCommerce
 * Plugin URI: https://messengeros.com
 * Description: With the MessengerOS platform and this plugin, we give you the tools — and the support —  to turn your WooCommerce store into an email marketing machine.
 * Version: 1.0.2
 * Author: MessengerOS
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: messengeros-for-woocommerce
 * Domain Path: /languages
 *
 * Requires Plugins: woocommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin basename constant early for use in deactivation
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_BASENAME' ) ) {
	define( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
}

// Check if WooCommerce is active
if ( ! function_exists( 'is_plugin_active' ) ) {
    include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
}

if ( ! is_plugin_active( 'woocommerce/woocommerce.php' ) && ! class_exists( 'WooCommerce' ) ) {
    add_action( 'admin_notices', 'messengeros_woocommerce_missing_notice' );
    add_action( 'admin_init', 'messengeros_woocommerce_deactivate_plugin' );
    return;
}

if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_VERSION' ) ) {	define( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_VERSION', '1.0.2' );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_DIR' ) ) {
	define( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_DIR', __DIR__ );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_URL' ) ) {
    define( 'MESSENGEROS_WOOCOMMERCE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_EVENTS_API_URL' ) ) {
    define( 'MESSENGEROS_WOOCOMMERCE_EVENTS_API_URL', 'https://e.messengeros.com' );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_PLATFORM_URL' ) ) {
    define( 'MESSENGEROS_WOOCOMMERCE_PLATFORM_URL', 'https://app.messengeros.com' );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_INBOUND_API_URL' ) ) {
    define( 'MESSENGEROS_WOOCOMMERCE_INBOUND_API_URL', getenv('MESSENGEROS_INBOUND_API_URL') ?: 'https://inbound.messengeros.com' );
}
if ( ! defined( 'MESSENGEROS_WOOCOMMERCE_REGISTRATION_SECRET' ) ) {
    // Pre-bundled registration secret for automatic OAuth client registration
    // This allows stores to connect automatically without manual configuration
    define( 'MESSENGEROS_WOOCOMMERCE_REGISTRATION_SECRET', getenv('MESSENGEROS_REGISTRATION_SECRET') ?: 'a474c13a9652b257f0dabca94f08ad3cb857629c71150efb89ecbfdb8a2238d8' );
}

/**
 * Admin notice when WooCommerce is not installed/active
 * Shows error message directing users to use standalone WordPress plugin
 * Called by: WordPress admin_notices hook when WooCommerce is missing
 */
function messengeros_woocommerce_missing_notice() {
    ?>
    <div class="notice notice-error">
        <p><strong>MessengerOS for WooCommerce</strong> requires WooCommerce to be installed and active.</p>
        <p>This plugin is designed specifically for WooCommerce stores. If you want to use MessengerOS on a regular WordPress site, please install our <a href="https://wordpress.org/plugins/messengeros/" target="_blank">MessengerOS WordPress plugin</a> from the WordPress repository instead.</p>
    </div>
    <?php
}

/**
 * Deactivates the plugin when WooCommerce is not available
 * Prevents plugin functionality from running without required dependency
 * Called by: WordPress admin_init hook when WooCommerce is missing
 */
function messengeros_woocommerce_deactivate_plugin() {
    if ( is_plugin_active( MESSENGEROS_WOOCOMMERCE_PLUGIN_BASENAME ) ) {
        deactivate_plugins( MESSENGEROS_WOOCOMMERCE_PLUGIN_BASENAME );
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Not processing form data, just cleaning up activation parameter
        if ( isset( $_GET['activate'] ) ) {
            unset( $_GET['activate'] );
        }
    }
}

include_once 'inc/helpers.php';
include_once 'inc/class-store-identifier.php';
include_once 'inc/oauth-handler.php';
include_once 'inc/oauth-auto-register.php';
include_once 'inc/woocommerce-integration.php';
include_once 'inc/admin-product-sync.php';
include_once 'inc/initial-sync-handler.php';
include_once 'inc/class-messengeros-api-client.php';
include_once 'inc/class-events-api-client.php';
include_once 'inc/class-product-tracker.php';
include_once 'inc/class-event-hooks.php';
include_once 'inc/admin-forms.php';
include_once 'inc/class-cart-automation-rest-endpoint.php';
include_once 'inc/class-cart-state-tracker.php';
include_once 'inc/class-cart-event-formatter.php';
include_once 'inc/class-cart-event-tracker.php';
include_once 'inc/class-order-event-formatter.php';
include_once 'inc/class-order-event-tracker.php';
include_once 'inc/class-customer-event-formatter.php';
include_once 'inc/class-customer-event-tracker.php';
include_once 'inc/class-product-event-formatter.php';
include_once 'inc/class-product-event-tracker.php';

// Plugin activation hook
register_activation_hook( __FILE__, 'messengeros_plugin_activate' );

/**
 * Plugin activation handler - called by WordPress when plugin is activated
 * Sets up rewrite rules for OAuth callback and schedules initial sync
 * Called by: WordPress plugin activation system
 */
function messengeros_plugin_activate() {
    // Register OAuth callback rewrite rule
    add_rewrite_rule( '^messos-woocommerce-oauth-callback/?', 'index.php?messos-woocommerce-oauth-callback=1', 'top' );
    add_rewrite_tag( '%messos-woocommerce-oauth-callback%', '([^&]+)' );
    
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Set flag to check for initial sync after OAuth connection
    update_option( 'messengeros_woocommerce_needs_initial_sync', true );
    
    // Schedule sync check
    if ( ! wp_next_scheduled( 'messengeros_woocommerce_check_initial_sync' ) ) {
        wp_schedule_single_event( time() + 60, 'messengeros_woocommerce_check_initial_sync' );
    }
}

// Plugin deactivation hook
register_deactivation_hook( __FILE__, 'messengeros_plugin_deactivate' );

/**
 * Plugin deactivation handler - called by WordPress when plugin is deactivated
 * Cleans up rewrite rules and scheduled events
 * Called by: WordPress plugin deactivation system
 */
function messengeros_plugin_deactivate() {
    // Flush rewrite rules to remove our custom rule
    flush_rewrite_rules();
    
    // Clear scheduled events
    wp_clear_scheduled_hook( 'messengeros_check_initial_sync' );
    wp_clear_scheduled_hook( 'messos_woocommerce_scheduled_product_sync' );
}

class MessengerOS_WooCommerce {

    private $oauth_handler;
    private $wc_integration;
    private $event_tracker;
    private $product_sync;
    private $cart_automation_endpoint;
    private $cart_event_tracker;
    private $order_event_tracker;
    private $customer_event_tracker;
    private $product_event_tracker;

    /**
     * Constructor - initializes plugin components and hooks
     * Sets up OAuth handler, WooCommerce integration, product sync, and event tracking
     * Called by: WordPress when plugin instance is created
     */
    public function __construct() {
        $this->oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        $this->wc_integration = new MessengerOS_WooCommerce_Integration( $this->oauth_handler );

        // Initialize product sync
        require_once plugin_dir_path( __FILE__ ) . 'inc/product-sync.php';
        $this->product_sync = new MessengerOS_WooCommerce_Product_Sync( $this->oauth_handler );

        // Initialize event tracker if connected
        if ( $this->oauth_handler->is_connected() ) {
            require_once plugin_dir_path( __FILE__ ) . 'inc/class-messengeros-event-tracker.php';
            $this->event_tracker = new MessengerOS_WooCommerce_Event_Tracker();
        }

        // Initialize cart automation REST endpoint
        $this->cart_automation_endpoint = new MessengerOS_WooCommerce_Cart_Automation_REST_Endpoint( $this->oauth_handler );

        // Initialize cart event tracker if connected
        if ( $this->oauth_handler->is_connected() ) {
            $this->cart_event_tracker = new MessengerOS_WooCommerce_Cart_Event_Tracker( $this->oauth_handler );
        }

        // Initialize order event tracker if connected
        if ( $this->oauth_handler->is_connected() ) {
            $this->order_event_tracker = new MessengerOS_WooCommerce_Order_Event_Tracker( $this->oauth_handler );
        }

        // Initialize customer event tracker if connected
        if ( $this->oauth_handler->is_connected() ) {
            $this->customer_event_tracker = new MessengerOS_WooCommerce_Customer_Event_Tracker( $this->oauth_handler );
        }

        // Initialize product event tracker if connected
        if ( $this->oauth_handler->is_connected() ) {
            $this->product_event_tracker = new MessengerOS_WooCommerce_Product_Event_Tracker( $this->oauth_handler );
        }

        // Register webhook processor
        add_action( 'messos_woocommerce_send_webhook', [ $this->wc_integration, 'process_webhook' ], 10, 2 );
        
        // Check if we need to flush rewrite rules
        if ( get_option( 'messengeros_flush_rewrite_rules' ) ) {
            add_action( 'init', [ $this, 'flush_rewrite_rules' ], 999 );
        }
		add_action( 'admin_menu', [ $this, 'messos_woocommerce_add_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'messos_woocommerce_remove_notices' ], 1 );
		add_action( 'admin_init', [ $this, 'messos_woocommerce_register_settings' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'messos_woocommerce_load_styles_admin' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'messos_woocommerce_load_styles' ] );
        add_action( 'wp_ajax_messos_woocommerce_send_subscriber', 'messos_woocommerce_send_subscriber' );
        add_action( 'wp_ajax_nopriv_messos_woocommerce_send_subscriber', 'messos_woocommerce_send_subscriber' );
        add_shortcode( 'messos-woocommerce-newsletter-form', [ $this, 'messos_woocommerce_newsletter_form_shortcode' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'messos_woocommerce_admin_enqueue_scripts' ] );
        add_filter( 'safe_style_css', [ $this, 'messengeros_allow_inline_styles' ] );
    }

    /**
     * Removes admin notices from plugin pages to keep interface clean
     * Uses output buffering to filter out WordPress admin notices
     * Called by: WordPress admin_init hook
     */
    public function messos_woocommerce_remove_notices()
    {
        add_action('admin_init', function () {
            ob_start(function ($buffer) {
                // Remove all admin notices
                return preg_replace('/<div[^>]*class="[^"]*\bnotice\b[^"]*"[^>]*>.*?<\/div>/is', '', $buffer);
            });
        });
    }
    /**
     * Allows additional CSS properties for inline styles in newsletter forms
     * Adds styling properties to WordPress safe style list for form customization
     * Called by: WordPress safe_style_css filter
     */
    function messengeros_allow_inline_styles( $styles ) {
        // Add the CSS properties you need to allow
        $styles[] = 'background-color';
        $styles[] = 'color';
        $styles[] = 'padding';
        $styles[] = 'margin';
        $styles[] = 'border';
        $styles[] = 'font-size';
        $styles[] = 'text-align';
        $styles[] = 'display';
        return $styles;
    }
    
    public function flush_rewrite_rules() {
        flush_rewrite_rules();
        delete_option( 'messengeros_flush_rewrite_rules' );
    }

    public function messos_woocommerce_register_settings() {
        // Settings registration removed - tracking is controlled via database only
    }

	public function messos_woocommerce_add_admin_menu() {
		add_menu_page(
			'MessengerOS',
			'MessengerOS',
			'manage_options',
			'messos_woocommerce_settings',
			[ $this, 'messos_woocommerce_render_settings_page' ],
			'dashicons-email',
			61
		);
	}


	public function messos_woocommerce_render_settings_page() {
		$is_connected = $this->oauth_handler->is_connected();
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading URL parameters for display only, not processing form data
		$oauth_status = sanitize_text_field( wp_unslash($_GET['oauth'] ?? '' ));
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading URL parameters for display only, not processing form data
		$oauth_error = sanitize_text_field( wp_unslash($_GET['error'] ?? '' ));
		
		// Handle OAuth messages
		$oauth_message = '';
		$message_type = '';
		
		if ( $oauth_status === 'connected' ) {
			$oauth_message = 'Successfully connected to MessengerOS!';
			$message_type = 'success';
		} elseif ( $oauth_status === 'error' ) {
			// Handle different error types
			switch ( $oauth_error ) {
				case 'access_denied':
					$oauth_message = 'Authorization was denied. Please try again or contact support if you need assistance.';
					break;
				case 'no_code':
					$oauth_message = 'Authorization failed: No authorization code received.';
					break;
				default:
					$oauth_message = 'Authorization failed: ' . esc_html( $oauth_error );
			}
			$message_type = 'error';
		}
		?>
        <div class="wrap">
            <?php 
            // Include admin header
            include plugin_dir_path( __FILE__ ) . 'inc/admin-header.php';
            ?>

            <?php if ($oauth_message): ?>
                <div class="notice notice-<?php echo esc_attr($message_type); ?> is-dismissible">
                    <p><?php echo esc_html($oauth_message); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (!$is_connected): ?>
                <div class="messos-woocommerce-oauth-section" style="margin: 0 auto;margin-top:100px;width:400px;padding-top:0">
                    <h2>Connect to MessengerOS</h2>
                    <p>Connect your WooCommerce store to MessengerOS to enable email and SMS marketing automation.</p>
                    <br />
                    <div class="messos-woocommerce-oauth-buttons">
                        <a href="<?php echo esc_url($this->oauth_handler->get_authorization_url('register')); ?>" class="button button-primary">Create Account</a>
                        <a href="<?php echo esc_url($this->oauth_handler->get_authorization_url('connect')); ?>" class="button button-secondary">Connect Account</a>
                    </div>
                </div>
            <?php else: ?>
                <div class="messos-woocommerce-oauth-section" style="float:right">
                    <div class="messos-woocommerce-connected-status">✓ Connected to MessengerOS</div>
                    <div class="messos-connected-button-holder">
                        <button type="button" class="button button-secondary" id="messos-woocommerce-disconnect-oauth">Disconnect</button>
                    </div>
                    <div style="clear:both"></div>
                </div>
                <div style="clear:both"></div>
            <?php endif; ?>
            <div class="messos-welcome-section">
                <div style="font-family: Arial, sans-serif; max-width: 800px; margin: auto; line-height: 1.6;">

                    <h2 style="font-size: 28px;">Welcome to the MessengerOS Plugin for WooCommerce</h2>

                    <p>
                        We’re a passionate team of <strong>Email Marketing experts</strong>, and our goal is simple:
                        <em>to help you build and grow your business through smart, effective email campaigns</em>.
                        With the MessengerOS platform and this plugin, we give you the tools — and the support —
                        to turn your WooCommerce store into an email marketing machine.
                    </p>

                    <p>
                        Have questions or need help? <strong>You can ask us anything</strong>, send us an email at: <a href="mailto:support@messengeros.com">support@messengeros.com</a> . We’re here to guide you
                        every step of the way — from setup to advanced strategy.
                    </p>

                    <hr style="margin: 30px 0;">
                    <h3 style="font-size: 22px;">What the Plugin does (right now)</h3>

                    <p>THere’s what you can do:</p>

                    <ul style="margin-left: 20px;">
                        <li style="margin-bottom: 15px;">
                            <strong>Collect Subscribers easily</strong><br>
                            Use the built-in form from the “Form” section to start building your email list right away.
                        </li>
                        <li style="margin-bottom: 15px;">
                            <strong>Build Newsletters faster</strong><br>
                            When creating a campaign in MessengerOS, your WooCommerce products are already waiting for you
                            in the builder. No need to copy-paste or upload them manually.
                        </li>
                        <li style="margin-bottom: 15px;">
                            <strong>Product Syncing Made Easy</strong><br>
                            <ul style="list-style-type: disc; margin-left: 20px;">
                                <li>Your store’s products are automatically synced with MessengerOS every 6 hours.</li>
                                <li>Want to do it right now? Just hit the <strong>Sync Now</strong> button in the plugin.</li>
                            </ul>
                        </li>
                        <li style="margin-bottom: 15px;">
                            <strong>Track Product Page Views</strong><br>
                            We automatically track which products your visitors view. This is allowing you to send <em>retargeting emails</em> based on each customer’s interest.
                        </li>
                    </ul>
                    <p>
                        More powerful features are on the way, and we're excited to grow with you.<br>
                        <strong>Let’s create better emails, together.</strong>
                    </p>
                </div>
            </div>
        </div>
		<?php
	}


	public function messos_woocommerce_load_styles_admin() {
		wp_enqueue_style( 'messengeros-woocommerce-styles-admin', MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . '/assets/header.css', [], '1.0.2' );
	}

	public function messos_woocommerce_load_styles() {
		wp_enqueue_style( 'messengeros-woocommerce-styles-subscribe',
            MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . '/assets/form.css?',
            [],
            '1.0.3'
        );
        wp_enqueue_script(
                'messengeros-woocommerce-js-subscribe',
                MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . '/assets/js/messos-woocommerce-subscribe.js',
                [],
                '1.0.2',
                true
        );
        wp_localize_script( 'messengeros-woocommerce-js-subscribe', 'messosWoocommerceData', [
            'ajax_url' => admin_url( 'admin-ajax.php' )
        ] );
	}




	public function messos_woocommerce_admin_enqueue_scripts($hook) {
        // Only enqueue on main settings page for OAuth disconnect functionality
        if ($hook === 'toplevel_page_messos_woocommerce_settings') {
            // Enqueue admin settings styles
            wp_enqueue_style(
                'messos-woocommerce-admin-settings-css',
                MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . '/assets/css/admin-settings.css',
                [],
                '1.0.0'
            );
            
            // Enqueue admin scripts
            wp_enqueue_script(
                'messos-woocommerce-admin-js',
                MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . '/assets/js/messos-woocommerce-admin.js',
                [ 'jquery' ],
                '1.0.2',
                true
            );
            wp_localize_script( 'messos-woocommerce-admin-js', 'messosWoocommerceAjax', [
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'messos_woocommerce_disconnect_oauth' ),
            ] );
        }
    }

	public function messos_woocommerce_newsletter_form_shortcode($atts) {
		$template = plugin_dir_path( __FILE__ ) . 'templates/form-subscribe.php';
		$attributes = shortcode_atts(array(
                'btn_txt' => esc_html__('Submit', "messengeros-for-woocommerce"),
                'email_txt' => esc_html__('Email', "messengeros-for-woocommerce"),
                'success_txt' => esc_html__('Thank you for subscribing!', 'messengeros-for-woocommerce') . '<br />' . __('You\'ve been successfully added to our mailing list.', "messengeros-for-woocommerce"),
                'fail_txt' => esc_html__('Subscription failed', "messengeros-for-woocommerce"),
        ), $atts);
		if ( file_exists( $template ) ) {
			ob_start();
			include $template;
			return ob_get_clean();
		} else {
            return '<p><strong>' . esc_html__('Error', 'messengeros-for-woocommerce') . '</strong>' . esc_html__('Form template not found.','messengeros-for-woocommerce') . '</p>';
        }
	}

    public static function messos_woocommerce_get_allowed_html() {
        return [
            'div' => [
                'id' => true,
                'class' => true,
                'style' => true,
            ],
            'form' => [
                'id' => true,
                'class' => true,
                'action' => true,
                'method' => true,
            ],
            'label' => [
                'for' => true,
                'class' => true,
            ],
            'input' => [
                'type' => true,
                'id' => true,
                'name' => true,
                'value' => true,
                'required' => true,
                'placeholder' => true,
                'checked' => true,
                'class' => true,
            ],
            'button' => [
                'type' => true,
                'id' => true,
                'disabled' => true,
                'class' => true,
            ],
            'br' => [],
            'span' => [
                'id' => true,
                'class' => true,
                'style' => true,
            ],
            'p' => [
                'class' => true,
                'style' => true,
            ],
            'strong' => [],
            'em' => [],
            'b' => [],
            'i' => [],
            'u' => [],
            'small' => [],
            'a' => [
                'href' => true,
                'title' => true,
                'target' => true,
                'rel' => true,
                'class' => true,
            ],
        ];
    }


}

// Test log to verify logging works
// error_log('MessengerOS: Plugin loaded successfully');

new MessengerOS_WooCommerce();