<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_Integration {
    
    private $oauth_handler;
    
    public function __construct( $oauth_handler ) {
        $this->oauth_handler = $oauth_handler;
        
        // Hook into WooCommerce events only if OAuth is connected
        add_action( 'init', [ $this, 'init_webhooks' ] );
    }
    
    /**
     * Initialize webhooks if OAuth is connected
     */
    public function init_webhooks() {
        if ( ! $this->oauth_handler->is_connected() ) {
            return;
        }
        
        // Order webhooks
        add_action( 'woocommerce_order_status_changed', [ $this, 'order_status_changed' ], 10, 4 );
        add_action( 'woocommerce_new_order', [ $this, 'new_order_created' ], 10, 1 );
        add_action( 'woocommerce_update_order', [ $this, 'order_updated' ], 10, 1 );
        
        // Customer webhooks
        add_action( 'woocommerce_created_customer', [ $this, 'customer_created' ], 10, 3 );
        add_action( 'woocommerce_update_customer', [ $this, 'customer_updated' ], 10, 2 );
        
        // Product webhooks
        add_action( 'woocommerce_update_product', [ $this, 'product_updated' ], 10, 1 );
        add_action( 'woocommerce_new_product', [ $this, 'product_created' ], 10, 1 );
        add_action( 'woocommerce_delete_product', [ $this, 'product_deleted' ], 10, 1 );
    }
    
    /**
     * Handle order status change
     */
    public function order_status_changed( $order_id, $old_status, $new_status, $order ) {
        $data = $this->prepare_order_data( $order );
        $data['event'] = 'order.status_changed';
        $data['old_status'] = $old_status;
        $data['new_status'] = $new_status;
        
        $this->send_webhook( 'webhooks/order', $data );
    }
    
    /**
     * Handle new order creation
     */
    public function new_order_created( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }
        
        $data = $this->prepare_order_data( $order );
        $data['event'] = 'order.created';
        
        $this->send_webhook( 'webhooks/order', $data );
    }
    
    /**
     * Handle order update
     */
    public function order_updated( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }
        
        $data = $this->prepare_order_data( $order );
        $data['event'] = 'order.updated';
        
        $this->send_webhook( 'webhooks/order', $data );
    }
    
    /**
     * Handle customer creation
     */
    public function customer_created( $customer_id, $new_customer_data, $password_generated ) {
        $customer = new WC_Customer( $customer_id );
        
        $data = $this->prepare_customer_data( $customer );
        $data['event'] = 'customer.created';
        
        $this->send_webhook( 'webhooks/customer', $data );
    }
    
    /**
     * Handle customer update
     */
    public function customer_updated( $customer_id, $customer ) {
        if ( ! $customer instanceof WC_Customer ) {
            $customer = new WC_Customer( $customer_id );
        }
        
        $data = $this->prepare_customer_data( $customer );
        $data['event'] = 'customer.updated';
        
        $this->send_webhook( 'webhooks/customer', $data );
    }
    
    /**
     * Handle product update
     */
    public function product_updated( $product_id ) {
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return;
        }
        
        $data = $this->prepare_product_data( $product );
        $data['event'] = 'product.updated';
        
        $this->send_webhook( 'webhooks/product', $data );
    }
    
    /**
     * Handle product creation
     */
    public function product_created( $product_id ) {
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return;
        }
        
        $data = $this->prepare_product_data( $product );
        $data['event'] = 'product.created';
        
        $this->send_webhook( 'webhooks/product', $data );
    }
    
    /**
     * Handle product deletion
     */
    public function product_deleted( $product_id ) {
        $data = [
            'event' => 'product.deleted',
            'product_id' => $product_id,
            'timestamp' => current_time( 'mysql' ),
        ];
        
        $this->send_webhook( 'webhooks/product', $data );
    }
    
    /**
     * Prepare order data for webhook
     */
    private function prepare_order_data( $order ) {
        return [
            'id' => $order->get_id(),
            'order_number' => $order->get_order_number(),
            'status' => $order->get_status(),
            'currency' => $order->get_currency(),
            'total' => $order->get_total(),
            'subtotal' => $order->get_subtotal(),
            'tax_total' => $order->get_total_tax(),
            'shipping_total' => $order->get_shipping_total(),
            'discount_total' => $order->get_discount_total(),
            'customer_id' => $order->get_customer_id(),
            'billing' => [
                'first_name' => $order->get_billing_first_name(),
                'last_name' => $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
                'address_1' => $order->get_billing_address_1(),
                'address_2' => $order->get_billing_address_2(),
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'postcode' => $order->get_billing_postcode(),
                'country' => $order->get_billing_country(),
            ],
            'shipping' => [
                'first_name' => $order->get_shipping_first_name(),
                'last_name' => $order->get_shipping_last_name(),
                'address_1' => $order->get_shipping_address_1(),
                'address_2' => $order->get_shipping_address_2(),
                'city' => $order->get_shipping_city(),
                'state' => $order->get_shipping_state(),
                'postcode' => $order->get_shipping_postcode(),
                'country' => $order->get_shipping_country(),
            ],
            'line_items' => $this->get_order_line_items( $order ),
            'created_at' => $order->get_date_created() ? $order->get_date_created()->format( 'c' ) : '',
            'updated_at' => $order->get_date_modified() ? $order->get_date_modified()->format( 'c' ) : '',
        ];
    }
    
    /**
     * Get order line items
     */
    private function get_order_line_items( $order ) {
        $items = [];
        
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            
            $items[] = [
                'id' => $item->get_id(),
                'product_id' => $item->get_product_id(),
                'variation_id' => $item->get_variation_id(),
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'total' => $item->get_total(),
                'subtotal' => $item->get_subtotal(),
                'sku' => $product ? $product->get_sku() : '',
            ];
        }
        
        return $items;
    }
    
    /**
     * Prepare customer data for webhook
     */
    private function prepare_customer_data( $customer ) {
        return [
            'id' => $customer->get_id(),
            'email' => $customer->get_email(),
            'first_name' => $customer->get_first_name(),
            'last_name' => $customer->get_last_name(),
            'username' => $customer->get_username(),
            'billing' => [
                'first_name' => $customer->get_billing_first_name(),
                'last_name' => $customer->get_billing_last_name(),
                'email' => $customer->get_billing_email(),
                'phone' => $customer->get_billing_phone(),
                'address_1' => $customer->get_billing_address_1(),
                'address_2' => $customer->get_billing_address_2(),
                'city' => $customer->get_billing_city(),
                'state' => $customer->get_billing_state(),
                'postcode' => $customer->get_billing_postcode(),
                'country' => $customer->get_billing_country(),
            ],
            'shipping' => [
                'first_name' => $customer->get_shipping_first_name(),
                'last_name' => $customer->get_shipping_last_name(),
                'address_1' => $customer->get_shipping_address_1(),
                'address_2' => $customer->get_shipping_address_2(),
                'city' => $customer->get_shipping_city(),
                'state' => $customer->get_shipping_state(),
                'postcode' => $customer->get_shipping_postcode(),
                'country' => $customer->get_shipping_country(),
            ],
            'total_spent' => $customer->get_total_spent(),
            'order_count' => $customer->get_order_count(),
            'created_at' => $customer->get_date_created() ? $customer->get_date_created()->format( 'c' ) : '',
            'updated_at' => $customer->get_date_modified() ? $customer->get_date_modified()->format( 'c' ) : '',
        ];
    }
    
    /**
     * Prepare product data for webhook
     */
    private function prepare_product_data( $product ) {
        return [
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'slug' => $product->get_slug(),
            'sku' => $product->get_sku(),
            'price' => $product->get_price(),
            'regular_price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'status' => $product->get_status(),
            'stock_status' => $product->get_stock_status(),
            'stock_quantity' => $product->get_stock_quantity(),
            'type' => $product->get_type(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'categories' => wp_get_post_terms( $product->get_id(), 'product_cat', [ 'fields' => 'names' ] ),
            'tags' => wp_get_post_terms( $product->get_id(), 'product_tag', [ 'fields' => 'names' ] ),
            'images' => $this->get_product_images( $product ),
            'created_at' => $product->get_date_created() ? $product->get_date_created()->format( 'c' ) : '',
            'updated_at' => $product->get_date_modified() ? $product->get_date_modified()->format( 'c' ) : '',
        ];
    }
    
    /**
     * Get product images
     */
    private function get_product_images( $product ) {
        $images = [];
        
        // Featured image
        if ( $product->get_image_id() ) {
            $images[] = [
                'id' => $product->get_image_id(),
                'src' => wp_get_attachment_url( $product->get_image_id() ),
                'is_featured' => true,
            ];
        }
        
        // Gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ( $gallery_ids as $image_id ) {
            $images[] = [
                'id' => $image_id,
                'src' => wp_get_attachment_url( $image_id ),
                'is_featured' => false,
            ];
        }
        
        return $images;
    }
    
    /**
     * Send webhook to MessengerOS
     */
    private function send_webhook( $endpoint, $data ) {
        // Add store information
        $data['store_url'] = get_site_url();
        $data['store_name'] = get_bloginfo( 'name' );
        
        // Send in background to avoid blocking
        wp_schedule_single_event( time(), 'messos_send_webhook', [ $endpoint, $data ] );
    }
    
    /**
     * Actually send the webhook (called by cron)
     */
    public function process_webhook( $endpoint, $data ) {
        $this->oauth_handler->make_api_request( $endpoint, 'POST', $data );
    }
}