<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_Product_Sync {
    
    private $oauth_handler;
    private $batch_size = 50;
    
    public function __construct( $oauth_handler ) {
        $this->oauth_handler = $oauth_handler;
        
        // Register AJAX endpoints
        add_action( 'wp_ajax_messos_woocommerce_sync_products', [ $this, 'ajax_sync_products' ] );
        add_action( 'wp_ajax_messos_woocommerce_get_sync_status', [ $this, 'ajax_get_sync_status' ] );
        
        // Register scheduled sync
        add_action( 'messos_woocommerce_scheduled_product_sync', [ $this, 'scheduled_sync' ] );
        
        // Schedule sync if not already scheduled
        if ( ! wp_next_scheduled( 'messos_woocommerce_scheduled_product_sync' ) ) {
            // Default: sync every 6 hours
            wp_schedule_event( time(), 'sixhours', 'messos_woocommerce_scheduled_product_sync' );
        }
        
        // Add custom cron schedule
        add_filter( 'cron_schedules', [ $this, 'add_cron_schedules' ] );
    }
    
    /**
     * Add custom cron schedules
     */
    public function add_cron_schedules( $schedules ) {
        $schedules['twohours'] = [
            'interval' => 7200,
            'display' => __( 'Every 2 Hours', 'messengeros-for-woocommerce' )
        ];
        
        $schedules['sixhours'] = [
            'interval' => 21600,
            'display' => __( 'Every 6 Hours', 'messengeros-for-woocommerce' )
        ];
        
        return $schedules;
    }
    
    /**
     * Get product data for sync
     */
    private function get_product_data( $product ) {
        $data = [
            'id' => $product->get_id(),
            'sku' => $product->get_sku(),
            'name' => $product->get_name(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'url' => get_permalink( $product->get_id() ),
            'status' => $product->get_status(),
            'stock_status' => $product->get_stock_status(),
            'stock_quantity' => $product->get_stock_quantity(),
            'modified_date' => $product->get_date_modified()->format( 'Y-m-d H:i:s' ),
            'created_at' => $product->get_date_created()->format( 'c' ), // ISO 8601 format
            'updated_at' => $product->get_date_modified()->format( 'c' ), // ISO 8601 format for Data Warehouse
        ];
        
        // Get featured image
        $image_id = $product->get_image_id();
        if ( $image_id ) {
            $data['image'] = wp_get_attachment_url( $image_id );
        } else {
            // Fallback to demo image if exists
            $demo_image = get_post_meta( $product->get_id(), '_demo_image_url', true );
            $data['image'] = $demo_image ?: '';
        }
        
        // Get gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        $data['gallery'] = array_map( 'wp_get_attachment_url', $gallery_ids );
        
        // Get categories
        $categories = [];
        $category_ids = $product->get_category_ids();
        foreach ( $category_ids as $cat_id ) {
            $term = get_term( $cat_id, 'product_cat' );
            if ( $term && ! is_wp_error( $term ) ) {
                $categories[] = [
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                ];
            }
        }
        $data['categories'] = $categories;
        
        // Get brand from attributes
        $data['brand'] = '';
        $attributes = $product->get_attributes();
        foreach ( $attributes as $attribute ) {
            if ( strtolower( $attribute->get_name() ) === 'brand' ) {
                $options = $attribute->get_options();
                $data['brand'] = ! empty( $options ) ? $options[0] : '';
                break;
            }
        }
        
        // Get all attributes
        $attrs = [];
        foreach ( $attributes as $attribute ) {
            $attrs[ $attribute->get_name() ] = $attribute->get_options();
        }
        $data['attributes'] = $attrs;
        
        return $data;
    }
    
    /**
     * Sync products to MessengerOS
     */
    public function sync_products( $page = 1, $since_date = null ) {
         error_log( 'MessengerOS: sync_products() called - Page: ' . $page . ', Since date: ' . ( $since_date ?: 'null (full sync)' ) );
        
        // Check if OAuth is connected before attempting sync
        if ( ! $this->oauth_handler->is_connected() ) {
            // error_log( 'MessengerOS: sync_products() aborted - not connected to MessengerOS' );
            return [
                'success' => false,
                'synced' => 0,
                'failed' => 0,
                'total' => 0,
                'pages' => 0,
                'current_page' => $page,
                'errors' => [ 'Not connected to MessengerOS' ],
            ];
        }
        
        $args = [
            'post_type' => 'product',
            'posts_per_page' => $this->batch_size,
            'paged' => $page,
            'post_status' => 'publish',
            'orderby' => 'modified',
            'order' => 'DESC',
        ];
        
        // If syncing since a specific date
        if ( $since_date ) {
            $args['date_query'] = [
                [
                    'column' => 'post_modified',
                    'after' => $since_date,
                ]
            ];
            // error_log( 'MessengerOS: Filtering products modified after ' . $since_date );
        }
        
        $products = new WP_Query( $args );
        $results = [
            'success' => true,
            'synced' => 0,
            'failed' => 0,
            'total' => $products->found_posts,
            'pages' => $products->max_num_pages,
            'current_page' => $page,
            'errors' => [],
        ];
        
         error_log( 'MessengerOS: Query found ' . $products->found_posts . ' total products, ' . $products->max_num_pages . ' pages' );
        
        if ( $products->have_posts() ) {
            $batch_data = [];
            $product_ids = [];
            
            while ( $products->have_posts() ) {
                $products->the_post();
                $product = wc_get_product( get_the_ID() );
                
                if ( $product ) {
                    $product_data = $this->get_product_data( $product );
                    $batch_data[] = $product_data;
                    $product_ids[] = $product->get_id();

                    error_log( 'MessengerOS: Added product to batch - ID: ' . $product->get_id() .
                              ', SKU: ' . $product->get_sku() .
                              ', Name: ' . $product->get_name() .
                              ', Modified: ' . $product->get_date_modified()->format( 'Y-m-d H:i:s' ) );
                }
            }
            
             error_log( 'MessengerOS: Batch contains ' . count( $batch_data ) . ' products: [' . implode( ', ', $product_ids ) . ']' );
            
            // Send batch to MessengerOS Data Warehouse
            if ( ! empty( $batch_data ) ) {
                // Initialize API client
                if ( ! class_exists('MessengerOS_WooCommerce_API_Client') ) {
                    require_once plugin_dir_path( dirname( __FILE__ ) ) . 'inc/class-messengeros-api-client.php';
                }
                
                $api_client = new MessengerOS_WooCommerce_API_Client();
                
                 error_log( 'MessengerOS: Sending batch of ' . count( $batch_data ) . ' products to Data Warehouse API' );
                
                // Send to Data Warehouse
                $response = $api_client->send_dw_request( 'batch', 'POST', [
                    'products' => $batch_data,
                    'sync_type' => $since_date ? 'incremental' : 'full',
                ] );
                
                if ( is_wp_error( $response ) ) {
                    $results['success'] = false;
                    $error_msg = $response->get_error_message();
                    
                     error_log( 'MessengerOS: API request failed - ' . $error_msg );
                    
                    // Add more detailed error info
                    if ( $response->get_error_code() === 'api_error' ) {
                        $error_data = $response->get_error_data();
                        // error_log( 'MessengerOS: DW API Error Response: ' . $error_data );
                    }
                    
                    $results['errors'][] = $error_msg;
                    $results['failed'] = count( $batch_data );
                } else {
                    $results['synced'] = count( $batch_data );
                    
                     error_log( 'MessengerOS: Successfully synced ' . count( $batch_data ) . ' products' );
                    
                    if ( is_array( $response ) && isset( $response['body'] ) ) {
                        $body = json_decode( $response['body'], true );
                        if ( $body ) {
                            // error_log( 'MessengerOS: API Response: ' . json_encode( $body ) );
                        }
                    }
                    
                    // Update last sync time
                    update_option( 'messos_woocommerce_last_product_sync', current_time( 'mysql' ) );
                    // error_log( 'MessengerOS: Updated last sync time to ' . current_time( 'mysql' ) );
                }
            } else {
                // error_log( 'MessengerOS: No products to sync in this batch' );
            }
        } else {
            // error_log( 'MessengerOS: No products found matching criteria' );
        }
        
        wp_reset_postdata();
        
        return $results;
    }
    
    /**
     * AJAX handler for manual sync
     */
    public function ajax_sync_products() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Unauthorized' );
        }
        
        // Verify nonce
        $nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'messos_woocommerce_sync_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce' );
            return;
        }
        
        $page = isset( $_POST['page'] ) ? intval( $_POST['page'] ) : 1;
        $full_sync = isset( $_POST['full_sync'] ) && $_POST['full_sync'] === 'true';
        
        $since_date = null;
        if ( ! $full_sync ) {
            $last_sync = get_option( 'messos_woocommerce_last_product_sync' );
            if ( $last_sync ) {
                $since_date = $last_sync;
            }
        }
        
        $results = $this->sync_products( $page, $since_date );
        
        // Send the results directly without double-wrapping
        wp_send_json( $results );
    }
    
    /**
     * Scheduled sync (incremental)
     */
    public function scheduled_sync() {
        // error_log( 'MessengerOS: Starting scheduled sync at ' . current_time( 'mysql' ) );
        
        // Check if OAuth is connected before attempting sync
        if ( ! $this->oauth_handler->is_connected() ) {
            // error_log( 'MessengerOS: Skipping scheduled sync - not connected to MessengerOS' );
            return;
        }
        
        // Only sync modified products
        $last_sync = get_option( 'messos_woocommerce_last_product_sync' );
        
        if ( ! $last_sync ) {
            // error_log( 'MessengerOS: No previous sync found - initiating full sync' );
            // First sync - do full sync
            $this->full_sync();
        } else {
            // error_log( 'MessengerOS: Last sync was at ' . $last_sync . ' - starting incremental sync' );
            
            // Count products to sync
            $count_args = [
                'post_type' => 'product',
                'post_status' => 'publish',
                'date_query' => [
                    [
                        'column' => 'post_modified',
                        'after' => $last_sync,
                    ]
                ],
                'posts_per_page' => -1,
                'fields' => 'ids',
            ];
            $count_query = new WP_Query( $count_args );
            $products_to_sync = $count_query->found_posts;
            
            // error_log( 'MessengerOS: Found ' . $products_to_sync . ' products modified since last sync' );
            
            // Incremental sync
            $page = 1;
            $total_synced = 0;
            $total_failed = 0;
            
            do {
                // error_log( 'MessengerOS: Syncing page ' . $page . ' of products' );
                $results = $this->sync_products( $page, $last_sync );
                
                $total_synced += $results['synced'];
                $total_failed += $results['failed'];
                
                // error_log( 'MessengerOS: Page ' . $page . ' results - Synced: ' . $results['synced'] . ', Failed: ' . $results['failed'] );
                
                $page++;
            } while ( $page <= $results['pages'] && $results['success'] );
            
            // error_log( 'MessengerOS: Incremental sync completed - Total synced: ' . $total_synced . ', Total failed: ' . $total_failed );
        }
    }
    
    /**
     * Full sync all products
     */
    private function full_sync() {
        // error_log( 'MessengerOS: Starting full product sync' );
        
        // Get total product count
        $total_products = wp_count_posts( 'product' )->publish;
        // error_log( 'MessengerOS: Total published products in store: ' . $total_products );
        
        $page = 1;
        $total_synced = 0;
        $total_failed = 0;
        
        do {
            // error_log( 'MessengerOS: Full sync - processing page ' . $page );
            
            $results = $this->sync_products( $page );
            
            $total_synced += $results['synced'];
            $total_failed += $results['failed'];
            
            /* error_log( 'MessengerOS: Full sync - page ' . $page . ' complete. ' .
                      'Page results: synced=' . $results['synced'] . ', failed=' . $results['failed'] . '. ' .
                      'Running totals: synced=' . $total_synced . ', failed=' . $total_failed );*/
            
            $page++;
            
            // Add small delay to prevent server overload
            if ( $page <= $results['pages'] ) {
                // error_log( 'MessengerOS: Pausing 1 second before next page to prevent server overload' );
                sleep( 1 );
            }
        } while ( $page <= $results['pages'] && $results['success'] );
        
        // error_log( 'MessengerOS: Full sync completed - Total synced: ' . $total_synced . ', Total failed: ' . $total_failed );
    }
    
    /**
     * Get sync status
     */
    public function ajax_get_sync_status() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Unauthorized' );
        }
        
        // Verify nonce
        $nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'messos_woocommerce_sync_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce' );
            return;
        }
        
        $last_sync = get_option( 'messos_woocommerce_last_product_sync' );
        $next_sync = wp_next_scheduled( 'messos_woocommerce_scheduled_product_sync' );
        
        // Count products
        $total_products = wp_count_posts( 'product' )->publish;
        
        // Count products modified since last sync
        $modified_count = 0;
        if ( $last_sync ) {
            $args = [
                'post_type' => 'product',
                'post_status' => 'publish',
                'date_query' => [
                    [
                        'column' => 'post_modified',
                        'after' => $last_sync,
                    ]
                ],
                'posts_per_page' => -1,
                'fields' => 'ids',
            ];
            $query = new WP_Query( $args );
            $modified_count = $query->found_posts;
        }
        
        wp_send_json([
            'last_sync' => $last_sync ? gmdate( 'Y-m-d H:i:s', strtotime( $last_sync ) ) : 'Never',
            'next_sync' => $next_sync ? gmdate( 'Y-m-d H:i:s', $next_sync ) : 'Not scheduled',
            'total_products' => $total_products,
            'modified_since_sync' => $modified_count,
            'sync_interval' => $this->get_sync_interval(),
        ]);
    }
    
    /**
     * Get current sync interval
     */
    private function get_sync_interval() {
        $schedules = wp_get_schedules();
        $event = wp_get_scheduled_event( 'messos_woocommerce_scheduled_product_sync' );
        
        if ( $event && isset( $schedules[ $event->schedule ] ) ) {
            return $schedules[ $event->schedule ]['display'];
        }
        
        return 'Not scheduled';
    }
    
    /**
     * Update sync interval
     */
    public function update_sync_interval( $interval ) {
        // Clear existing schedule
        wp_clear_scheduled_hook( 'messos_woocommerce_scheduled_product_sync' );
        
        // Set new schedule
        if ( in_array( $interval, [ 'twohours', 'sixhours', 'daily' ] ) ) {
            wp_schedule_event( time(), $interval, 'messos_woocommerce_scheduled_product_sync' );
        }
    }
}