<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_OAuth_Handler {
    
    const OAUTH_OPTION_NAME = 'messos_woocommerce_oauth_credentials';
    const OAUTH_STATE_OPTION = 'messos_woocommerce_oauth_state';
    const OAUTH_CALLBACK_ROUTE = 'messos-woocommerce-oauth-callback';
    
    private $client_id;
    private $client_secret;
    private $oauth_base_url;
    
    /**
     * Constructor - initializes OAuth credentials and registers hooks
     * Sets up OAuth URLs, client credentials, and WordPress action hooks
     * Called by: WordPress when class is instantiated
     */
    public function __construct() {
        // Initialize OAuth URLs
        $this->oauth_base_url = MESSENGEROS_WOOCOMMERCE_PLATFORM_URL;
        
        // Initialize client credentials
        $this->client_id = get_option( 'messos_woocommerce_oauth_client_id', '' );
        $this->client_secret = get_option( 'messos_woocommerce_oauth_client_secret', '' );
        
        // Register OAuth callback endpoint
        add_action( 'init', [ $this, 'register_oauth_callback' ] );
        add_action( 'template_redirect', [ $this, 'handle_oauth_callback' ] );
        add_action( 'wp_ajax_messos_woocommerce_disconnect_oauth', [ $this, 'handle_disconnect' ] );
        add_action( 'wp_ajax_messos_woocommerce_check_oauth_status', [ $this, 'check_oauth_status' ] );
        
        // Schedule token refresh check
        add_action( 'messos_woocommerce_check_token_validity', [ $this, 'check_and_refresh_token' ] );
        if ( ! wp_next_scheduled( 'messos_woocommerce_check_token_validity' ) ) {
            wp_schedule_event( time(), 'hourly', 'messos_woocommerce_check_token_validity' );
        }
    }
    
    /**
     * Register OAuth callback endpoint with WordPress rewrite rules
     * Creates URL endpoint for handling OAuth authorization code callbacks
     * Called by: WordPress init hook
     */
    public function register_oauth_callback() {
        add_rewrite_rule( '^' . self::OAUTH_CALLBACK_ROUTE . '/?', 'index.php?' . self::OAUTH_CALLBACK_ROUTE . '=1', 'top' );
        add_rewrite_tag( '%' . self::OAUTH_CALLBACK_ROUTE . '%', '([^&]+)' );
    }
    
    /**
     * Get OAuth redirect URI for authorization callbacks
     * Returns the full URL where MessengerOS should redirect after authorization
     * Called by: OAuth authorization URL generation methods
     */
    public function get_redirect_uri() {
        return home_url( '/' . self::OAUTH_CALLBACK_ROUTE );
    }
    
    /**
     * Check if OAuth is connected by verifying stored access token exists
     * Returns connection status for determining plugin functionality availability
     * Called by: Plugin initialization, admin pages, and feature availability checks
     */
    public function is_connected() {
        $credentials = get_option( self::OAUTH_OPTION_NAME );
        return ! empty( $credentials['access_token'] );
    }
    
    /**
     * Get stored OAuth credentials with automatic token decryption
     * Retrieves and decrypts access and refresh tokens for API usage
     * Called by: API clients when making authenticated requests to MessengerOS
     */
    public function get_credentials() {
        $credentials = get_option( self::OAUTH_OPTION_NAME, [] );
        
        if ( ! empty( $credentials['access_token'] ) ) {
            $credentials['access_token'] = $this->decrypt_token( $credentials['access_token'] );
        }
        
        if ( ! empty( $credentials['refresh_token'] ) ) {
            $credentials['refresh_token'] = $this->decrypt_token( $credentials['refresh_token'] );
        }
        
        return $credentials;
    }
    
    /**
     * Generate OAuth authorization URL with store metadata and security state
     * Creates secure authorization URL for connecting store to MessengerOS platform
     * Called by: Admin connect buttons and OAuth initiation flows
     */
    public function get_authorization_url( $action = 'connect' ) {
         error_log( 'MessengerOS: get_authorization_url() called with action: ' . $action );
        
        $client_id = $this->get_client_id();

        if ( empty( $client_id ) ) {
             error_log( 'MessengerOS: Cannot generate authorization URL - no client ID configured' );
            return '';
        }
        
        // error_log( 'MessengerOS: Generating authorization URL with client ID: ' . substr( $client_id, 0, 10 ) . '...' );
        $state = wp_generate_password( 32, false );
        update_option( self::OAUTH_STATE_OPTION, $state );
        
        // Include store metadata in the state
        $state_data = [
            'nonce' => $state,
            'store_uuid' => MessengerOS_WooCommerce_Store_Identifier::get_store_uuid(),
            'store_url' => get_site_url(),
            'store_name' => get_bloginfo( 'name' ),
            'fingerprint' => MessengerOS_WooCommerce_Store_Identifier::get_store_fingerprint(),
        ];
        
        $encoded_state = base64_encode( json_encode( $state_data ) );
        update_option( self::OAUTH_STATE_OPTION, $state ); // Still save just the nonce for validation
        
        $params = [
            'response_type' => 'code',
            'client_id' => $this->get_client_id(),
            'redirect_uri' => $this->get_redirect_uri(),
            'scope' => 'read_products write_products read_customers write_customers read_orders write_orders webhooks',
            'state' => $encoded_state,
        ];
        
        if ( $action === 'register' ) {
            return $this->oauth_base_url . '/register/show?' . http_build_query( $params );
        }
        
        return $this->oauth_base_url . '/oauth/authorize?' . http_build_query( $params );
    }
    
    /**
     * Handle OAuth callback
     */
    public function handle_oauth_callback() {
        if ( ! get_query_var( self::OAUTH_CALLBACK_ROUTE ) ) {
            return;
        }
        
        // Verify state parameter
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external provider, not form submission
        $state = sanitize_text_field( wp_unslash($_GET['state'] ?? '' ));
        $saved_state = get_option( self::OAUTH_STATE_OPTION );
        
        // Decode state to extract nonce
        $decoded_state = base64_decode( $state );
        $state_data = json_decode( $decoded_state, true );
        
        if ( empty( $state_data['nonce'] ) || $state_data['nonce'] !== $saved_state ) {
            wp_die( 'Invalid OAuth state' );
        }
        
        // Check for OAuth errors (user denied, etc.)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external provider
        $error = sanitize_text_field( wp_unslash($_GET['error'] ?? '' ));
        if ( ! empty( $error ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external provider
            $error_description = sanitize_text_field( wp_unslash($_GET['error_description'] ?? '' ));
            
            // Log the error
            // error_log( 'MessengerOS: OAuth Error: ' . $error . ' - ' . $error_description );
            
            // Redirect to admin settings page with error message
            wp_redirect( admin_url( 'admin.php?page=messos_woocommerce_settings&oauth=error&error=' . urlencode( $error ) ) );
            exit;
        }
        
        // Handle authorization code
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- OAuth callback from external provider
        $code = sanitize_text_field( wp_unslash($_GET['code'] ?? '' ));
        if ( empty( $code ) ) {
            wp_redirect( admin_url( 'admin.php?page=messos_woocommerce_settings&oauth=error&error=no_code' ) );
            exit;
        }
        
        // Exchange code for access token
        $token_response = $this->exchange_code_for_token( $code );
        
        if ( is_wp_error( $token_response ) ) {
            // Log detailed error information
//            // error_log( 'OAuth Token Exchange Failed: ' . $token_response->get_error_code() . ' - ' . $token_response->get_error_message() );
//            // error_log( 'Client ID: ' . $this->get_client_id() );
//            // error_log( 'Redirect URI: ' . $this->get_redirect_uri() );
            
            wp_die( 'Failed to obtain access token: ' . esc_html($token_response->get_error_message()) . '<br><br>Please check the error logs for more details.' );
        }
        
        // error_log( 'MessengerOS: OAuth token exchange successful' );
        
        // Store credentials
        $this->store_credentials( $token_response );
        
        // Clean up state
        delete_option( self::OAUTH_STATE_OPTION );
        
        // error_log( 'MessengerOS: OAuth callback completed successfully, redirecting to settings page' );
        
        // Redirect to admin settings page
        wp_redirect( admin_url( 'admin.php?page=messos_woocommerce_settings&oauth=connected' ) );
        exit;
    }
    
    /**
     * Exchange authorization code for access token
     */
    private function exchange_code_for_token( $code ) {
        $client_id = $this->get_client_id();
        $client_secret = $this->get_client_secret();
        $redirect_uri = $this->get_redirect_uri();
        
        // Get store UUID to pass with token request
        $store_uuid = MessengerOS_WooCommerce_Store_Identifier::get_store_uuid();
        
        // Log request details for debugging
//        // error_log( 'OAuth Token Request Details:' );
//        // error_log( 'Endpoint: ' . $this->oauth_base_url . '/oauth/token' );
//        // error_log( 'Client ID: ' . $client_id );
//        // error_log( 'Client Secret: ' . substr( $client_secret, 0, 10 ) . '...' );
//        // error_log( 'Redirect URI: ' . $redirect_uri );
//        // error_log( 'Code: ' . substr( $code, 0, 20 ) . '...' );
        
        $response = wp_remote_post( $this->oauth_base_url . '/oauth/token', [
            'body' => [
                'grant_type' => 'authorization_code',
                'code' => $code,
                'redirect_uri' => $redirect_uri,
                'client_id' => $client_id,
                'client_secret' => $client_secret,
                // Add store metadata
                'store_uuid' => $store_uuid,
                'store_name' => get_bloginfo( 'name' ),
                'store_url' => get_site_url(),
                'store_fingerprint' => json_encode( MessengerOS_WooCommerce_Store_Identifier::get_store_fingerprint() ),
            ],
            'timeout' => 30,
        ] );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        // Log the response for debugging
//        // error_log( 'OAuth Token Response Code: ' . $response_code );
//        // error_log( 'OAuth Token Response Body: ' . $body );
        
        if ( $response_code !== 200 ) {
            $error_message = isset( $data['error_description'] ) ? $data['error_description'] : 'HTTP ' . $response_code;
            return new WP_Error( 'token_error', $error_message );
        }
        
        if ( empty( $data['access_token'] ) ) {
            return new WP_Error( 'invalid_response', 'Invalid token response: ' . $body );
        }
        
        return $data;
    }
    
    /**
     * Store OAuth credentials
     */
    private function store_credentials( $token_data ) {
        $credentials = [
            'access_token' => $this->encrypt_token( $token_data['access_token'] ),
            'refresh_token' => ! empty( $token_data['refresh_token'] ) ? $this->encrypt_token( $token_data['refresh_token'] ) : '',
            'expires_at' => time() + ( $token_data['expires_in'] ?? 3600 ),
            'scope' => $token_data['scope'] ?? '',
            'connected_at' => time(),
        ];
        
        update_option( self::OAUTH_OPTION_NAME, $credentials );
        
        // Store service credentials if present
        if ( ! empty( $token_data['service_credentials'] ) ) {
            $this->store_service_credentials( $token_data['service_credentials'] );
        }
        
        // Store WooCommerce API key if present
        if ( ! empty( $token_data['woocommerce_api_key'] ) ) {
            $api_settings = get_option( 'messos_woocommerce_api_settings', [] );
            $api_settings['api_key'] = $token_data['woocommerce_api_key'];
            update_option( 'messos_woocommerce_api_settings', $api_settings );
            
//            // error_log( 'MessengerOS: Stored WooCommerce API key from OAuth response: ' . $token_data['woocommerce_api_key'] );
        }
        
        // error_log( 'MessengerOS: OAuth credentials stored successfully' );
        
        // Trigger OAuth connected action
        do_action( 'messengeros_woocommerce_oauth_connected' );
        
        // error_log( 'MessengerOS: Triggered messengeros_oauth_connected action' );
    }
    
    /**
     * Store service credentials for Data Warehouse and Events
     */
    private function store_service_credentials( $service_credentials ) {
        // Store Data Warehouse credentials
        if ( ! empty( $service_credentials['data_warehouse'] ) ) {
            $dw_creds = $service_credentials['data_warehouse'];
            update_option( 'messengeros_woocommerce_dw_api_key', $this->encrypt_token( $dw_creds['api_key'] ) );
            update_option( 'messengeros_woocommerce_dw_api_secret', $this->encrypt_token( $dw_creds['api_secret'] ) );
            
            if ( ! empty( $dw_creds['webhook_secret'] ) ) {
                update_option( 'messengeros_woocommerce_dw_webhook_secret', $this->encrypt_token( $dw_creds['webhook_secret'] ) );
            }
            
            if ( ! empty( $dw_creds['endpoints'] ) ) {
                update_option( 'messengeros_woocommerce_dw_endpoints', $dw_creds['endpoints'] );
            }
        }
        
        // Store Events service credentials
        if ( ! empty( $service_credentials['events'] ) ) {
            $events_creds = $service_credentials['events'];
            
            if ( ! empty( $events_creds['api_key'] ) ) {
                update_option( 'messengeros_woocommerce_events_api_key', $events_creds['api_key'] );
                update_option( 'messengeros_woocommerce_events_api_secret', $this->encrypt_token( $events_creds['api_secret'] ) );
            }
            
            if ( ! empty( $events_creds['endpoints'] ) ) {
                update_option( 'messengeros_woocommerce_events_endpoints', $events_creds['endpoints'] );
            }
        }
        
        // Store common store ID
        if ( ! empty( $service_credentials['store_id'] ) ) {
            update_option( 'messengeros_woocommerce_store_id', $service_credentials['store_id'] );
        }
        
//        // error_log( 'MessengerOS: Service credentials stored successfully' );
    }
    
    /**
     * Encrypt sensitive token data
     */
    private function encrypt_token( $token ) {
        if ( empty( $token ) ) {
            return '';
        }
        
        $salt = wp_salt( 'auth' );
        $key = substr( hash( 'sha256', $salt ), 0, 32 );
        $iv = openssl_random_pseudo_bytes( 16 );
        
        $encrypted = openssl_encrypt( $token, 'AES-256-CBC', $key, 0, $iv );
        
        return base64_encode( $iv . $encrypted );
    }
    
    /**
     * Decrypt sensitive token data
     */
    public function decrypt_token( $encrypted_token ) {
        if ( empty( $encrypted_token ) ) {
            return '';
        }
        
        $salt = wp_salt( 'auth' );
        $key = substr( hash( 'sha256', $salt ), 0, 32 );
        
        $data = base64_decode( $encrypted_token );
        $iv = substr( $data, 0, 16 );
        $encrypted = substr( $data, 16 );
        
        return openssl_decrypt( $encrypted, 'AES-256-CBC', $key, 0, $iv );
    }
    
    /**
     * Refresh access token
     */
    public function refresh_token() {
        $raw_credentials = get_option( self::OAUTH_OPTION_NAME, [] );
        
        if ( empty( $raw_credentials['refresh_token'] ) ) {
            return new WP_Error( 'no_refresh_token', 'No refresh token available' );
        }
        
        $refresh_token = $this->decrypt_token( $raw_credentials['refresh_token'] );
        
        $response = wp_remote_post( $this->oauth_base_url . '/oauth/token', [
            'body' => [
                'grant_type' => 'refresh_token',
                'refresh_token' => $refresh_token,
                'client_id' => $this->get_client_id(),
                'client_secret' => $this->get_client_secret(),
            ],
            'timeout' => 30,
        ] );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( empty( $data['access_token'] ) ) {
            return new WP_Error( 'invalid_response', 'Failed to refresh token' );
        }
        
        $this->store_credentials( $data );
        return true;
    }
    
    /**
     * Check and refresh token if needed
     */
    public function check_and_refresh_token() {
        $credentials = $this->get_credentials();
        
        if ( empty( $credentials['access_token'] ) ) {
            return;
        }
        
        // Check if token is about to expire (within 1 hour)
        if ( $credentials['expires_at'] - time() < 3600 ) {
            $this->refresh_token();
        }
    }
    
    /**
     * Disconnect OAuth
     */
    public function disconnect() {
        delete_option( self::OAUTH_OPTION_NAME );
        delete_option( self::OAUTH_STATE_OPTION );
        
        // Clear service credentials
        delete_option( 'messengeros_woocommerce_dw_api_key' );
        delete_option( 'messengeros_woocommerce_dw_api_secret' );
        delete_option( 'messengeros_woocommerce_dw_webhook_secret' );
        delete_option( 'messengeros_woocommerce_dw_endpoints' );
        delete_option( 'messengeros_woocommerce_events_endpoints' );
        delete_option( 'messengeros_woocommerce_store_id' );
        
        // Clear last sync time
        delete_option( 'messos_woocommerce_last_product_sync' );
        
        // Clear API key from settings
        $api_settings = get_option( 'messos_woocommerce_api_settings', [] );
        if ( isset( $api_settings['api_key'] ) ) {
            unset( $api_settings['api_key'] );
            update_option( 'messos_woocommerce_api_settings', $api_settings );
        }
        
        // If auto-registered, also clear the credentials
        if ( get_option( 'messos_woocommerce_oauth_auto_registered' ) ) {
            delete_option( 'messos_woocommerce_oauth_client_id' );
            delete_option( 'messos_woocommerce_oauth_client_secret' );
            delete_option( 'messos_woocommerce_oauth_auto_registered' );
        }
    }
    
    /**
     * Handle disconnect AJAX request
     */
    public function handle_disconnect() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Unauthorized' );
        }
        
        // Verify nonce
        $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'messos_woocommerce_disconnect_oauth' ) ) {
            wp_send_json_error( 'Invalid nonce' );
            return;
        }
        
        $this->disconnect();
        wp_send_json_success();
    }
    
    /**
     * Check OAuth status AJAX handler
     */
    public function check_oauth_status() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Unauthorized' );
        }
        
        $is_connected = $this->is_connected();
        $credentials = $this->get_credentials();
        
        wp_send_json_success( [
            'connected' => $is_connected,
            'expires_at' => $credentials['expires_at'] ?? null,
        ] );
    }
    
    /**
     * Get OAuth client ID
     */
    private function get_client_id() {
        // Return cached value or get from options
        if ( empty( $this->client_id ) ) {
            $this->client_id = get_option( 'messos_woocommerce_oauth_client_id', '' );
            // Try auto-registration if client ID is empty
            if ( empty( $this->client_id ) ) {
                 error_log( 'MessengerOS: Client ID empty, attempting auto-registration' );
                $registration_result = MessengerOS_WooCommerce_OAuth_Auto_Register::maybe_auto_register();
                 error_log( 'MessengerOS: Auto-registration result: ' . ( $registration_result ? 'success' : 'failed' ) );
                $this->client_id = get_option( 'messos_woocommerce_oauth_client_id', '' );
                 error_log( 'MessengerOS: Client ID after registration: ' . $this->client_id );
            }
        }
        return $this->client_id;
    }
    
    /**
     * Get OAuth client secret
     */
    private function get_client_secret() {
        // Return cached value or get from options
        if ( empty( $this->client_secret ) ) {
            $this->client_secret = get_option( 'messos_woocommerce_oauth_client_secret', '' );
        }
        return $this->client_secret;
    }
    
    /**
     * Make authenticated API request
     */
    public function make_api_request( $endpoint, $method = 'GET', $body = null ) {
        $credentials = $this->get_credentials();
        
        if ( empty( $credentials['access_token'] ) ) {
            return new WP_Error( 'not_connected', 'OAuth not connected' );
        }
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ],
            'timeout' => 30,
        ];
        
        if ( $body !== null ) {
            $args['body'] = json_encode( $body );
        }
        
        $response = wp_remote_request( $this->oauth_base_url . '/api/' . $endpoint, $args );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $body_str = wp_remote_retrieve_body( $response );
        $body_data = json_decode( $body_str, true );
        
        // Handle various disconnection scenarios
        if ( $response_code === 401 ) {
            // Check if it's a password change or account issue
            if ( isset( $body_data['error'] ) ) {
                if ( strpos( $body_data['error'], 'password_changed' ) !== false ||
                     strpos( $body_data['error'], 'account_suspended' ) !== false ||
                     strpos( $body_data['error'], 'invalid_token' ) !== false ) {
                    // Force disconnect on critical auth errors
                    $this->disconnect();
                    return new WP_Error( 'auth_error', 'Authentication failed. Please reconnect your account.' );
                }
            }
            
            // Try to refresh token
            $refresh_result = $this->refresh_token();
            if ( ! is_wp_error( $refresh_result ) ) {
                // Retry request with new token
                return $this->make_api_request( $endpoint, $method, $body );
            } else {
                // If refresh fails, disconnect
                $this->disconnect();
                return new WP_Error( 'token_refresh_failed', 'Failed to refresh token. Please reconnect your account.' );
            }
        }
        
        // Handle rate limiting
        if ( $response_code === 429 ) {
            return new WP_Error( 'rate_limited', 'API rate limit exceeded. Please try again later.' );
        }
        
        return $body_data;
    }
    
    /**
     * Check account status
     */
    public function check_account_status() {
        if ( ! $this->is_connected() ) {
            return false;
        }
        
        $response = $this->make_api_request( 'account/status', 'GET' );
        
        if ( is_wp_error( $response ) ) {
            return false;
        }
        
        // Check for inactivity or other account issues
        if ( isset( $response['status'] ) && $response['status'] === 'inactive' ) {
            $this->disconnect();
            return false;
        }

        return true;
    }

    /**
     * Validate Bearer token for REST API authentication
     *
     * @param string $token The Bearer token to validate.
     * @return bool True if token is valid, false otherwise.
     */
    public function validate_bearer_token( $token ) {
        if ( empty( $token ) ) {
            error_log( 'Token validation failed: token is empty' );
            return false;
        }

        $credentials = $this->get_credentials();

        if ( empty( $credentials['access_token'] ) ) {
            error_log( 'Token validation failed: no stored access token' );
            return false;
        }

        error_log( 'Validating token. Provided token (first 20 chars): ' . substr( $token, 0, 20 ) );
        error_log( 'Stored token (first 20 chars): ' . substr( $credentials['access_token'], 0, 20 ) );

        // First, try exact match with stored token
        if ( hash_equals( $credentials['access_token'], $token ) ) {
            error_log( 'Token validation: exact match success' );
            return true;
        }

        error_log( 'Token validation: exact match failed, trying OAuth server verification' );

        // Fallback: Verify token with OAuth server
        $result = $this->verify_token_with_server( $token );
        error_log( 'OAuth server verification result: ' . ( $result ? 'success' : 'failed' ) );
        return $result;
    }

    /**
     * Verify token with OAuth server
     *
     * @param string $token The token to verify.
     * @return bool True if valid, false otherwise.
     */
    private function verify_token_with_server( $token ) {
        $verify_url = $this->oauth_base_url . '/public/verify-token';
        error_log( 'Verifying token with server: ' . $verify_url );

        $response = wp_remote_get( $verify_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
            ],
            'timeout' => 5,
        ] );

        if ( is_wp_error( $response ) ) {
            error_log( 'Token verification error: ' . $response->get_error_message() );
            return false;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        error_log( 'Token verification response code: ' . $response_code );
        error_log( 'Token verification response body: ' . $response_body );

        return $response_code === 200;
    }
}