<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_OAuth_Auto_Register {
    
    /**
     * Register OAuth client with MessengerOS
     */
    private static function register_oauth_client() {
        $base_url = MESSENGEROS_WOOCOMMERCE_PLATFORM_URL;

        $store_url = home_url();
        $store_name = get_bloginfo( 'name' ) ?: wp_parse_url( $store_url, PHP_URL_HOST );

        // error_log( 'MessengerOS: Starting OAuth client registration for ' . $store_url );

        // Prepare request data
        $body = json_encode([
            'store_url' => $store_url,
            'store_name' => $store_name,
        ]);

        // Generate HMAC signature
        $timestamp = time();
        $nonce = wp_generate_password( 32, false );
        $payload = $timestamp . '.' . $nonce . '.' . $body;
        $secret = self::get_registration_secret();
        $signature = hash_hmac( 'sha256', $payload, $secret );

//         error_log( 'MessengerOS: Registration URL: ' . $base_url . '/oauth/register-client' );
//         error_log( 'MessengerOS: Secret configured: ' . ( !empty( $secret ) ? 'yes' : 'no' ) );

        $response = wp_remote_post( $base_url . '/oauth/register-client', [
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Registration-Signature' => $signature,
                'X-Registration-Timestamp' => $timestamp,
                'X-Registration-Nonce' => $nonce,
            ],
            'body' => $body,
            'timeout' => 30,
        ]);

        if ( is_wp_error( $response ) ) {
//             error_log( 'MessengerOS: OAuth auto-registration failed: ' . $response->get_error_message() );
            return false;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

//         error_log( 'MessengerOS: Registration response code: ' . $response_code );
//         error_log( 'MessengerOS: Registration response body: ' . $response_body );

        if ( $response_code !== 200 || empty( $data['client_id'] ) || empty( $data['client_secret'] ) ) {
            // error_log( 'MessengerOS: OAuth auto-registration failed. Response: ' . $response_body );
            return false;
        }

        // Save credentials
        update_option( 'messos_woocommerce_oauth_client_id', $data['client_id'] );
        update_option( 'messos_woocommerce_oauth_client_secret', $data['client_secret'] );
        update_option( 'messos_woocommerce_oauth_auto_registered', true );

        // Log success
//         error_log( 'OAuth client auto-registered successfully for ' . $store_name );

        return true;
    }

    /**
     * Get registration secret from secure location
     */
    private static function get_registration_secret() {
        // Check if defined in wp-config.php (recommended)
        if ( defined( 'MESSENGEROS_WOOCOMMERCE_REGISTRATION_SECRET' ) ) {
            return MESSENGEROS_WOOCOMMERCE_REGISTRATION_SECRET;
        }
        
        // Return empty string if not configured
        return '';
    }
    
    /**
     * Public method to trigger OAuth client auto-registration
     * Attempts to register OAuth client credentials with MessengerOS platform
     * Called by: OAuth handler when client credentials are missing
     */
    public static function maybe_auto_register() {
         error_log( 'MessengerOS: maybe_auto_register() called' );
        
        // Check if credentials are already available
        if ( get_option( 'messos_woocommerce_oauth_client_id' ) && get_option( 'messos_woocommerce_oauth_client_secret' ) ) {
            error_log( 'MessengerOS: OAuth credentials already exist' );
            return true;
        }
        
        // Check if registration secret is configured
        $secret = self::get_registration_secret();
        if ( empty( $secret ) || $secret === 'YOUR_REGISTRATION_SECRET_HERE' ) {
            error_log( 'MessengerOS: Registration secret not configured properly' );
            return false;
        }
        
        // Basic rate limiting - only try once per hour
        $last_attempt = get_option( 'messos_woocommerce_last_registration_attempt', 0 );
        if ( ( time() - $last_attempt ) < 3600 ) {
            error_log( 'MessengerOS: Rate limiting active, skipping registration' );
//            return false;
        }
        
         error_log( 'MessengerOS: Proceeding with auto-registration' );
        update_option( 'messos_woocommerce_last_registration_attempt', time() );

        // Attempt auto-registration
        $result = self::register_oauth_client();
        
        if ( $result ) {
            // Clear rate limiting on success
            delete_option( 'messos_woocommerce_last_registration_attempt' );
             error_log( 'MessengerOS: Auto-registration successful' );
        } else {
             error_log( 'MessengerOS: Auto-registration failed' );
        }
        
        return $result;
    }
}