<?php
/**
 * Handle initial product sync after OAuth connection
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_Initial_Sync {
    
    public function __construct() {
        // Hook for checking initial sync
        add_action( 'messengeros_woocommerce_check_initial_sync', [ $this, 'check_and_perform_initial_sync' ] );
        add_action( 'messengeros_woocommerce_perform_initial_sync', [ $this, 'perform_initial_sync' ] );
        
        // Hook after OAuth connection is established
        add_action( 'messengeros_woocommerce_oauth_connected', [ $this, 'schedule_initial_sync' ] );
        
        // Admin notice for sync status
        add_action( 'admin_notices', [ $this, 'show_sync_notice' ] );
        
        // AJAX endpoint for dismissing notice
        add_action( 'wp_ajax_messengeros_woocommerce_dismiss_sync_notice', [ $this, 'dismiss_sync_notice' ] );
    }
    
    /**
     * Schedule initial sync after OAuth connection
     */
    public function schedule_initial_sync() {
        // error_log( 'MessengerOS: schedule_initial_sync() called after OAuth connection' );
        
        // Check if automatic sync is enabled
        if ( ! apply_filters( 'messengeros_woocommerce_enable_auto_sync', true ) ) {
            // error_log( 'MessengerOS: Automatic sync is disabled via filter' );
            return;
        }
        
        // Check if we need initial sync
        if ( get_option( 'messengeros_woocommerce_needs_initial_sync' ) ) {
            // error_log( 'MessengerOS: Initial sync needed flag is set' );
            // Schedule sync in 5 seconds to allow OAuth process to complete
            if ( ! wp_next_scheduled( 'messengeros_woocommerce_perform_initial_sync' ) ) {
                wp_schedule_single_event( time() + 5, 'messengeros_woocommerce_perform_initial_sync' );
                // error_log( 'MessengerOS: Scheduled initial sync to run in 5 seconds' );
            } else {
                // error_log( 'MessengerOS: Initial sync already scheduled' );
            }
        } else {
            // error_log( 'MessengerOS: Initial sync not needed (flag not set)' );
        }
    }
    
    /**
     * Check if connected and perform initial sync
     */
    public function check_and_perform_initial_sync() {
        // error_log( 'MessengerOS: check_and_perform_initial_sync() called' );
        
        // Only run if we need initial sync
        if ( ! get_option( 'messengeros_woocommerce_needs_initial_sync' ) ) {
            // error_log( 'MessengerOS: Initial sync not needed, exiting' );
            return;
        }
        
        // Check if OAuth is connected
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ( ! $oauth_handler->is_connected() ) {
            // error_log( 'MessengerOS: Not connected to MessengerOS yet, rescheduling check for 5 minutes' );
            // Not connected yet, reschedule check
            wp_schedule_single_event( time() + 300, 'messengeros_woocommerce_check_initial_sync' ); // Check again in 5 minutes
            return;
        }
        
        // error_log( 'MessengerOS: Connected to MessengerOS, performing initial sync now' );
        // Perform initial sync
        $this->perform_initial_sync();
    }
    
    /**
     * Perform the initial product sync
     */
    public function perform_initial_sync() {
        // error_log( 'MessengerOS: Starting initial product sync' );
        
        // Set sync status
        update_option( 'messengeros_woocommerce_initial_sync_status', 'in_progress' );
        update_option( 'messengeros_woocommerce_initial_sync_start', current_time( 'mysql' ) );
        
        // error_log( 'MessengerOS: Initial sync started at ' . current_time( 'mysql' ) );
        
        // Initialize product sync
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'inc/product-sync.php';
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        $product_sync = new MessengerOS_WooCommerce_Product_Sync( $oauth_handler );
        
        // Get total product count
        $total_products = wp_count_posts( 'product' )->publish;
        update_option( 'messengeros_woocommerce_initial_sync_total', $total_products );
        
        // error_log( 'MessengerOS: Initial sync will process ' . $total_products . ' products' );
        
        // Sync in batches
        $batch_size = apply_filters( 'messengeros_woocommerce_sync_batch_size', 50 );
        $pages = ceil( $total_products / $batch_size );
        $synced = 0;
        $failed = 0;
        $errors = [];
        
        for ( $page = 1; $page <= $pages; $page++ ) {
            // error_log( 'MessengerOS: Initial sync processing page ' . $page . ' of ' . $pages );
            
            $result = $product_sync->sync_products( $page );
            
            if ( $result['success'] ) {
                $synced += $result['synced'];
                $failed += $result['failed'];
                if ( ! empty( $result['errors'] ) ) {
                    $errors = array_merge( $errors, $result['errors'] );
                }
                
                // error_log( 'MessengerOS: Initial sync page ' . $page . ' - synced: ' . $result['synced'] . ', failed: ' . $result['failed'] );
                
                // Update progress
                update_option( 'messengeros_woocommerce_initial_sync_progress', [
                    'synced' => $synced,
                    'failed' => $failed,
                    'current_page' => $page,
                    'total_pages' => $pages
                ] );
            } else {
                // Sync failed, log error
                // error_log( 'MessengerOS: Initial sync page ' . $page . ' failed completely' );
                $errors[] = 'Batch ' . $page . ' failed';
                $failed += $batch_size;
            }
            
            // Small delay between batches to avoid overwhelming the server
            if ( $page < $pages ) {
                // error_log( 'MessengerOS: Pausing 1 second before next batch' );
                sleep( 1 );
            }
        }
        
        // Mark sync as complete
        update_option( 'messengeros_woocommerce_initial_sync_status', 'completed' );
        update_option( 'messengeros_woocommerce_initial_sync_end', current_time( 'mysql' ) );
        update_option( 'messengeros_woocommerce_initial_sync_results', [
            'synced' => $synced,
            'failed' => $failed,
            'total' => $total_products,
            'errors' => array_slice( $errors, 0, 10 ) // Keep only first 10 errors
        ] );
        
        // error_log( 'MessengerOS: Initial sync completed at ' . current_time( 'mysql' ) );
        // error_log( 'MessengerOS: Initial sync results - Total: ' . $total_products . ', Synced: ' . $synced . ', Failed: ' . $failed );
        
        if ( ! empty( $errors ) ) {
            // error_log( 'MessengerOS: Initial sync errors: ' . implode( ', ', array_slice( $errors, 0, 5 ) ) );
        }
        
        // Remove the needs sync flag
        delete_option( 'messengeros_woocommerce_needs_initial_sync' );
        
        // Set last sync time
        update_option( 'messos_woocommerce_last_product_sync', current_time( 'mysql' ) );
        
        // error_log( 'MessengerOS: Initial sync cleanup complete, regular sync schedule will take over' );
    }
    
    /**
     * Show admin notice about sync status
     */
    public function show_sync_notice() {
        // Only show on WooCommerce or MessengerOS pages
        $screen = get_current_screen();
        if ( ! $screen || ( strpos( $screen->id, 'woocommerce' ) === false && strpos( $screen->id, 'messengeros' ) === false ) ) {
            return;
        }
        
        // Check if notice was dismissed
        if ( get_option( 'messengeros_woocommerce_sync_notice_dismissed' ) ) {
            return;
        }
        
        $sync_status = get_option( 'messengeros_woocommerce_initial_sync_status' );
        
        if ( $sync_status === 'in_progress' ) {
            $progress = get_option( 'messengeros_woocommerce_initial_sync_progress', [] );
            ?>
            <div class="notice notice-info">
                <p>
                    <strong>MessengerOS:</strong> Initial product sync in progress... 
                    <?php if ( ! empty( $progress ) ) : ?>
                        (<?php echo esc_html( $progress['synced'] ); ?> products synced, 
                        page <?php echo esc_html( $progress['current_page'] ); ?> of <?php echo esc_html( $progress['total_pages'] ); ?>)
                    <?php endif; ?>
                </p>
            </div>
            <?php
        } elseif ( $sync_status === 'completed' ) {
            $results = get_option( 'messengeros_woocommerce_initial_sync_results', [] );
            $notice_class = ( $results['failed'] > 0 ) ? 'notice-warning' : 'notice-success';
            ?>
            <div class="notice <?php echo esc_attr( $notice_class ); ?> is-dismissible">
                <p>
                    <strong>MessengerOS:</strong> Initial product sync completed! 
                    <?php echo esc_html( $results['synced'] ); ?> products synced successfully.
                    <?php if ( $results['failed'] > 0 ) : ?>
                        <?php echo esc_html( $results['failed'] ); ?> products failed to sync.
                    <?php endif; ?>
                </p>
                <?php if ( ! empty( $results['errors'] ) ) : ?>
                    <details>
                        <summary>View errors</summary>
                        <ul>
                            <?php foreach ( $results['errors'] as $error ) : ?>
                                <li><?php echo esc_html( $error ); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </details>
                <?php endif; ?>
                <button type="button" class="notice-dismiss" onclick="messengerosDismissSyncNotice();">
                    <span class="screen-reader-text">Dismiss this notice.</span>
                </button>
            </div>
            <?php
            // Add inline script properly using wp_add_inline_script
            $inline_script = sprintf(
                'function messengerosDismissSyncNotice() {
                    jQuery.post(ajaxurl, {
                        action: "messengeros_woocommerce_dismiss_sync_notice",
                        _wpnonce: "%s"
                    });
                }',
                esc_js( wp_create_nonce( 'messengeros_woocommerce_dismiss_sync_notice' ) )
            );
            wp_add_inline_script( 'jquery', $inline_script );
        }
    }
    
    /**
     * AJAX handler for dismissing notice
     */
    public function dismiss_sync_notice() {
        check_ajax_referer( 'messengeros_woocommerce_dismiss_sync_notice' );
        update_option( 'messengeros_woocommerce_sync_notice_dismissed', true );
        wp_die();
    }
}

// Initialize
new MessengerOS_WooCommerce_Initial_Sync();