<?php
/**
 * Store Identifier Handler
 *
 * Manages the unique store UUID for consistent identification
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class MessengerOS_WooCommerce_Store_Identifier {
    
    const UUID_OPTION_NAME = 'messengeros_store_uuid';
    const UUID_META_OPTION_NAME = 'messengeros_store_uuid_meta';
    
    /**
     * Get or generate store UUID
     *
     * @return string
     */
    public static function get_store_uuid() {
        $uuid = get_option( self::UUID_OPTION_NAME );
        
        if ( ! $uuid ) {
            $uuid = self::generate_uuid();
            update_option( self::UUID_OPTION_NAME, $uuid );
            
            // Store metadata about when and where it was generated
            update_option( self::UUID_META_OPTION_NAME, [
                'generated_at' => current_time( 'mysql' ),
                'site_url' => get_site_url(),
                'wp_version' => get_bloginfo( 'version' ),
                'wc_version' => WC()->version,
                'plugin_version' => MESSENGEROS_WOOCOMMERCE_PLUGIN_VERSION
            ]);
        }
        
        return $uuid;
    }
    
    /**
     * Generate a UUID v4
     *
     * @return string
     */
    private static function generate_uuid() {
        if ( function_exists( 'wp_generate_uuid4' ) ) {
            return wp_generate_uuid4();
        }
        
        // Fallback UUID generation
        $data = openssl_random_pseudo_bytes( 16 );
        
        // Set version (4) and variant bits
        $data[6] = chr( ord( $data[6] ) & 0x0f | 0x40 );
        $data[8] = chr( ord( $data[8] ) & 0x3f | 0x80 );
        
        return vsprintf( '%s%s-%s-%s-%s-%s%s%s', str_split( bin2hex( $data ), 4 ) );
    }
    
    /**
     * Get store fingerprint for additional identification
     *
     * @return array
     */
    public static function get_store_fingerprint() {
        global $wpdb;
        
        return [
            'uuid' => self::get_store_uuid(),
            'domain' => wp_parse_url( get_site_url(), PHP_URL_HOST ),
            'path' => wp_parse_url( get_site_url(), PHP_URL_PATH ) ?: '/',
            'admin_email' => get_option( 'admin_email' ),
            'store_name' => get_bloginfo( 'name' ),
            'store_email' => get_option( 'woocommerce_email_from_address' ),
            'timezone' => get_option( 'timezone_string' ) ?: 'UTC',
            'currency' => get_woocommerce_currency(),
            'table_prefix' => $wpdb->prefix,
            'is_multisite' => is_multisite(),
            'blog_id' => get_current_blog_id(),
            'environment' => self::detect_environment()
        ];
    }
    
    /**
     * Detect environment type
     *
     * @return string
     */
    private static function detect_environment() {
        $domain = wp_parse_url( get_site_url(), PHP_URL_HOST );
        
        if ( strpos( $domain, 'localhost' ) !== false || strpos( $domain, '.local' ) !== false ) {
            return 'development';
        }
        
        if ( strpos( $domain, 'dev.' ) === 0 || strpos( $domain, 'develop.' ) === 0 ) {
            return 'development';
        }
        
        if ( strpos( $domain, 'staging.' ) === 0 || strpos( $domain, 'stage.' ) === 0 || strpos( $domain, 'test.' ) === 0 ) {
            return 'staging';
        }
        
        return 'production';
    }
    
    /**
     * Get store metadata for OAuth connection
     *
     * @return array
     */
    public static function get_store_metadata() {
        return [
            'uuid' => self::get_store_uuid(),
            'fingerprint' => self::get_store_fingerprint(),
            'platform' => 'woocommerce',
            'platform_version' => WC()->version,
            'plugin_version' => MESSENGEROS_VERSION,
            'php_version' => PHP_VERSION,
            'wordpress_version' => get_bloginfo( 'version' )
        ];
    }
    
    /**
     * Clear store UUID (use with caution)
     *
     * @return void
     */
    public static function reset_store_uuid() {
        delete_option( self::UUID_OPTION_NAME );
        delete_option( self::UUID_META_OPTION_NAME );
    }
}