<?php
/**
 * MessengerOS Product Tracker
 * 
 * Lightweight product view tracking for WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class MessengerOS_WooCommerce_Product_Tracker {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Only load on frontend
        if ( ! is_admin() ) {
            add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_tracker_script' ] );
        }
    }
    
    /**
     * Enqueue tracking script on product pages
     */
    public function enqueue_tracker_script() {
        // Only load on single product pages
        if ( ! is_product() ) {
            return;
        }
        
        // Check if tracking is enabled and connected
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ( ! $oauth_handler->is_connected() ) {
            return;
        }
        
        // Get events endpoint
        $events_endpoints = get_option( 'messengeros_events_endpoints', [] );
        $events_url = $events_endpoints['public'] ?? '';
        
        if ( empty( $events_url ) ) {
            return;
        }
        
        // Get product data
        global $product;
        if ( ! $product ) {
            return;
        }
        
        // Enqueue the tracker script
        wp_enqueue_script(
            'messengeros-product-tracker',
            plugin_dir_url( dirname( __FILE__ ) ) . 'assets/js/product-tracker.js',
            [],
            '1.0.1',
            true // Load in footer
        );
        
        // Prepare product data
        $product_data = [
            'id' => $product->get_id(),
            'sku' => $product->get_sku(),
            'name' => $product->get_name(),
            'price' => $product->get_price(),
            'currency' => get_woocommerce_currency(),
            'in_stock' => $product->is_in_stock(),
        ];
        
        // Get primary category
        $categories = get_the_terms( $product->get_id(), 'product_cat' );
        if ( $categories && ! is_wp_error( $categories ) ) {
            $product_data['category'] = $categories[0]->name;
            $product_data['category_id'] = $categories[0]->term_id;
        }
        
        // Pass configuration to JavaScript
        wp_localize_script( 'messengeros-product-tracker', 'messengerOSTracker', [
            'eventsUrl' => $events_url,
            'storeId' => get_option( 'messengeros_store_id' ),
            'storeUuid' => MessengerOS_WooCommerce_Store_Identifier::get_store_uuid(),
            'productId' => (string) $product->get_id(),
            'productData' => $product_data,
        ] );
    }
    
    /**
     * Track server-side events (for important actions)
     */
    public static function track_event( $event_type, $product_id, $additional_data = [] ) {
        // Only track if connected
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ( ! $oauth_handler->is_connected() ) {
            return;
        }
        
        $api_client = new MessengerOS_WooCommerce_API_Client();
        
        // Prepare event data
        $event_data = array_merge( [
            'product_id' => $product_id,
            'customer_id' => get_current_user_id(),
            'is_logged_in' => is_user_logged_in(),
        ], $additional_data );
        
        // Send event (non-blocking)
        $api_client->send_event( $event_type, $event_data );
    }
}

// Initialize tracker
new MessengerOS_WooCommerce_Product_Tracker();