<?php
/**
 * Product Event Tracker
 *
 * Tracks WooCommerce product events and sends webhooks to MessengerOS immediately.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Product_Event_Tracker
 */
class MessengerOS_WooCommerce_Product_Event_Tracker {

	/**
	 * Product event formatter instance
	 *
	 * @var MessengerOS_WooCommerce_Product_Event_Formatter
	 */
	private $event_formatter;

	/**
	 * OAuth handler instance
	 *
	 * @var MessengerOS_WooCommerce_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Track recently created products to avoid duplicate events
	 *
	 * @var array
	 */
	private $recently_created = [];

	/**
	 * Constructor
	 *
	 * @param MessengerOS_WooCommerce_OAuth_Handler $oauth_handler OAuth handler instance.
	 */
	public function __construct( $oauth_handler ) {
		error_log( 'MessengerOS: Product Event Tracker initialized' );

		$this->event_formatter = new MessengerOS_WooCommerce_Product_Event_Formatter();
		$this->oauth_handler   = $oauth_handler;

		$this->register_hooks();
		error_log( 'MessengerOS: Product Event Tracker hooks registered' );
	}

	/**
	 * Register WooCommerce hooks
	 */
	private function register_hooks() {
		// Hook into product creation and updates.
		add_action( 'woocommerce_new_product', [ $this, 'handle_product_create' ], 10, 1 );
		add_action( 'woocommerce_update_product', [ $this, 'handle_product_update' ], 10, 1 );

		// Also hook into save_post for products to catch all changes.
		add_action( 'save_post_product', [ $this, 'handle_save_post_product' ], 10, 3 );
	}

	/**
	 * Handle product create event
	 *
	 * @param int $product_id Product ID.
	 */
	public function handle_product_create( $product_id ) {
		error_log( 'MessengerOS: Product create detected - Product ID: ' . $product_id );

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			error_log( 'MessengerOS: Failed to get product object for ID: ' . $product_id );
			return;
		}

		// Mark as recently created to avoid duplicate update event.
		$this->recently_created[ $product_id ] = time();

		$this->send_product_event( 'product_create', $product );
	}

	/**
	 * Handle product update event
	 *
	 * @param int $product_id Product ID.
	 */
	public function handle_product_update( $product_id ) {
		// Skip if this was just created (avoid duplicate events).
		if ( isset( $this->recently_created[ $product_id ] ) ) {
			if ( ( time() - $this->recently_created[ $product_id ] ) < 5 ) {
				error_log( 'MessengerOS: Skipping product_update for recently created product ID: ' . $product_id );
				return;
			}
			unset( $this->recently_created[ $product_id ] );
		}

		error_log( 'MessengerOS: Product update detected - Product ID: ' . $product_id );

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			error_log( 'MessengerOS: Failed to get product object for ID: ' . $product_id );
			return;
		}

		$this->send_product_event( 'product_update', $product );
	}

	/**
	 * Handle save_post_product event
	 *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post    Post object.
	 * @param bool    $update  Whether this is an update.
	 */
	public function handle_save_post_product( $post_id, $post, $update ) {
		// Skip autosaves and revisions.
		if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
			return;
		}

		// Skip if product is being trashed or is in trash.
		if ( 'trash' === $post->post_status ) {
			return;
		}

		// Let WooCommerce hooks handle the event (this is a backup).
		// Only process if WooCommerce hooks didn't fire.
		if ( ! did_action( 'woocommerce_new_product' ) && ! did_action( 'woocommerce_update_product' ) ) {
			error_log( 'MessengerOS: Product save detected via save_post - Product ID: ' . $post_id . ', Update: ' . ( $update ? 'yes' : 'no' ) );

			$product = wc_get_product( $post_id );
			if ( ! $product ) {
				return;
			}

			$event_type = $update ? 'product_update' : 'product_create';
			$this->send_product_event( $event_type, $product );
		}
	}

	/**
	 * Send product event to MessengerOS
	 *
	 * @param string     $event_type Event type ('product_create' or 'product_update').
	 * @param WC_Product $product    WooCommerce product object.
	 */
	private function send_product_event( $event_type, $product ) {
		// Format event data.
		$event_data = $this->event_formatter->format_product_event( $event_type, $product );

		if ( empty( $event_data ) ) {
			error_log( 'MessengerOS: Failed to format product event data' );
			return;
		}

		error_log( 'MessengerOS: Product event type: ' . $event_type . ', Product ID: ' . $product->get_id() );
		error_log( 'MessengerOS: Product event data: ' . wp_json_encode( $event_data ) );

		// Use OAuth handler's make_api_request to send to MessengerOS API.
		$response = $this->oauth_handler->make_api_request( 'webhooks/product', 'POST', $event_data );

		if ( is_wp_error( $response ) ) {
			error_log( 'MessengerOS: Failed to send product event - ' . $response->get_error_message() );
		} else {
			error_log( 'MessengerOS: Product event sent to MessengerOS successfully' );
			error_log( 'MessengerOS: Product event API response: ' . wp_json_encode( $response ) );
		}
	}
}
