<?php
/**
 * Product Event Formatter
 *
 * Formats product data into webhook payload structure for MessengerOS events.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Product_Event_Formatter
 */
class MessengerOS_WooCommerce_Product_Event_Formatter {

	/**
	 * Format product event data for webhook
	 *
	 * @param string     $event_type Event type ('product_create' or 'product_update').
	 * @param WC_Product $product    WooCommerce product object.
	 * @return array Formatted event data.
	 */
	public function format_product_event( $event_type, $product ) {
		if ( ! $product ) {
			return [];
		}

		$payload = [
			'event'      => $event_type,
			'timestamp'  => gmdate( 'c' ),
			'store_id'   => (string) get_option( 'messengeros_woocommerce_store_id', '' ),
			'product_id' => (int) $product->get_id(),
			'product'    => $this->get_product_data( $product ),
		];

		return $payload;
	}

	/**
	 * Get product data
	 *
	 * @param WC_Product $product WooCommerce product object.
	 * @return array Product data.
	 */
	private function get_product_data( $product ) {
		$product_data = [
			'id'                => (int) $product->get_id(),
			'name'              => $product->get_name(),
			'slug'              => $product->get_slug(),
			'permalink'         => get_permalink( $product->get_id() ),
			'type'              => $product->get_type(),
			'status'            => $product->get_status(),
			'featured'          => $product->get_featured(),
			'catalog_visibility' => $product->get_catalog_visibility(),
			'description'       => $product->get_description(),
			'short_description' => $product->get_short_description(),
			'sku'               => $product->get_sku() ?: null,
			'price'             => $product->get_price() ? (float) $product->get_price() : null,
			'regular_price'     => $product->get_regular_price() ? (float) $product->get_regular_price() : null,
			'sale_price'        => $product->get_sale_price() ? (float) $product->get_sale_price() : null,
			'on_sale'           => $product->is_on_sale(),
			'purchasable'       => $product->is_purchasable(),
			'total_sales'       => (int) $product->get_total_sales(),
			'virtual'           => $product->is_virtual(),
			'downloadable'      => $product->is_downloadable(),
			'manage_stock'      => $product->managing_stock(),
			'stock_quantity'    => $product->get_stock_quantity(),
			'stock_status'      => $product->get_stock_status(),
			'backorders'        => $product->get_backorders(),
			'sold_individually' => $product->is_sold_individually(),
			'weight'            => $product->get_weight() ?: null,
			'dimensions'        => [
				'length' => $product->get_length() ?: null,
				'width'  => $product->get_width() ?: null,
				'height' => $product->get_height() ?: null,
			],
			'shipping_class'    => $product->get_shipping_class(),
			'reviews_allowed'   => $product->get_reviews_allowed(),
			'average_rating'    => $product->get_average_rating(),
			'rating_count'      => (int) $product->get_rating_count(),
			'parent_id'         => (int) $product->get_parent_id(),
			'categories'        => $this->get_product_categories( $product ),
			'tags'              => $this->get_product_tags( $product ),
			'images'            => $this->get_product_images( $product ),
			'attributes'        => $this->get_product_attributes( $product ),
			'date_created'      => $product->get_date_created() ? $product->get_date_created()->format( 'c' ) : null,
			'date_modified'     => $product->get_date_modified() ? $product->get_date_modified()->format( 'c' ) : null,
		];

		// Add variation-specific data if this is a variation.
		if ( $product->is_type( 'variation' ) ) {
			$product_data['variation_attributes'] = $product->get_variation_attributes();
		}

		return $product_data;
	}

	/**
	 * Get product categories
	 *
	 * @param WC_Product $product WooCommerce product object.
	 * @return array Product categories.
	 */
	private function get_product_categories( $product ) {
		$category_ids = $product->get_category_ids();
		$categories   = [];

		foreach ( $category_ids as $category_id ) {
			$category = get_term( $category_id, 'product_cat' );
			if ( $category && ! is_wp_error( $category ) ) {
				$categories[] = [
					'id'   => (int) $category->term_id,
					'name' => $category->name,
					'slug' => $category->slug,
				];
			}
		}

		return $categories;
	}

	/**
	 * Get product tags
	 *
	 * @param WC_Product $product WooCommerce product object.
	 * @return array Product tags.
	 */
	private function get_product_tags( $product ) {
		$tag_ids = $product->get_tag_ids();
		$tags    = [];

		foreach ( $tag_ids as $tag_id ) {
			$tag = get_term( $tag_id, 'product_tag' );
			if ( $tag && ! is_wp_error( $tag ) ) {
				$tags[] = [
					'id'   => (int) $tag->term_id,
					'name' => $tag->name,
					'slug' => $tag->slug,
				];
			}
		}

		return $tags;
	}

	/**
	 * Get product images
	 *
	 * @param WC_Product $product WooCommerce product object.
	 * @return array Product images.
	 */
	private function get_product_images( $product ) {
		$images = [];

		// Main image.
		$image_id = $product->get_image_id();
		if ( $image_id ) {
			$images[] = [
				'id'   => (int) $image_id,
				'src'  => wp_get_attachment_url( $image_id ),
				'name' => get_the_title( $image_id ),
				'alt'  => get_post_meta( $image_id, '_wp_attachment_image_alt', true ),
			];
		}

		// Gallery images.
		$gallery_ids = $product->get_gallery_image_ids();
		foreach ( $gallery_ids as $gallery_id ) {
			$images[] = [
				'id'   => (int) $gallery_id,
				'src'  => wp_get_attachment_url( $gallery_id ),
				'name' => get_the_title( $gallery_id ),
				'alt'  => get_post_meta( $gallery_id, '_wp_attachment_image_alt', true ),
			];
		}

		return $images;
	}

	/**
	 * Get product attributes
	 *
	 * @param WC_Product $product WooCommerce product object.
	 * @return array Product attributes.
	 */
	private function get_product_attributes( $product ) {
		$attributes      = [];
		$product_attrs   = $product->get_attributes();

		foreach ( $product_attrs as $attribute ) {
			$attributes[] = [
				'id'        => $attribute->get_id(),
				'name'      => $attribute->get_name(),
				'position'  => $attribute->get_position(),
				'visible'   => $attribute->get_visible(),
				'variation' => $attribute->get_variation(),
				'options'   => $attribute->get_options(),
			];
		}

		return $attributes;
	}
}
