<?php
/**
 * Order Event Tracker
 *
 * Tracks WooCommerce order events and sends webhooks to MessengerOS immediately.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Order_Event_Tracker
 */
class MessengerOS_WooCommerce_Order_Event_Tracker {

	/**
	 * Order event formatter instance
	 *
	 * @var MessengerOS_WooCommerce_Order_Event_Formatter
	 */
	private $event_formatter;

	/**
	 * OAuth handler instance
	 *
	 * @var MessengerOS_WooCommerce_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Constructor
	 *
	 * @param MessengerOS_WooCommerce_OAuth_Handler $oauth_handler OAuth handler instance.
	 */
	public function __construct( $oauth_handler ) {
		error_log( 'MessengerOS: Order Event Tracker initialized' );

		$this->event_formatter = new MessengerOS_WooCommerce_Order_Event_Formatter();
		$this->oauth_handler   = $oauth_handler;

		$this->register_hooks();
		error_log( 'MessengerOS: Order Event Tracker hooks registered' );
	}

	/**
	 * Register WooCommerce hooks
	 */
	private function register_hooks() {
		// Hook into order creation.
		add_action( 'woocommerce_new_order', [ $this, 'handle_order_create' ], 10, 1 );

		// Hook into order updates.
		add_action( 'woocommerce_update_order', [ $this, 'handle_order_update' ], 10, 1 );

		// Hook into order status changes.
		add_action( 'woocommerce_order_status_changed', [ $this, 'handle_order_status_change' ], 10, 4 );
	}

	/**
	 * Handle order create event
	 *
	 * @param int $order_id Order ID.
	 */
	public function handle_order_create( $order_id ) {
		error_log( 'MessengerOS: Order create detected - Order ID: ' . $order_id );

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			error_log( 'MessengerOS: Failed to get order object for ID: ' . $order_id );
			return;
		}

		$this->send_order_event( 'order_create', $order );
	}

	/**
	 * Handle order update event
	 *
	 * @param int $order_id Order ID.
	 */
	public function handle_order_update( $order_id ) {
		error_log( 'MessengerOS: Order update detected - Order ID: ' . $order_id );

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			error_log( 'MessengerOS: Failed to get order object for ID: ' . $order_id );
			return;
		}

		// Avoid duplicate events - woocommerce_new_order triggers before woocommerce_update_order.
		// Only send update if order already has a date_created (not a brand new order).
		$date_created = $order->get_date_created();
		if ( ! $date_created || ( time() - $date_created->getTimestamp() ) < 5 ) {
			error_log( 'MessengerOS: Skipping order_update for new order (will be handled by order_create)' );
			return;
		}

		$this->send_order_event( 'order_update', $order );
	}

	/**
	 * Handle order status change event
	 *
	 * @param int    $order_id   Order ID.
	 * @param string $old_status Old order status.
	 * @param string $new_status New order status.
	 * @param WC_Order $order    Order object.
	 */
	public function handle_order_status_change( $order_id, $old_status, $new_status, $order ) {
		error_log( 'MessengerOS: Order status changed - Order ID: ' . $order_id . ', From: ' . $old_status . ', To: ' . $new_status );

		// Send order_update event for status changes.
		$this->send_order_event( 'order_update', $order );
	}

	/**
	 * Send order event to MessengerOS
	 *
	 * @param string   $event_type Event type ('order_create' or 'order_update').
	 * @param WC_Order $order      WooCommerce order object.
	 */
	private function send_order_event( $event_type, $order ) {
		// Format event data.
		$event_data = $this->event_formatter->format_order_event( $event_type, $order );

		if ( empty( $event_data ) ) {
			error_log( 'MessengerOS: Failed to format order event data' );
			return;
		}

		error_log( 'MessengerOS: Order event type: ' . $event_type . ', Order ID: ' . $order->get_id() );
		error_log( 'MessengerOS: Order event data: ' . wp_json_encode( $event_data ) );

		// Use OAuth handler's make_api_request to send to MessengerOS API.
		$response = $this->oauth_handler->make_api_request( 'webhooks/order', 'POST', $event_data );

		if ( is_wp_error( $response ) ) {
			error_log( 'MessengerOS: Failed to send order event - ' . $response->get_error_message() );
		} else {
			error_log( 'MessengerOS: Order event sent to MessengerOS successfully' );
			error_log( 'MessengerOS: Order event API response: ' . wp_json_encode( $response ) );
		}
	}
}
