<?php
/**
 * Order Event Formatter
 *
 * Formats order data into webhook payload structure for MessengerOS events.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Order_Event_Formatter
 */
class MessengerOS_WooCommerce_Order_Event_Formatter {

	/**
	 * Format order event data for webhook
	 *
	 * @param string   $event_type Event type ('order_create' or 'order_update').
	 * @param WC_Order $order      WooCommerce order object.
	 * @return array Formatted event data.
	 */
	public function format_order_event( $event_type, $order ) {
		if ( ! $order ) {
			return [];
		}

		$payload = [
			'event'      => $event_type,
			'timestamp'  => gmdate( 'c' ),
			'store_id'   => (string) get_option( 'messengeros_woocommerce_store_id', '' ),
			'order_id'   => (int) $order->get_id(),
			'order'      => $this->get_order_data( $order ),
			'customer'   => $this->get_customer_data( $order ),
		];

		return $payload;
	}

	/**
	 * Get order data
	 *
	 * @param WC_Order $order WooCommerce order object.
	 * @return array Order data.
	 */
	private function get_order_data( $order ) {
		$order_data = [
			'id'               => (int) $order->get_id(),
			'order_number'     => $order->get_order_number(),
			'status'           => $order->get_status(),
			'currency'         => $order->get_currency(),
			'total'            => (float) $order->get_total(),
			'subtotal'         => (float) $order->get_subtotal(),
			'total_tax'        => (float) $order->get_total_tax(),
			'shipping_total'   => (float) $order->get_shipping_total(),
			'discount_total'   => (float) $order->get_discount_total(),
			'payment_method'   => $order->get_payment_method(),
			'payment_method_title' => $order->get_payment_method_title(),
			'date_created'     => $order->get_date_created() ? $order->get_date_created()->format( 'c' ) : null,
			'date_modified'    => $order->get_date_modified() ? $order->get_date_modified()->format( 'c' ) : null,
			'date_completed'   => $order->get_date_completed() ? $order->get_date_completed()->format( 'c' ) : null,
			'date_paid'        => $order->get_date_paid() ? $order->get_date_paid()->format( 'c' ) : null,
			'customer_note'    => $order->get_customer_note(),
			'billing'          => $this->get_address_data( $order, 'billing' ),
			'shipping'         => $this->get_address_data( $order, 'shipping' ),
			'items'            => $this->get_order_items( $order ),
		];

		return $order_data;
	}

	/**
	 * Get customer data
	 *
	 * @param WC_Order $order WooCommerce order object.
	 * @return array Customer data.
	 */
	private function get_customer_data( $order ) {
		$customer_data = [
			'id'         => (int) $order->get_customer_id(),
			'email'      => $order->get_billing_email(),
			'first_name' => $order->get_billing_first_name(),
			'last_name'  => $order->get_billing_last_name(),
			'phone'      => $order->get_billing_phone(),
		];

		return $customer_data;
	}

	/**
	 * Get address data
	 *
	 * @param WC_Order $order WooCommerce order object.
	 * @param string   $type  Address type ('billing' or 'shipping').
	 * @return array Address data.
	 */
	private function get_address_data( $order, $type = 'billing' ) {
		$address = [
			'first_name' => $order->{"get_{$type}_first_name"}(),
			'last_name'  => $order->{"get_{$type}_last_name"}(),
			'company'    => $order->{"get_{$type}_company"}(),
			'address_1'  => $order->{"get_{$type}_address_1"}(),
			'address_2'  => $order->{"get_{$type}_address_2"}(),
			'city'       => $order->{"get_{$type}_city"}(),
			'state'      => $order->{"get_{$type}_state"}(),
			'postcode'   => $order->{"get_{$type}_postcode"}(),
			'country'    => $order->{"get_{$type}_country"}(),
		];

		if ( 'billing' === $type ) {
			$address['email'] = $order->get_billing_email();
			$address['phone'] = $order->get_billing_phone();
		}

		return $address;
	}

	/**
	 * Get order items
	 *
	 * @param WC_Order $order WooCommerce order object.
	 * @return array Array of order items.
	 */
	private function get_order_items( $order ) {
		$items = [];

		foreach ( $order->get_items() as $item_id => $item ) {
			$product = $item->get_product();

			if ( ! $product ) {
				continue;
			}

			$item_data = [
				'id'           => (int) $item_id,
				'name'         => $item->get_name(),
				'product_id'   => (int) $item->get_product_id(),
				'variation_id' => (int) $item->get_variation_id(),
				'quantity'     => (int) $item->get_quantity(),
				'subtotal'     => (float) $item->get_subtotal(),
				'total'        => (float) $item->get_total(),
				'tax'          => (float) $item->get_total_tax(),
				'sku'          => $product->get_sku() ?: null,
				'price'        => (float) $product->get_price( 'edit' ),
			];

			$items[] = $item_data;
		}

		return $items;
	}
}
