<?php
/**
 * MessengerOS Event Tracker
 * 
 * Tracks user events and sends them to the Events service
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class MessengerOS_WooCommerce_Event_Tracker {
    
    /**
     * API client instance
     */
    private $api_client;
    
    /**
     * Constructor - initializes API client and registers WooCommerce event hooks
     * Sets up automatic tracking of customer behavior and e-commerce events
     * Called by: MessengerOS main class when store is connected via OAuth
     */
    public function __construct() {
        // Initialize API client
        if ( ! class_exists('MessengerOS_WooCommerce_API_Client') ) {
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'inc/class-messengeros-api-client.php';
        }
        
        $this->api_client = new MessengerOS_WooCommerce_API_Client();
        
        // Hook into various WooCommerce events
        $this->register_hooks();
    }
    
    /**
     * Register WordPress/WooCommerce hooks for automatic event tracking
     * Sets up listeners for key customer actions and e-commerce events
     * Called by: Constructor during initialization
     */
    private function register_hooks() {
        // Only track if we have credentials
        if ( ! $this->api_client->get_store_id() ) {
            return;
        }
        
        // Page view tracking (frontend only) - controlled by setting
        if ( ! is_admin() ) {
            add_action( 'wp_footer', [ $this, 'track_page_view' ] );
        }
        
        // WooCommerce events - always tracked
        add_action( 'woocommerce_add_to_cart', [ $this, 'track_add_to_cart' ], 10, 6 );
        add_action( 'woocommerce_remove_cart_item', [ $this, 'track_remove_from_cart' ], 10, 2 );
        add_action( 'woocommerce_before_checkout_form', [ $this, 'track_checkout_start' ] );
        add_action( 'woocommerce_thankyou', [ $this, 'track_purchase_complete' ] );
        add_action( 'woocommerce_after_single_product', [ $this, 'track_product_view' ] ); // Product views always tracked
        
        // User events (authenticated)
        add_action( 'wp_login', [ $this, 'track_user_login' ], 10, 2 );
        add_action( 'user_register', [ $this, 'track_user_register' ] );
    }
    
    /**
     * Track page view events for analytics and user behavior analysis
     * Captures page context, product info, and referrer data for marketing insights
     * Called by: WordPress wp_footer hook on frontend pages (when enabled)
     */
    public function track_page_view() {
        // Don't track admin pages or AJAX requests
        if ( is_admin() || wp_doing_ajax() ) {
            return;
        }
        
        // Check if page view tracking is enabled
        if ( ! get_option( 'messengeros_track_page_views', false ) ) {
            return;
        }
        
        $data = [
            'page_type' => $this->get_page_type(),
            'title' => get_the_title(),
            'referrer' => esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'] ?? '')),
        ];
        
        // Add product ID if on product page
        if ( is_product() ) {
            global $product;
            if ( $product ) {
                $data['product_id'] = $product->get_id();
                $data['product_name'] = $product->get_name();
                $data['product_price'] = $product->get_price();
            }
        }
        
        // Add category if on category page
        if ( is_product_category() ) {
            $category = get_queried_object();
            if ( $category ) {
                $data['category_id'] = $category->term_id;
                $data['category_name'] = $category->name;
            }
        }
        
        $this->api_client->send_event( 'page_view', $data );
    }
    
    /**
     * Track add to cart events for conversion tracking and abandoned cart analysis
     * Captures product details, pricing, and cart value for marketing automation
     * Called by: WooCommerce woocommerce_add_to_cart action hook
     */
    public function track_add_to_cart( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {
        $product = wc_get_product( $variation_id ?: $product_id );
        
        if ( ! $product ) {
            return;
        }
        
        $data = [
            'product_id' => $product_id,
            'product_name' => $product->get_name(),
            'product_price' => $product->get_price(),
            'quantity' => $quantity,
            'variation_id' => $variation_id,
            'cart_value' => WC()->cart->get_cart_contents_total(),
        ];
        
        $this->api_client->send_event( 'add_to_cart', $data );
    }
    
    /**
     * Track remove from cart
     */
    public function track_remove_from_cart( $cart_item_key, $cart ) {
        $item = $cart->removed_cart_contents[ $cart_item_key ] ?? null;
        
        if ( ! $item ) {
            return;
        }
        
        $product = wc_get_product( $item['product_id'] );
        
        if ( ! $product ) {
            return;
        }
        
        $data = [
            'product_id' => $item['product_id'],
            'product_name' => $product->get_name(),
            'product_price' => $product->get_price(),
            'quantity' => $item['quantity'],
            'cart_value' => WC()->cart->get_cart_contents_total(),
        ];
        
        $this->api_client->send_event( 'remove_from_cart', $data );
    }
    
    /**
     * Track checkout start
     */
    public function track_checkout_start() {
        $cart = WC()->cart;
        
        $data = [
            'cart_total' => $cart->get_cart_contents_total(),
            'cart_items' => $cart->get_cart_contents_count(),
            'currency' => get_woocommerce_currency(),
        ];
        
        // Add cart items
        $items = [];
        foreach ( $cart->get_cart() as $cart_item ) {
            $product = $cart_item['data'];
            $items[] = [
                'product_id' => $cart_item['product_id'],
                'product_name' => $product->get_name(),
                'quantity' => $cart_item['quantity'],
                'price' => $product->get_price(),
            ];
        }
        $data['items'] = $items;
        
        $this->api_client->send_event( 'checkout_start', $data );
    }
    
    /**
     * Track completed purchase events for revenue tracking and customer lifetime value
     * Captures order details, payment info, and customer data for business analytics
     * Called by: WooCommerce woocommerce_thankyou action hook
     */
    public function track_purchase_complete( $order_id ) {
        $order = wc_get_order( $order_id );
        
        if ( ! $order ) {
            return;
        }
        
        $data = [
            'order_id' => $order_id,
            'order_total' => $order->get_total(),
            'currency' => $order->get_currency(),
            'payment_method' => $order->get_payment_method(),
            'customer_id' => $order->get_customer_id(),
            'customer_email' => $order->get_billing_email(),
        ];
        
        // Add order items
        $items = [];
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( $product ) {
                $items[] = [
                    'product_id' => $product->get_id(),
                    'product_name' => $product->get_name(),
                    'quantity' => $item->get_quantity(),
                    'price' => $item->get_total() / $item->get_quantity(),
                ];
            }
        }
        $data['items'] = $items;
        
        // Use authenticated event if user is logged in
        if ( is_user_logged_in() ) {
            $this->api_client->send_authenticated_event( 'purchase_complete', $data );
        } else {
            $this->api_client->send_event( 'purchase_complete', $data );
        }
    }
    
    /**
     * Track product view
     */
    public function track_product_view() {
        global $product;
        
        if ( ! $product ) {
            return;
        }
        
        $data = [
            'product_id' => $product->get_id(),
            'product_name' => $product->get_name(),
            'product_price' => $product->get_price(),
            'product_category' => wp_get_post_terms( $product->get_id(), 'product_cat', [ 'fields' => 'names' ] ),
        ];
        
        $this->api_client->send_event( 'product_view', $data );
    }
    
    /**
     * Track user login (authenticated event)
     */
    public function track_user_login( $user_login, $user ) {
        $data = [
            'user_id' => $user->ID,
            'user_email' => $user->user_email,
            'user_role' => implode( ',', $user->roles ),
        ];
        
        // Wait a bit for the OAuth token to be available
        add_action( 'init', function() use ( $data ) {
            $this->api_client->send_authenticated_event( 'user_login', $data );
        } );
    }
    
    /**
     * Track user registration
     */
    public function track_user_register( $user_id ) {
        $user = get_user_by( 'id', $user_id );
        
        if ( ! $user ) {
            return;
        }
        
        $data = [
            'user_id' => $user_id,
            'user_email' => $user->user_email,
            'user_role' => implode( ',', $user->roles ),
        ];
        
        $this->api_client->send_event( 'user_register', $data );
    }
    
    /**
     * Get page type for tracking
     */
    private function get_page_type() {
        if ( is_front_page() ) return 'home';
        if ( is_product() ) return 'product';
        if ( is_product_category() ) return 'category';
        if ( is_shop() ) return 'shop';
        if ( is_cart() ) return 'cart';
        if ( is_checkout() ) return 'checkout';
        if ( is_account_page() ) return 'account';
        if ( is_search() ) return 'search';
        if ( is_page() ) return 'page';
        if ( is_single() ) return 'post';
        
        return 'other';
    }
}