<?php
/**
 * MessengerOS API Client
 * 
 * Handles communication with Data Warehouse and Events services
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class MessengerOS_WooCommerce_API_Client {
    
    /**
     * OAuth handler instance
     */
    private $oauth_handler;
    
    /**
     * Initialize API client with OAuth credentials and service endpoints
     * Sets up authenticated communication with MessengerOS platform and services
     * Called by: Event tracker and other API-consuming classes
     */
    public function __construct() {
        $this->oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
    }
    
    /**
     * Get Data Warehouse API credentials with automatic token decryption
     * Retrieves encrypted credentials for authenticated requests to Data Warehouse service
     * Called by: Product sync methods and Data Warehouse API requests
     */
    public function get_dw_credentials() {
        $api_key = get_option( 'messengeros_woocommerce_dw_api_key', '' );
        $api_secret = get_option( 'messengeros_woocommerce_dw_api_secret', '' );
        
        if ( empty( $api_key ) || empty( $api_secret ) ) {
            return false;
        }
        
        return [
            'api_key' => $this->oauth_handler->decrypt_token( $api_key ),
            'api_secret' => $this->oauth_handler->decrypt_token( $api_secret ),
            'webhook_secret' => $this->oauth_handler->decrypt_token( get_option( 'messengeros_woocommerce_dw_webhook_secret', '' ) ),
            'endpoints' => get_option( 'messengeros_woocommerce_dw_endpoints', [] ),
        ];
    }
    
    /**
     * Get Events service endpoints configuration from stored credentials
     * Returns endpoint URLs for sending tracking events to MessengerOS Events API
     * Called by: Event tracking methods when sending customer behavior data
     */
    public function get_events_endpoints() {
        return get_option( 'messengeros_woocommerce_events_endpoints', [] );
    }
    
    /**
     * Get store ID
     */
    public function get_store_id() {
        return get_option( 'messengeros_woocommerce_store_id', '' );
    }
    
    /**
     * Get OAuth access token
     */
    public function get_access_token() {
        $credentials = $this->oauth_handler->get_credentials();
        return $credentials['access_token'] ?? '';
    }
    
    /**
     * Send request to Data Warehouse API
     */
    public function send_dw_request( $endpoint, $method = 'GET', $data = null ) {
        $dw_creds = $this->get_dw_credentials();

        if ( ! $dw_creds ) {
            return new WP_Error( 'no_credentials', 'Data Warehouse credentials not found' );
        }
        
        $endpoints = $dw_creds['endpoints'];
        $url = $endpoints[ $endpoint ] ?? '';
        
        if ( empty( $url ) ) {
            return new WP_Error( 'invalid_endpoint', 'Invalid endpoint: ' . $endpoint );
        }
        
        // Generate signature
        $timestamp = time();
        $body = $data ? json_encode( $data ) : '';
        $payload = $dw_creds['api_key'] . $timestamp . $body;
        $signature = hash_hmac( 'sha256', $payload, $dw_creds['api_secret'] );
        
        $args = [
            'method' => $method,
            'headers' => [
                'X-DW-API-Key' => $dw_creds['api_key'],
                'X-DW-Timestamp' => $timestamp,
                'X-DW-Signature' => $signature,
                'X-Store-UUID' => MessengerOS_WooCommerce_Store_Identifier::get_store_uuid(),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 30,
        ];
        
        if ( $data && in_array( $method, ['POST', 'PUT', 'PATCH'] ) ) {
            $args['body'] = $body;
        }
        
        $response = wp_remote_request( $url, $args );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        
        if ( $response_code >= 400 ) {
            $error_data = json_decode( $response_body, true );
            $error_message = $error_data['error'] ?? 'HTTP ' . $response_code;
            return new WP_Error( 'api_error', $error_message, $response_body );
        }
        
        return json_decode( $response_body, true );
    }
    
    /**
     * Send event to Events service (public endpoint)
     */
    public function send_event( $event_type, $event_data = [] ) {
        $endpoints = $this->get_events_endpoints();
        // Handle both old and new endpoint formats
        $url = $endpoints['public'] ?? $endpoints['events'] ?? '';
        
        if ( empty( $url ) ) {
            return new WP_Error( 'no_endpoint', 'Events endpoint not configured' );
        }
        
        $store_id = $this->get_store_id();
        if ( empty( $store_id ) ) {
            return new WP_Error( 'no_store_id', 'Store ID not found' );
        }
        
        $data = [
            'store_id' => $store_id,
            'event' => $event_type,
            'timestamp' => gmdate('c'), // ISO 8601 format
            'url' => isset( $event_data['url'] ) ? $event_data['url'] : home_url( esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'] ?? '')) ?? '/' ),
            'session_id' => $this->get_session_id(),
            'platform' => 1, // WooCommerce platform
            'data' => $event_data,
        ];
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode( $data ),
            'timeout' => 5, // Short timeout for events
            'blocking' => false, // Don't wait for response
        ];
        
        wp_remote_post( $url, $args );
        
        return true;
    }
    
    /**
     * Send authenticated event to Events service
     */
    public function send_authenticated_event( $event_type, $event_data = [], $retry = true ) {
        $endpoints = $this->get_events_endpoints();
        // Handle both old and new endpoint formats
        $url = $endpoints['authenticated'] ?? $endpoints['events_auth'] ?? '';
        
        if ( empty( $url ) ) {
            return new WP_Error( 'no_endpoint', 'Authenticated events endpoint not configured' );
        }
        
        // Check and refresh token if needed
        $this->oauth_handler->check_and_refresh_token();
        
        $access_token = $this->get_access_token();
        if ( empty( $access_token ) ) {
            return new WP_Error( 'no_token', 'OAuth access token not found' );
        }
        
        $store_id = $this->get_store_id();
        $data = [
            'store_id' => $store_id,
            'event' => $event_type,
            'timestamp' => gmdate('c'), // ISO 8601 format required by Events API
            'url' => isset( $event_data['url'] ) ? $event_data['url'] : home_url( esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'] ?? '')) ?? '/' ),
            'session_id' => $this->get_session_id(),
            'platform' => 1, // WooCommerce platform
            'data' => $event_data,
        ];
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode( $data ),
            'timeout' => 10,
            'blocking' => true, // Must be blocking to handle auth errors
            'sslverify' => true,
        ];
        
        $response = wp_remote_post( $url, $args );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        
        // Handle authentication errors
        if ( $response_code === 401 && $retry ) {
            // Token might be expired, try to refresh
            $refresh_result = $this->oauth_handler->refresh_token();
            if ( ! is_wp_error( $refresh_result ) ) {
                // Retry with new token
                return $this->send_authenticated_event( $event_type, $event_data, false );
            }
            return new WP_Error( 'auth_failed', 'Authentication failed and token refresh failed' );
        }
        
        if ( $response_code >= 400 ) {
            $error_data = json_decode( $response_body, true );
            $error_message = $error_data['error'] ?? $error_data['message'] ?? 'HTTP ' . $response_code;
            return new WP_Error( 'api_error', $error_message, $response_body );
        }
        
        return json_decode( $response_body, true );
    }
    
    /**
     * Get or create session ID for event tracking
     */
    private function get_session_id() {
        // Use cookie-based session ID to avoid session_start issues
        $session_id = isset( $_COOKIE['messengeros_woocommerce_session_id'] ) ? esc_url_raw(wp_unslash($_COOKIE['messengeros_woocommerce_session_id'])) : '';

        if ( empty( $session_id ) ) {
            $session_id = wp_generate_uuid4();
            // Only set cookie if headers haven't been sent (not in cron/CLI context)
            if ( ! headers_sent() ) {
                setcookie( 'messengeros_woocommerce_session_id', $session_id, time() + 86400, '/', '', is_ssl(), true );
            }
        }

        return $session_id;
    }
    
    /**
     * Verify webhook signature from Data Warehouse
     */
    public function verify_webhook_signature( $payload, $signature ) {
        $dw_creds = $this->get_dw_credentials();
        
        if ( ! $dw_creds || empty( $dw_creds['webhook_secret'] ) ) {
            return false;
        }
        
        $expected_signature = hash_hmac( 'sha256', $payload, $dw_creds['webhook_secret'] );
        
        return hash_equals( $expected_signature, $signature );
    }
}