<?php
/**
 * MessengerOS Events API Client
 * 
 * Handles HMAC-authenticated requests to Events service
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class MessengerOS_WooCommerce_Events_API_Client {
    
    /**
     * Send authenticated event to Events service with HMAC signature
     * Packages event data with store context and sends to MessengerOS Events API
     * Called by: Event tracking classes (MessengerOS_Event_Tracker, MessengerOS_API_Client)
     */
    public static function send_event( $event_type, $event_data = [] ) {
        // Get API credentials
        $api_key = get_option( 'messengeros_events_api_key' );
        $api_secret = get_option( 'messengeros_events_api_secret' );
        
        if ( ! $api_key || ! $api_secret ) {
            return new WP_Error( 'no_credentials', 'Events API credentials not found' );
        }
        
        // Get store data
        $store_id = get_option( 'messengeros_store_id' );
        $store_uuid = get_option( 'messengeros_store_uuid' );
        
        // Prepare event data
        $data = [
            'store_id' => $store_id,
            'event' => $event_type,
            'timestamp' => gmdate( 'c' ),
            'url' => isset($_SERVER['REQUEST_URI']) ? home_url( esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) ?? '/' ) : '',
            'session_id' => self::get_session_id(),
            'platform' => 1, // WooCommerce
            'data' => $event_data
        ];
        
        $body = json_encode( $data );
        $timestamp = time();
        
        // Generate HMAC signature
        $payload = $api_key . $timestamp . $body;
        $signature = hash_hmac( 'sha256', $payload, self::decrypt_secret( $api_secret ) );
        
        // Send request
        $url = MESSENGEROS_WOOCOMMERCE_EVENTS_API_URL . '/api/v1/events/auth';
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Events-API-Key' => $api_key,
                'X-Events-Signature' => $signature,
                'X-Events-Timestamp' => $timestamp,
                'X-Store-UUID' => $store_uuid,
            ],
            'body' => $body,
            'timeout' => 10,
            'sslverify' => true,
        ];
        
        $response = wp_remote_post( $url, $args );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        if ( $response_code >= 400 ) {
            $body = wp_remote_retrieve_body( $response );
            return new WP_Error( 'api_error', 'HTTP ' . $response_code, $body );
        }
        
        return json_decode( wp_remote_retrieve_body( $response ), true );
    }
    
    /**
     * Register API key with Events service using master key authentication
     * Creates new API credentials for the store in the Events service
     * Called by: OAuth authorization flow when setting up store credentials
     */
    public static function register_api_key( $store_identifier, $master_key ) {
        $url = MESSENGEROS_WOOCOMMERCE_EVENTS_API_URL . '/internal/api/register-key';
        
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Master-Key' => $master_key,
            ],
            'body' => json_encode( [
                'store_identifier' => $store_identifier,
                'description' => 'WooCommerce Store: ' . get_bloginfo( 'name' ),
                'scopes' => [ 'events:write', 'analytics:read' ]
            ] ),
            'timeout' => 30,
        ];



        $response = wp_remote_post( $url, $args );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( $response_code >= 400 ) {
            return new WP_Error( 'api_error', $data['error'] ?? 'Registration failed' );
        }
        
        // Store encrypted credentials
        if ( isset( $data['api_key'] ) && isset( $data['api_secret'] ) ) {
            update_option( 'messengeros_events_api_key', $data['api_key'] );
            update_option( 'messengeros_events_api_secret', self::encrypt_secret( $data['api_secret'] ) );
            return true;
        }
        
        return new WP_Error( 'invalid_response', 'Invalid response from Events API' );
    }
    
    /**
     * Get or create session ID for tracking user sessions across events
     * Creates persistent session ID cookie for analytics and user behavior tracking
     * Called by: send_event() method when building event data payload
     */
    private static function get_session_id() {
        if ( isset( $_COOKIE['messengeros_session_id'] ) ) {
            return esc_url_raw(wp_unslash($_COOKIE['messengeros_session_id']));
        }
        
        $session_id = wp_generate_uuid4();
        setcookie( 'messengeros_session_id', $session_id, time() + 86400, '/', '', is_ssl(), true );
        
        return $session_id;
    }
    
    /**
     * Simple encryption for API secret storage using WordPress salt
     * Encrypts API secrets before storing in WordPress database for security
     * Called by: register_api_key() method when storing credentials
     */
    private static function encrypt_secret( $secret ) {
        $key = wp_salt( 'auth' );
        return base64_encode( $secret ^ $key );
    }
    
    /**
     * Decrypt API secret using WordPress salt for sending authenticated requests
     * Decrypts stored API secrets when building HMAC signatures for Events API
     * Called by: send_event() method when generating authentication signature
     */
    private static function decrypt_secret( $encrypted ) {
        $key = wp_salt( 'auth' );
        return base64_decode( $encrypted ) ^ $key;
    }
}