<?php
/**
 * MessengerOS Event Hooks
 * 
 * Tracks important WooCommerce events server-side
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}
class MessengerOS_WooCommerce_Event_Hooks {
    
    /**
     * Constructor
     */
    public function __construct() {
        
        // Cart events
        add_action( 'woocommerce_add_to_cart', [ $this, 'track_add_to_cart' ], 10, 4 );
        add_action( 'woocommerce_remove_cart_item', [ $this, 'track_remove_from_cart' ], 10, 2 );
        
        // Checkout events
        add_action( 'woocommerce_checkout_create_order', [ $this, 'track_checkout_started' ], 10, 2 );
        add_action( 'woocommerce_thankyou', [ $this, 'track_purchase' ], 10, 1 );
        
        // Customer events
        add_action( 'user_register', [ $this, 'track_customer_registered' ], 10, 1 );
        add_action( 'wp_login', [ $this, 'track_customer_login' ], 10, 2 );
        
        // Product search
        add_action( 'pre_get_posts', [ $this, 'track_product_search' ] );
    }
    
    /**
     * Track add to cart
     */
    public function track_add_to_cart( $cart_item_key, $product_id, $quantity, $variation_id ) {
        // Check if Events API is configured
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $product = wc_get_product( $variation_id ?: $product_id );
        if ( ! $product ) {
            return;
        }
        
        $event_data = [
            'product_id' => (string) $product_id,
            'variation_id' => (string) $variation_id,
            'product_name' => $product->get_name(),
            'product_price' => $product->get_price(),
            'quantity' => $quantity,
            'cart_value' => WC()->cart->get_cart_contents_total(),
            'currency' => get_woocommerce_currency(),
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'add_to_cart', $event_data );
    }
    
    /**
     * Track remove from cart
     */
    public function track_remove_from_cart( $cart_item_key, $cart ) {
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $cart_item = $cart->cart_contents[ $cart_item_key ] ?? null;
        if ( ! $cart_item ) {
            return;
        }
        
        $product = $cart_item['data'];
        
        $event_data = [
            'product_id' => (string) $cart_item['product_id'],
            'product_name' => $product->get_name(),
            'quantity' => $cart_item['quantity'],
            'cart_value' => WC()->cart->get_cart_contents_total(),
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'remove_from_cart', $event_data );
    }
    
    /**
     * Track checkout started
     */
    public function track_checkout_started( $order, $data ) {
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $items = [];
        foreach ( $order->get_items() as $item ) {
            $items[] = [
                'product_id' => (string) $item->get_product_id(),
                'product_name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'price' => $item->get_total(),
            ];
        }
        
        $event_data = [
            'order_value' => $order->get_total(),
            'currency' => $order->get_currency(),
            'items_count' => count( $items ),
            'items' => $items,
            'customer_email' => $order->get_billing_email(),
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'checkout_started', $event_data );
    }
    
    /**
     * Track purchase
     */
    public function track_purchase( $order_id ) {
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }
        
        // Prevent duplicate tracking
        if ( $order->get_meta( '_messengeros_tracked' ) ) {
            return;
        }
        
        $items = [];
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            $items[] = [
                'product_id' => (string) $item->get_product_id(),
                'product_name' => $item->get_name(),
                'product_sku' => $product ? $product->get_sku() : '',
                'quantity' => $item->get_quantity(),
                'price' => $item->get_total(),
            ];
        }
        
        $event_data = [
            'order_id' => (string) $order_id,
            'order_value' => $order->get_total(),
            'order_subtotal' => $order->get_subtotal(),
            'shipping' => $order->get_shipping_total(),
            'tax' => $order->get_total_tax(),
            'discount' => $order->get_discount_total(),
            'currency' => $order->get_currency(),
            'payment_method' => $order->get_payment_method(),
            'items' => $items,
            'customer' => [
                'id' => (string) $order->get_customer_id(),
                'email' => $order->get_billing_email(),
                'first_name' => $order->get_billing_first_name(),
                'last_name' => $order->get_billing_last_name(),
                'phone' => $order->get_billing_phone(),
                'country' => $order->get_billing_country(),
                'city' => $order->get_billing_city(),
            ],
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'purchase', $event_data );
        
        // Mark as tracked
        $order->update_meta_data( '_messengeros_tracked', true );
        $order->save();
    }
    
    /**
     * Track customer registration
     */
    public function track_customer_registered( $user_id ) {
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $user = get_userdata( $user_id );
        if ( ! $user ) {
            return;
        }
        
        $event_data = [
            'customer_id' => (string) $user_id,
            'customer_email' => $user->user_email,
            'customer_name' => $user->display_name,
            'registration_date' => current_time( 'c' ),
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'customer_registered', $event_data );
    }
    
    /**
     * Track customer login
     */
    public function track_customer_login( $user_login, $user ) {
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $event_data = [
            'customer_id' => (string) $user->ID,
            'customer_email' => $user->user_email,
            'customer_name' => $user->display_name,
            'login_time' => current_time( 'c' ),
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'customer_login', $event_data );
    }
    
    /**
     * Track product search
     */
    public function track_product_search( $query ) {
        if ( ! $query->is_main_query() || ! $query->is_search() || is_admin() ) {
            return;
        }
        
        if ( ! get_option( 'messengeros_events_api_key' ) ) {
            return;
        }
        
        $search_term = get_search_query();
        if ( empty( $search_term ) ) {
            return;
        }
        
        $event_data = [
            'search_term' => $search_term,
            'results_count' => $query->found_posts,
            'page' => $query->get( 'paged' ) ?: 1,
        ];
        
        MessengerOS_WooCommerce_Events_API_Client::send_event( 'product_search', $event_data );
    }
}

// Initialize event hooks
new MessengerOS_WooCommerce_Event_Hooks();