<?php
/**
 * Customer Event Tracker
 *
 * Tracks WooCommerce customer events and sends webhooks to MessengerOS immediately.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Customer_Event_Tracker
 */
class MessengerOS_WooCommerce_Customer_Event_Tracker {

	/**
	 * Customer event formatter instance
	 *
	 * @var MessengerOS_WooCommerce_Customer_Event_Formatter
	 */
	private $event_formatter;

	/**
	 * OAuth handler instance
	 *
	 * @var MessengerOS_WooCommerce_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Constructor
	 *
	 * @param MessengerOS_WooCommerce_OAuth_Handler $oauth_handler OAuth handler instance.
	 */
	public function __construct( $oauth_handler ) {
		error_log( 'MessengerOS: Customer Event Tracker initialized' );

		$this->event_formatter = new MessengerOS_WooCommerce_Customer_Event_Formatter();
		$this->oauth_handler   = $oauth_handler;

		$this->register_hooks();
		error_log( 'MessengerOS: Customer Event Tracker hooks registered' );
	}

	/**
	 * Register WooCommerce hooks
	 */
	private function register_hooks() {
		// Hook into customer creation.
		add_action( 'woocommerce_created_customer', [ $this, 'handle_customer_create' ], 10, 3 );

		// Hook into customer updates.
		add_action( 'woocommerce_update_customer', [ $this, 'handle_customer_update' ], 10, 1 );
		add_action( 'profile_update', [ $this, 'handle_profile_update' ], 10, 2 );
	}

	/**
	 * Handle customer create event
	 *
	 * @param int   $customer_id       Customer ID.
	 * @param array $new_customer_data Customer data.
	 * @param bool  $password_generated Whether password was generated.
	 */
	public function handle_customer_create( $customer_id, $new_customer_data, $password_generated ) {
		error_log( 'MessengerOS: Customer create detected - Customer ID: ' . $customer_id );

		$customer = get_user_by( 'id', $customer_id );
		if ( ! $customer ) {
			error_log( 'MessengerOS: Failed to get customer object for ID: ' . $customer_id );
			return;
		}

		$this->send_customer_event( 'customer_create', $customer );
	}

	/**
	 * Handle customer update event (WooCommerce specific)
	 *
	 * @param int $customer_id Customer ID.
	 */
	public function handle_customer_update( $customer_id ) {
		error_log( 'MessengerOS: Customer update detected (WC) - Customer ID: ' . $customer_id );

		$customer = get_user_by( 'id', $customer_id );
		if ( ! $customer ) {
			error_log( 'MessengerOS: Failed to get customer object for ID: ' . $customer_id );
			return;
		}

		$this->send_customer_event( 'customer_update', $customer );
	}

	/**
	 * Handle profile update event
	 *
	 * @param int      $user_id   User ID.
	 * @param WP_User  $old_user_data Old user data object.
	 */
	public function handle_profile_update( $user_id, $old_user_data ) {
		// Only track customers (users with customer role).
		$user = get_user_by( 'id', $user_id );
		if ( ! $user || ! in_array( 'customer', $user->roles, true ) ) {
			return;
		}

		error_log( 'MessengerOS: Customer profile update detected - Customer ID: ' . $user_id );

		$this->send_customer_event( 'customer_update', $user );
	}

	/**
	 * Send customer event to MessengerOS
	 *
	 * @param string  $event_type Event type ('customer_create' or 'customer_update').
	 * @param WP_User $customer   WP_User object.
	 */
	private function send_customer_event( $event_type, $customer ) {
		// Format event data.
		$event_data = $this->event_formatter->format_customer_event( $event_type, $customer );

		if ( empty( $event_data ) ) {
			error_log( 'MessengerOS: Failed to format customer event data' );
			return;
		}

		error_log( 'MessengerOS: Customer event type: ' . $event_type . ', Customer ID: ' . $customer->ID );
		error_log( 'MessengerOS: Customer event data: ' . wp_json_encode( $event_data ) );

		// Use OAuth handler's make_api_request to send to MessengerOS API.
		$response = $this->oauth_handler->make_api_request( 'webhooks/customer', 'POST', $event_data );

		if ( is_wp_error( $response ) ) {
			error_log( 'MessengerOS: Failed to send customer event - ' . $response->get_error_message() );
		} else {
			error_log( 'MessengerOS: Customer event sent to MessengerOS successfully' );
			error_log( 'MessengerOS: Customer event API response: ' . wp_json_encode( $response ) );
		}
	}
}
