<?php
/**
 * Customer Event Formatter
 *
 * Formats customer data into webhook payload structure for MessengerOS events.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Customer_Event_Formatter
 */
class MessengerOS_WooCommerce_Customer_Event_Formatter {

	/**
	 * Format customer event data for webhook
	 *
	 * @param string      $event_type   Event type ('customer_create' or 'customer_update').
	 * @param int|WP_User $customer     Customer ID or WP_User object.
	 * @return array Formatted event data.
	 */
	public function format_customer_event( $event_type, $customer ) {
		// Get WP_User object if ID was passed.
		if ( is_numeric( $customer ) ) {
			$customer = get_userdata( $customer );
		}

		if ( ! $customer ) {
			return [];
		}

		$payload = [
			'event'      => $event_type,
			'timestamp'  => gmdate( 'c' ),
			'store_id'   => (string) get_option( 'messengeros_woocommerce_store_id', '' ),
			'customer_id' => (int) $customer->ID,
			'customer'   => $this->get_customer_data( $customer ),
		];

		return $payload;
	}

	/**
	 * Get customer data
	 *
	 * @param WP_User $customer WP_User object.
	 * @return array Customer data.
	 */
	private function get_customer_data( $customer ) {
		$wc_customer = new WC_Customer( $customer->ID );

		$customer_data = [
			'id'               => (int) $customer->ID,
			'email'            => $customer->user_email,
			'username'         => $customer->user_login,
			'first_name'       => $wc_customer->get_first_name(),
			'last_name'        => $wc_customer->get_last_name(),
			'display_name'     => $customer->display_name,
			'role'             => $this->get_primary_role( $customer ),
			'date_created'     => $customer->user_registered,
			'billing'          => $this->get_billing_data( $wc_customer ),
			'shipping'         => $this->get_shipping_data( $wc_customer ),
			'total_spent'      => (float) $wc_customer->get_total_spent(),
			'orders_count'     => (int) $wc_customer->get_order_count(),
			'avatar_url'       => get_avatar_url( $customer->ID ),
		];

		return $customer_data;
	}

	/**
	 * Get primary role
	 *
	 * @param WP_User $customer WP_User object.
	 * @return string Primary role.
	 */
	private function get_primary_role( $customer ) {
		$roles = $customer->roles;
		return ! empty( $roles ) ? $roles[0] : 'customer';
	}

	/**
	 * Get billing data
	 *
	 * @param WC_Customer $wc_customer WooCommerce customer object.
	 * @return array Billing data.
	 */
	private function get_billing_data( $wc_customer ) {
		return [
			'first_name' => $wc_customer->get_billing_first_name(),
			'last_name'  => $wc_customer->get_billing_last_name(),
			'company'    => $wc_customer->get_billing_company(),
			'address_1'  => $wc_customer->get_billing_address_1(),
			'address_2'  => $wc_customer->get_billing_address_2(),
			'city'       => $wc_customer->get_billing_city(),
			'state'      => $wc_customer->get_billing_state(),
			'postcode'   => $wc_customer->get_billing_postcode(),
			'country'    => $wc_customer->get_billing_country(),
			'email'      => $wc_customer->get_billing_email(),
			'phone'      => $wc_customer->get_billing_phone(),
		];
	}

	/**
	 * Get shipping data
	 *
	 * @param WC_Customer $wc_customer WooCommerce customer object.
	 * @return array Shipping data.
	 */
	private function get_shipping_data( $wc_customer ) {
		return [
			'first_name' => $wc_customer->get_shipping_first_name(),
			'last_name'  => $wc_customer->get_shipping_last_name(),
			'company'    => $wc_customer->get_shipping_company(),
			'address_1'  => $wc_customer->get_shipping_address_1(),
			'address_2'  => $wc_customer->get_shipping_address_2(),
			'city'       => $wc_customer->get_shipping_city(),
			'state'      => $wc_customer->get_shipping_state(),
			'postcode'   => $wc_customer->get_shipping_postcode(),
			'country'    => $wc_customer->get_shipping_country(),
		];
	}
}
