<?php
/**
 * Cart State Tracker
 *
 * Tracks cart state to distinguish between cart_create and cart_update events.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Cart_State_Tracker
 */
class MessengerOS_WooCommerce_Cart_State_Tracker {

	/**
	 * Transient expiry time (30 days in seconds)
	 */
	const TRANSIENT_EXPIRY = 30 * DAY_IN_SECONDS;

	/**
	 * Get the cart state key for the current user/session
	 *
	 * @return string Transient key.
	 */
	private function get_cart_state_key() {
		if ( is_user_logged_in() ) {
			return 'messengeros_cart_state_user_' . get_current_user_id();
		}

		// For guests, use WooCommerce session ID or fallback to cookie.
		$session_id = $this->get_session_id();
		return 'messengeros_cart_state_session_' . $session_id;
	}

	/**
	 * Get session ID for guest users
	 *
	 * @return string Session ID.
	 */
	private function get_session_id() {
		// Try to get WooCommerce session key.
		if ( WC()->session ) {
			$session_key = WC()->session->get_customer_id();
			if ( ! empty( $session_key ) ) {
				return md5( $session_key );
			}
		}

		// Fallback: Generate consistent hash from IP + User Agent.
		// This allows tracking guests without requiring cookies or sessions.
		$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
		$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
		$fallback_id = md5( $ip . '|' . $user_agent );

		return $fallback_id;
	}

	/**
	 * Get previous cart item count
	 *
	 * @return int Previous cart item count.
	 */
	public function get_previous_cart_count() {
		$key   = $this->get_cart_state_key();
		$state = get_transient( $key );

		if ( false === $state || ! isset( $state['count'] ) ) {
			error_log( 'MessengerOS: No previous cart state found for key: ' . $key );
			return 0;
		}

		error_log( 'MessengerOS: Retrieved previous cart count: ' . $state['count'] . ' from key: ' . $key );
		return (int) $state['count'];
	}

	/**
	 * Get current cart item count
	 *
	 * @return int Current cart item count.
	 */
	public function get_current_cart_count() {
		if ( ! WC()->cart ) {
			return 0;
		}

		return WC()->cart->get_cart_contents_count();
	}

	/**
	 * Update cart state with current count
	 */
	public function update_cart_state() {
		$key   = $this->get_cart_state_key();
		$count = $this->get_current_cart_count();

		$state = [
			'count'     => $count,
			'timestamp' => time(),
		];

		set_transient( $key, $state, self::TRANSIENT_EXPIRY );
		error_log( 'MessengerOS: Cart state saved - Key: ' . $key . ', Count: ' . $count );
	}

	/**
	 * Determine if this is a cart_create or cart_update event
	 *
	 * Logic:
	 * - cart_create: Previous count = 0 AND current count >= 1
	 * - cart_update: Previous count >= 1 OR (previous = 0 but after multiple changes)
	 *
	 * @return string Either 'cart_create' or 'cart_update'.
	 */
	public function determine_event_type() {
		$previous_count = $this->get_previous_cart_count();
		$current_count  = $this->get_current_cart_count();

		error_log( 'MessengerOS: Determining event type - Previous count: ' . $previous_count . ', Current count: ' . $current_count );

		// Cart create: Going from empty (0) to having items (1+).
		if ( 0 === $previous_count && $current_count >= 1 ) {
			error_log( 'MessengerOS: Event type determined: cart_create' );
			return 'cart_create';
		}

		// Cart update: All other scenarios.
		// This includes:
		// - Adding more items when cart already has items
		// - Removing items
		// - Changing quantities
		// - Emptying cart (going to 0).
		error_log( 'MessengerOS: Event type determined: cart_update' );
		return 'cart_update';
	}

	/**
	 * Get session ID for webhook payload
	 *
	 * @return string Session ID.
	 */
	public function get_session_id_for_webhook() {
		return $this->get_session_id();
	}
}
