<?php
/**
 * Cart Event Tracker
 *
 * Tracks WooCommerce cart events and sends webhooks to MessengerOS immediately.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Cart_Event_Tracker
 */
class MessengerOS_WooCommerce_Cart_Event_Tracker {

	/**
	 * Cart state tracker instance
	 *
	 * @var MessengerOS_WooCommerce_Cart_State_Tracker
	 */
	private $state_tracker;

	/**
	 * Cart event formatter instance
	 *
	 * @var MessengerOS_WooCommerce_Cart_Event_Formatter
	 */
	private $event_formatter;

	/**
	 * OAuth handler instance
	 *
	 * @var MessengerOS_WooCommerce_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Constructor
	 *
	 * @param MessengerOS_WooCommerce_OAuth_Handler $oauth_handler OAuth handler instance.
	 */
	public function __construct( $oauth_handler ) {
		error_log( 'MessengerOS: Cart Event Tracker initialized' );

		$this->state_tracker   = new MessengerOS_WooCommerce_Cart_State_Tracker();
		$this->event_formatter = new MessengerOS_WooCommerce_Cart_Event_Formatter();
		$this->oauth_handler   = $oauth_handler;

		$this->register_hooks();
		error_log( 'MessengerOS: Cart Event Tracker hooks registered' );
	}

	/**
	 * Register WooCommerce hooks
	 */
	private function register_hooks() {
		// Hook into cart actions.
		add_action( 'woocommerce_add_to_cart', [ $this, 'handle_cart_add' ], 10, 6 );
		add_action( 'woocommerce_remove_cart_item', [ $this, 'handle_cart_remove' ], 10, 2 );
		add_action( 'woocommerce_after_cart_item_quantity_update', [ $this, 'handle_cart_quantity_update' ], 10, 4 );

		// Also hook into other cart update actions to catch all changes.
		add_action( 'woocommerce_cart_item_removed', [ $this, 'handle_cart_item_removed' ], 10, 2 );
		add_action( 'woocommerce_cart_item_restored', [ $this, 'handle_cart_item_restored' ], 10, 2 );
		add_action( 'woocommerce_before_cart_item_quantity_zero', [ $this, 'handle_cart_quantity_zero' ], 10, 1 );
	}

	/**
	 * Handle cart add event
	 */
	public function handle_cart_add() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_add_to_cart' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart remove event
	 */
	public function handle_cart_remove() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_remove_cart_item' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart quantity update event
	 */
	public function handle_cart_quantity_update() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_after_cart_item_quantity_update' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart item removed event
	 */
	public function handle_cart_item_removed() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_cart_item_removed' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart item restored event
	 */
	public function handle_cart_item_restored() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_cart_item_restored' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart quantity zero event
	 */
	public function handle_cart_quantity_zero() {
		error_log( 'MessengerOS: Cart change detected - woocommerce_before_cart_item_quantity_zero' );
		$this->handle_cart_change();
	}

	/**
	 * Handle cart change events
	 *
	 * This method is called by all cart action hooks.
	 * Sends events immediately without debouncing.
	 */
	private function handle_cart_change() {

		// Check if automation is enabled.
		if ( ! $this->is_automation_enabled() ) {
			error_log( 'MessengerOS: Cart automation is not enabled, skipping event' );
			return;
		}

		error_log( 'MessengerOS: Cart automation is enabled, sending event immediately' );

		// Determine event type.
		$event_type = $this->state_tracker->determine_event_type();

		// Get session ID.
		$session_id = $this->state_tracker->get_session_id_for_webhook();

		// Format event data.
		$event_data = $this->event_formatter->format_cart_event( $event_type, $session_id );

		error_log( 'MessengerOS: Event type: ' . $event_type . ', Cart count: ' . ( $event_data['cart']['count'] ?? 0 ) );
		error_log( 'MessengerOS: Event data: ' . wp_json_encode( $event_data ) );

		// Send event to MessengerOS immediately.
		$this->send_cart_event( $event_type, $event_data );
		error_log( 'MessengerOS: Cart event sent to MessengerOS' );

		// Update cart state for next comparison.
		$this->state_tracker->update_cart_state();
	}

	/**
	 * Send cart event to MessengerOS
	 *
	 * @param string $event_type Event type ('cart_create' or 'cart_update').
	 * @param array  $event_data Event data payload.
	 */
	private function send_cart_event( $event_type, $event_data ) {
		// Use OAuth handler's make_api_request to send to MessengerOS API.
		$response = $this->oauth_handler->make_api_request( 'webhooks/cart', 'POST', $event_data );

		if ( is_wp_error( $response ) ) {
			error_log( 'MessengerOS: Failed to send cart event - ' . $response->get_error_message() );
		} else {
			error_log( 'MessengerOS: Cart event API response: ' . wp_json_encode( $response ) );
		}
	}

	/**
	 * Check if cart automation is enabled
	 *
	 * @return bool Whether automation is enabled.
	 */
	private function is_automation_enabled() {
		return MessengerOS_WooCommerce_Cart_Automation_REST_Endpoint::is_automation_enabled();
	}
}
