<?php
/**
 * Cart Event Formatter
 *
 * Formats cart data into webhook payload structure for MessengerOS events.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Cart_Event_Formatter
 */
class MessengerOS_WooCommerce_Cart_Event_Formatter {

	/**
	 * Format cart event data for webhook
	 *
	 * @param string $event_type Event type ('cart_create' or 'cart_update').
	 * @param string $session_id Session ID.
	 * @return array Formatted event data.
	 */
	public function format_cart_event( $event_type, $session_id ) {
		$cart = WC()->cart;

		if ( ! $cart ) {
			return [];
		}

		$payload = [
			'event'      => $event_type,
			'timestamp'  => gmdate( 'c' ),
			'store_id'   => (string) get_option( 'messengeros_woocommerce_store_id', '' ),
			'session_id' => $session_id,
			'user'       => $this->get_user_data(),
			'cart'       => $this->get_cart_data( $cart ),
		];

		return $payload;
	}

	/**
	 * Get user/customer data
	 *
	 * @return array User data.
	 */
	private function get_user_data() {
		$user_data = [
			'id'    => null,
			'email' => null,
			'type'  => 'guest',
		];

		if ( is_user_logged_in() ) {
			$user              = wp_get_current_user();
			$user_data['id']   = (string) $user->ID;
			$user_data['email'] = $user->user_email;
			$user_data['type'] = 'registered';
		} elseif ( WC()->customer ) {
			// Try to get email from WooCommerce customer object (may be set during checkout).
			$email = WC()->customer->get_billing_email();
			if ( ! empty( $email ) ) {
				$user_data['email'] = $email;
			}
		}

		return $user_data;
	}

	/**
	 * Get cart data
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 * @return array Cart data.
	 */
	private function get_cart_data( $cart ) {
		$cart_data = [
			'count'    => $cart->get_cart_contents_count(),
			'total'    => (float) $cart->get_total( 'edit' ),
			'subtotal' => (float) $cart->get_subtotal(),
			'tax'      => (float) $cart->get_total_tax(),
			'shipping' => (float) $cart->get_shipping_total(),
			'currency' => get_woocommerce_currency(),
			'items'    => $this->get_cart_items( $cart ),
		];

		return $cart_data;
	}

	/**
	 * Get cart items
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 * @return array Array of cart items.
	 */
	private function get_cart_items( $cart ) {
		$items = [];

		foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
			$product = $cart_item['data'];

			if ( ! $product ) {
				continue;
			}

			$item = [
				'product_id'   => (int) $cart_item['product_id'],
				'name'         => $product->get_name(),
				'sku'          => $product->get_sku() ?: null,
				'quantity'     => (int) $cart_item['quantity'],
				'price'        => (float) $product->get_price( 'edit' ),
				'variation_id' => isset( $cart_item['variation_id'] ) && $cart_item['variation_id'] > 0
					? (int) $cart_item['variation_id']
					: null,
			];

			$items[] = $item;
		}

		return $items;
	}
}
