<?php
/**
 * REST API Endpoint for Cart Automation Control
 *
 * Allows MessengerOS platform to enable/disable cart automation tracking
 * via REST API with Bearer token authentication.
 *
 * @package MessengerOS_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class MessengerOS_WooCommerce_Cart_Automation_REST_Endpoint
 */
class MessengerOS_WooCommerce_Cart_Automation_REST_Endpoint {

	/**
	 * OAuth handler instance
	 *
	 * @var MessengerOS_WooCommerce_OAuth_Handler
	 */
	private $oauth_handler;

	/**
	 * Constructor
	 *
	 * @param MessengerOS_WooCommerce_OAuth_Handler $oauth_handler OAuth handler instance.
	 */
	public function __construct( $oauth_handler ) {
		$this->oauth_handler = $oauth_handler;
		add_action( 'rest_api_init', [ $this, 'register_routes' ] );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		register_rest_route(
			'messengeros/v1',
			'/cart-automation',
			[
				'methods'             => 'POST',
				'callback'            => [ $this, 'handle_automation_request' ],
				'permission_callback' => [ $this, 'check_permission' ],
			]
		);
	}

	/**
	 * Check if request is authorized
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool|WP_Error True if authorized, WP_Error otherwise.
	 */
	public function check_permission( $request ) {
		// Get Authorization header.
		$auth_header = $request->get_header( 'Authorization' );

		if ( empty( $auth_header ) ) {
			return new WP_Error(
				'missing_authorization',
				__( 'Authorization header is missing.', 'messengeros-for-woocommerce' ),
				[ 'status' => 401 ]
			);
		}

		// Extract Bearer token.
		if ( ! preg_match( '/^Bearer\s+(.+)$/i', $auth_header, $matches ) ) {
			return new WP_Error(
				'invalid_authorization_format',
				__( 'Invalid authorization format. Expected: Bearer {token}', 'messengeros-for-woocommerce' ),
				[ 'status' => 401 ]
			);
		}

		$token = $matches[1];

		// Validate token using OAuth handler.
		if ( ! $this->oauth_handler->validate_bearer_token( $token ) ) {
			return new WP_Error(
				'invalid_token',
				__( 'Invalid or expired access token.', 'messengeros-for-woocommerce' ),
				[ 'status' => 403 ]
			);
		}

		return true;
	}

	/**
	 * Handle cart automation enable/disable request
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function handle_automation_request( $request ) {
		$params = $request->get_json_params();

		// Validate 'enabled' parameter.
		if ( ! isset( $params['enabled'] ) ) {
			return new WP_Error(
				'missing_parameter',
				__( 'Missing required parameter: enabled', 'messengeros-for-woocommerce' ),
				[ 'status' => 400 ]
			);
		}

		$enabled = (bool) $params['enabled'];

		// Update option.
		update_option( 'messengeros_cart_automation_enabled', $enabled );

		// Return success response.
		return new WP_REST_Response(
			[
				'success' => true,
				'enabled' => $enabled,
				'message' => $enabled
					? __( 'Cart automation enabled successfully.', 'messengeros-for-woocommerce' )
					: __( 'Cart automation disabled successfully.', 'messengeros-for-woocommerce' ),
			],
			200
		);
	}

	/**
	 * Get current automation status
	 *
	 * @return bool Whether cart automation is enabled.
	 */
	public static function is_automation_enabled() {
		return (bool) get_option( 'messengeros_cart_automation_enabled', false );
	}
}
