<?php
/**
 * Admin Product Sync Page
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MessengerOS_WooCommerce_Admin_Product_Sync {
    
    private $product_sync;
    
    public function __construct() {
        // Add submenu with priority 20 to ensure parent menu exists
        add_action( 'admin_menu', [ $this, 'add_sync_menu' ], 20 );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
    }
    
    /**
     * Add product sync submenu
     */
    public function add_sync_menu() {
        // Only add submenu if connected via OAuth
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ( $oauth_handler->is_connected() ) {
            add_submenu_page(
                'messos_woocommerce_settings',
                'Product Sync',
                'Product Sync',
                'manage_options',
                'messos_woocommerce_product_sync',
                [ $this, 'render_sync_page' ]
            );
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_scripts( $hook ) {
        // Only load on the product sync page
        // The hook format for submenus is: {parent_slug}_page_{submenu_slug}
        // So it should be: messos_woocommerce_settings_page_messos_woocommerce_product_sync
        
        // For now, let's check if we're on the product sync page by checking the GET parameter
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only checking page context for script loading, not processing form data
        if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'messos_woocommerce_product_sync' ) {
            return;
        }
        
        wp_enqueue_script(
            'messos-woocommerce-product-sync',
            plugin_dir_url( dirname( __FILE__ ) ) . 'assets/js/product-sync.js',
            [ 'jquery' ],
            '1.0.2',
            true
        );
        
        wp_localize_script( 'messos-woocommerce-product-sync', 'messosWoocommerceSync', [
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'messos_woocommerce_sync_nonce' ),
            'strings' => [
                'syncing' => __( 'Syncing products...', 'messengeros-for-woocommerce' ),
                'success' => __( 'Sync completed!', 'messengeros-for-woocommerce' ),
                'error' => __( 'Sync failed. Please try again.', 'messengeros-for-woocommerce' ),
            ]
        ] );
        
        wp_enqueue_style(
            'messos-woocommerce-admin',
            plugin_dir_url( dirname( __FILE__ ) ) . 'assets/css/admin.css',
            [],
            '1.0.2'
        );
    }
    
    /**
     * Render the sync page
     */
    public function render_sync_page() {
        // Check OAuth connection
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ( ! $oauth_handler->is_connected() ) {
            wp_redirect( admin_url( 'admin.php?page=messos_woocommerce_settings' ) );
            exit;
        }
        
        // Get sync status
        $last_sync = get_option( 'messos_woocommerce_last_product_sync' );
        $total_products = wp_count_posts( 'product' )->publish;
        
        // Count products modified since last sync
        $modified_count = 0;
        if ( $last_sync ) {
            $args = [
                'post_type' => 'product',
                'post_status' => 'publish',
                'date_query' => [
                    [
                        'column' => 'post_modified',
                        'after' => $last_sync,
                    ]
                ],
                'posts_per_page' => -1,
                'fields' => 'ids',
            ];
            $query = new WP_Query( $args );
            $modified_count = $query->found_posts;
            wp_reset_postdata();
        }
        ?>
        <div class="wrap">
            <?php 
            // Include admin header
            include plugin_dir_path( dirname( __FILE__ ) ) . 'inc/admin-header.php';
            ?>
            <h1>Product Sync</h1>
            
            <div class="messos-woocommerce-sync-status">
                <h2>Sync Status</h2>
                <table class="form-table">
                    <tr>
                        <th>Total Products</th>
                        <td><?php echo esc_html( $total_products ); ?></td>
                    </tr>
                    <tr>
                        <th>Last Sync</th>
                        <td>
                            <?php 
                            if ( $last_sync ) {
                                echo esc_html( gmdate( 'F j, Y g:i a', strtotime( $last_sync ) ) );
                            } else {
                                echo esc_html( 'Never' );
                            }
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th>Products Modified Since Last Sync</th>
                        <td><?php echo esc_html( $modified_count ); ?></td>
                    </tr>
                    <tr>
                        <th>Sync Schedule</th>
                        <td>
                            <?php
                            $next_sync = wp_next_scheduled( 'messos_woocommerce_scheduled_product_sync' );
                            if ( $next_sync ) {
                                echo esc_html( 'Next sync: ' . gmdate( 'F j, Y g:i a', $next_sync ) );
                            } else {
                                echo esc_html( 'Not scheduled' );
                            }
                            ?>
                        </td>
                    </tr>
                </table>
            </div>
            
            <div class="messos-woocommerce-sync-actions">
                <h2>Manual Sync</h2>
                <p>Click the button below to manually sync your products to MessengerOS.</p>
                
                <div class="sync-options">
                    <label>
                        <input type="radio" name="sync_type" value="incremental" checked>
                        <strong>Incremental Sync</strong> - Only sync products modified since last sync
                        <?php if ( $modified_count > 0 ) : ?>
                            (<?php echo esc_html( $modified_count ); ?> products)
                        <?php endif; ?>
                    </label>
                    <br><br>
                    <label>
                        <input type="radio" name="sync_type" value="full">
                        <strong>Full Sync</strong> - Sync all products (<?php echo esc_html( $total_products ); ?> products)
                    </label>
                </div>
                
                <p class="submit">
                    <button id="messos-woocommerce-sync-products" class="button button-primary">
                        Sync Products Now
                    </button>
                    <span class="spinner"></span>
                </p>
                
                <div id="sync-progress" style="display: none;">
                    <div class="progress-bar">
                        <div class="progress-bar-fill" style="width: 0%;"></div>
                    </div>
                    <p class="progress-text">Starting sync...</p>
                </div>
                
                <div id="sync-results" style="display: none;">
                    <h3>Sync Results</h3>
                    <div class="sync-results-content"></div>
                </div>
            </div>
            
            <div class="messos-woocommerce-sync-settings">
                <h2>Sync Settings</h2>
                <form method="post" action="">
                    <?php wp_nonce_field( 'messos_woocommerce_sync_settings' ); ?>
                    <table class="form-table">
                        <tr>
                            <th>Automatic Sync Schedule</th>
                            <td>
                                <select name="sync_schedule">
                                    <option value="disabled">Disabled</option>
                                    <option value="twohours" <?php selected( $this->get_current_schedule(), 'twohours' ); ?>>Every 2 Hours</option>
                                    <option value="sixhours" <?php selected( $this->get_current_schedule(), 'sixhours' ); ?>>Every 6 Hours</option>
                                    <option value="daily" <?php selected( $this->get_current_schedule(), 'daily' ); ?>>Daily</option>
                                </select>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="submit" name="save_sync_settings" class="button">Save Settings</button>
                    </p>
                </form>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get current sync schedule
     */
    private function get_current_schedule() {
        $event = wp_get_scheduled_event( 'messos_woocommerce_scheduled_product_sync' );
        return $event ? $event->schedule : 'disabled';
    }
}

// Initialize
new MessengerOS_WooCommerce_Admin_Product_Sync();