<?php
/**
 * Admin Forms Page for MessengerOS WooCommerce Plugin
 * Handles newsletter subscription form configuration
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class MessengerOS_WooCommerce_Admin_Forms {
    
    /**
     * Initialize the forms admin functionality and register WordPress hooks
     * Sets up admin menu, settings registration, and AJAX handlers
     * Called by: Plugin initialization when class is instantiated
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'add_forms_menu'], 11); // Priority 11 to run after main menu
        add_action('admin_init', [$this, 'register_forms_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_messos_woocommerce_save_forms_settings', [$this, 'ajax_save_forms_settings']);
    }
    
    /**
     * Add forms submenu page to WordPress admin menu (OAuth-gated)
     * Creates Forms submenu item only when store is connected via OAuth
     * Called by: WordPress admin_menu hook
     */
    public function add_forms_menu() {
        // Only add submenu if connected via OAuth
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if ($oauth_handler->is_connected()) {
            add_submenu_page(
                'messos_woocommerce_settings',
                'Forms',
                'Forms',
                'manage_options',
                'messos_woocommerce_forms',
                [$this, 'render_forms_page']
            );
        }
    }
    
    /**
     * Register form settings with WordPress Settings API
     * Sets up form validation and sanitization for API keys and configuration
     * Called by: WordPress admin_init hook
     */
    public function register_forms_settings() {
        // Register combined forms settings (includes API key, project, list, and form template)
        register_setting(
            'messos_woocommerce_forms_settings_group',
            'messos_woocommerce_forms_settings',
            [$this, 'validate_forms_settings']
        );
        
        // Add single forms settings section
        add_settings_section(
            'messos_woocommerce_forms_settings_section',
            'Manual API Configuration & Form Settings',
            [$this, 'render_forms_section'],
            'messos_woocommerce_forms'
        );
        
        // Add API key field
        add_settings_field(
            'messos_woocommerce_api_key',
            'Api Key',
            [$this, 'render_field_api_key'],
            'messos_woocommerce_forms',
            'messos_woocommerce_forms_settings_section'
        );
        
        // Add project key field
        add_settings_field(
            'messos_woocommerce_project_key',
            'Project Key',
            [$this, 'render_field_project'],
            'messos_woocommerce_forms',
            'messos_woocommerce_forms_settings_section'
        );
        
        // Add list key field
        add_settings_field(
            'messos_woocommerce_list_key',
            'Subscribers List Key',
            [$this, 'render_field_list'],
            'messos_woocommerce_forms',
            'messos_woocommerce_forms_settings_section'
        );
        
        // Add form template field
        add_settings_field(
            'messos_woocommerce_form_template',
            'Subscription Form Template',
            [$this, 'render_field_form_template'],
            'messos_woocommerce_forms',
            'messos_woocommerce_forms_settings_section'
        );
    }
    
    /**
     * Render the forms page
     */
    public function render_forms_page() {
        // Check if connected, redirect to main settings if not
        $oauth_handler = new MessengerOS_WooCommerce_OAuth_Handler();
        if (!$oauth_handler->is_connected()) {
            wp_redirect(admin_url('admin.php?page=messos_woocommerce_settings'));
            exit;
        }
        ?>
        <div class="wrap">
            <?php 
            // Include admin header
            include plugin_dir_path(dirname(__FILE__)) . 'inc/admin-header.php';
            ?>
            <h1>Forms Configuration</h1>
            
            <form id="messos-woocommerce-forms-settings-form" method="post" action="" onsubmit="return false;">
                <?php do_settings_sections('messos_woocommerce_forms'); ?>
                <?php settings_errors(); ?>
                <div id="messos-woocommerce-forms-settings-message"></div>
                <?php submit_button('Save Changes', 'primary', 'submit', false); ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Render API section
     */
    public function render_api_section() {
        echo '<p>Use this section only if you want to manually configure the API key instead of using OAuth authentication.</p>';
    }
    
    /**
     * Render forms section
     */
    public function render_forms_section() {
        echo '<p>Configure your newsletter subscription forms here.</p>';
    }
    
    /**
     * Render API key field
     */
    public function render_field_api_key() {
        // Get value from API settings for backward compatibility
        $api_settings = get_option('messos_woocommerce_api_settings', []);
        $forms_settings = get_option('messos_woocommerce_forms_settings', []);
        
        $value = isset($forms_settings['api_key']) ? $forms_settings['api_key'] : ($api_settings['api_key'] ?? '');
        
        echo "<input type='text' name='messos_woocommerce_forms_settings[api_key]' value='" . esc_attr($value) . "' size='50' />";
        echo wp_kses_post(__("<p>
                <strong>This is a mandatory field.</strong><br />
                The Api key can be generated in the <a href='https://app.messengeros.com'>MessengerOS Platform</a>: Settings -> <a href='https://app.messengeros.com/crossroads/api-keys'>Api Keys</a> -> View/Create Api Key</p>", "messengeros-for-woocommerce"));
    }
    
    /**
     * Render project key field
     */
    public function render_field_project() {
        // Get value from API settings for backward compatibility
        $api_settings = get_option('messos_woocommerce_api_settings', []);
        $forms_settings = get_option('messos_woocommerce_forms_settings', []);
        
        $value = isset($forms_settings['project']) ? $forms_settings['project'] : ($api_settings['project'] ?? '');
        
        echo "<input type='text' name='messos_woocommerce_forms_settings[project]' value='" . esc_attr($value) . "' size='50' />";
        echo wp_kses_post(__("<p>
                This is an optional field. If left empty, it will send the leads to your <i>default</i> project.<br />
                The Project key can be found in the <a href='https://app.messengeros.com'>MessengerOS Platform</a>: Settings -> <a href='https://app.messengeros.com/crossroads/projects'>Projects</a> -> View/Create Project</p>", "messengeros-for-woocommerce"));
    }
    
    /**
     * Render list key field
     */
    public function render_field_list() {
        // Get value from API settings for backward compatibility
        $api_settings = get_option('messos_woocommerce_api_settings', []);
        $forms_settings = get_option('messos_woocommerce_forms_settings', []);
        
        $value = isset($forms_settings['list']) ? $forms_settings['list'] : ($api_settings['list'] ?? '');
        
        echo "<input type='text' name='messos_woocommerce_forms_settings[list]' value='" . esc_attr($value) . "' size='50' />";
        echo wp_kses_post(__("<p>
                This is an optional field. If left empty, it will send the leads to your <i>default</i> email list from your project.<br />
                The subscribers list key can be found in the <a href='https://app.messengeros.com'>MessengerOS Platform</a>: Settings -> Audience -> <a href='https://app.messengeros.com/crossroads/distribution-lists'>Lists</a> -> View / Create List</p>", "messengeros-for-woocommerce"));
    }
    
    /**
     * Render form template field
     */
    public function render_field_form_template() {
        // Get form template from API settings for backward compatibility
        $api_settings = get_option('messos_woocommerce_api_settings', []);
        $forms_settings = get_option('messos_woocommerce_forms_settings', []);
        
        // Use forms settings if available, otherwise fall back to API settings
        $value = isset($forms_settings['form_template']) ? $forms_settings['form_template'] : 
                (isset($api_settings['form']) && strlen($api_settings['form']) > 2 ? $api_settings['form'] :
                '<div id="messos-woocommerce-newsletter-form-container">
    <form id="messos-woocommerce-newsletter-form" class="messos-woocommerce-newsletter-form-wrapper">
        <label for="messos-woocommerce-email">%email_txt%:</label>
        <input type="email" id="messos-woocommerce-email" name="messos-woocommerce-email" required>
        <label>
            <input type="checkbox" id="messos-woocommerce-terms" name="messos-woocommerce-terms" required>
            I agree with the terms and conditions
        </label><br />
        <div id="messos-woocommerce-error-message" style="display: none;">%fail_txt%</div>
        <button type="submit" id="messos-woocommerce-submit-button" disabled>%btn_txt%</button>
    </form>
    <div id="messos-woocommerce-success-message" style="display: none;">%success_txt%</div>
</div>');
        
        echo "<textarea id='messos_woocommerce_forms_form_template' name='messos_woocommerce_forms_settings[form_template]' rows='10' cols='70'>" . esc_textarea($value) . "</textarea><br />";
        echo esc_html__("To include the form anywhere on the website, use this shortcode:", "messengeros-for-woocommerce") . " <input type='text' value='[messos-woocommerce-newsletter-form]' readonly style='width: 200px;' />";

        echo "<p>" . esc_html__("For dynamic and translated strings, use these predefined variables:", "messengeros-for-woocommerce") . " <code>%email_txt%</code>, <code>%btn_txt%</code>, <code>%success_txt%</code>, <code>%fail_txt%</code>." . "</p>";
        echo "<p>" . esc_html__("These will be replaced by the shortcode parameters that can be added like this:", "messengeros-for-woocommerce") . "</p>";
        echo "<p><code>" . esc_html__('[messos-woocommerce-newsletter-form email_txt="Email label text" btn_txt="Submit button text" success_txt="Subscription sent successfully" fail_txt="Subscription failed"]', "messengeros-for-woocommerce") . "</code></p>";
        echo "<p><strong>" . esc_html__("Required elements:", "messengeros-for-woocommerce") . "</strong> 'messos-email', 'messos-submit-button'</p>";
        echo "<p><strong>" . esc_html__("Required message containers:", "messengeros-for-woocommerce") . "</strong> 'messos-woocommerce-success-message', 'messos-woocommerce-error-message'</p>";
        echo "<p><em>" . esc_html__("To reset to default, leave this field empty.", "messengeros-for-woocommerce") . "</em></p>";
    }
    
    /**
     * Validate API settings
     */
    public function validate_api_settings($input) {
        $valid = [];
        if (empty($input['api_key']) || strlen(trim($input['api_key'])) < 36) {
            add_settings_error(
                'messos_woocommerce_api_settings',
                'api_key_error',
                'API Key must be at least 36 characters.',
                'error'
            );
        } else {
            $valid['api_key'] = sanitize_text_field($input['api_key']);
            $response = wp_remote_post(MESSENGEROS_WOOCOMMERCE_INBOUND_API_URL . '/2.0/check-api-key', [
                'method'  => 'POST',
                'headers' => [
                    'Content-Type' => 'application/json',
                    'x-api-key' => $valid['api_key'],
                ],
                'body'    => "",
            ]);

            if (is_wp_error($response)) {
                $success = false;
                add_settings_error(
                    'messos_woocommerce_api_settings',
                    'api_key_error',
                    'Could not connect to MessengerOS.',
                    'error'
                );
            } else {
                $response_body_str = wp_remote_retrieve_body($response);
                $response_body = json_decode($response_body_str, true);
                if (isset($response_body['status']) && $response_body['status'] === 'success') {
                    $success = true;
                } else {
                    $success = false;
                    add_settings_error(
                        'messos_woocommerce_api_settings',
                        'api_key_error',
                        'The API Key is invalid. Please generate and use an API Key from your MessengerOS Account',
                        'error'
                    );
                }
            }
        }

        // Also save project and list if they exist
        if (isset($input['project'])) {
            $valid['project'] = sanitize_text_field($input['project']);
        }
        if (isset($input['list'])) {
            $valid['list'] = sanitize_text_field($input['list']);
        }

        return $valid;
    }
    
    /**
     * Validate forms settings
     */
    public function validate_forms_settings($input) {
        $valid = [];
        
        // Validate API key if present
        if (isset($input['api_key']) && !empty($input['api_key'])) {
            if (strlen(trim($input['api_key'])) < 36) {
                add_settings_error(
                    'messos_woocommerce_forms_settings',
                    'api_key_error',
                    'API Key must be at least 36 characters.',
                    'error'
                );
            } else {
                $valid['api_key'] = sanitize_text_field($input['api_key']);
                
                // Validate API key against MessengerOS
                $response = wp_remote_post(MESSENGEROS_WOOCOMMERCE_INBOUND_API_URL . '/2.0/check-api-key', [
                    'method'  => 'POST',
                    'headers' => [
                        'Content-Type' => 'application/json',
                        'x-api-key' => $valid['api_key'],
                    ],
                    'body'    => "",
                ]);

                if (is_wp_error($response)) {
                    add_settings_error(
                        'messos_woocommerce_forms_settings',
                        'api_key_error',
                        'Could not connect to MessengerOS.',
                        'error'
                    );
                } else {
                    $response_body_str = wp_remote_retrieve_body($response);
                    $response_body = json_decode($response_body_str, true);
                    if (!isset($response_body['status']) || $response_body['status'] !== 'success') {
                        add_settings_error(
                            'messos_woocommerce_forms_settings',
                            'api_key_error',
                            'The API Key is invalid. Please generate and use an API Key from your MessengerOS Account',
                            'error'
                        );
                    }
                }
            }
        }
        
        // Validate project key
        if (isset($input['project'])) {
            $valid['project'] = sanitize_text_field($input['project']);
        }
        
        // Validate list key
        if (isset($input['list'])) {
            $valid['list'] = sanitize_text_field($input['list']);
        }
        
        // Validate form template
        if (isset($input['form_template'])) {
            $valid['form_template'] = $input['form_template']; // Do NOT sanitize HTML template
        }
        
        return $valid;
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our forms page
        // Check by page parameter instead of hook for now
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only checking page context for script loading, not processing form data
        if (!isset($_GET['page']) || $_GET['page'] !== 'messos_woocommerce_forms') {
            return;
        }
        
        // Enqueue CodeMirror
        wp_enqueue_script('wp-codemirror');
        wp_enqueue_style('wp-codemirror');
        $cm_settings = wp_enqueue_code_editor(['type' => 'text/html']);
        
        // Initialize CodeMirror
        if (false !== $cm_settings) {
            wp_add_inline_script(
                'wp-codemirror',
                'jQuery(document).ready(function($) {
                    wp.codeEditor.initialize($("#messos_woocommerce_forms_form_template"), ' . wp_json_encode($cm_settings) . ');
                });'
            );
        }
        
        // Enqueue our forms admin script
        wp_enqueue_script(
            'messos-woocommerce-forms-admin-js',
            MESSENGEROS_WOOCOMMERCE_PLUGIN_URL . 'assets/js/messos-woocommerce-forms-admin.js',
            ['jquery'],
            '1.0.2',
            true
        );
        
        // Localize script
        wp_localize_script('messos-woocommerce-forms-admin-js', 'messosWoocommerceFormsAjax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('messos_woocommerce_save_forms_settings_action')
        ]);
    }
    
    /**
     * AJAX handler for saving forms settings
     */
    public function ajax_save_forms_settings() {
        $allowed_html = MessengerOS_WooCommerce::messos_woocommerce_get_allowed_html();
        
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'messos_woocommerce_save_forms_settings_action')) {
            wp_send_json_error(['message' => 'Invalid nonce.']);
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Unauthorized.', "messengeros-for-woocommerce")]);
        }
        
        // Convert serializedArray to proper associative array
        $input = [];
        
        if (isset($_POST['settings']) && is_array($_POST['settings'])) {
            $settings_raw = wp_unslash($_POST['settings']);
            
            foreach ($settings_raw as $item) {
                if (
                    isset($item['name'], $item['value']) &&
                    is_string($item['name']) &&
                    is_string($item['value'])
                ) {
                    $name = sanitize_text_field($item['name']);
                    
                    if ($name === 'messos_woocommerce_forms_settings[form_template]') {
                        // Apply custom HTML sanitization only for the form template field
                        $value = wp_kses($item['value'], $allowed_html);
                    } else {
                        $value = sanitize_text_field($item['value']);
                    }
                    
                    parse_str("{$name}=" . urlencode($value), $parsed);
                    $input = array_merge_recursive($input, $parsed);
                }
            }
        }
        
        // Validate API settings if present
        $api_validated = [];
        if (!empty($input['messos_woocommerce_api_settings'])) {
            $api_validated = $this->validate_api_settings($input['messos_woocommerce_api_settings']);
            
            // Check for API validation errors
            $api_errors = get_settings_errors('messos_woocommerce_api_settings');
            if (!empty($api_errors)) {
                wp_send_json_error([
                    'message' => 'API validation failed.',
                    'errors'  => $api_errors
                ]);
            }
            
            // Save API settings
            update_option('messos_woocommerce_api_settings', $api_validated);
        }
        
        // Validate forms settings
        $forms_validated = $this->validate_forms_settings($input['messos_woocommerce_forms_settings'] ?? []);
        
        // Check for forms validation errors
        $errors = get_settings_errors('messos_woocommerce_forms_settings');
        if (!empty($errors)) {
            wp_send_json_error([
                'message' => 'Validation failed.',
                'errors'  => $errors
            ]);
        }
        
        // Save forms settings
        update_option('messos_woocommerce_forms_settings', $forms_validated);
        
        // Also update the API settings for backward compatibility
        $api_settings = get_option('messos_woocommerce_api_settings', []);
        if (isset($forms_validated['api_key'])) {
            $api_settings['api_key'] = $forms_validated['api_key'];
        }
        if (!isset($api_validated['project']) && isset($forms_validated['project'])) {
            $api_settings['project'] = $forms_validated['project'];
        }
        if (!isset($api_validated['list']) && isset($forms_validated['list'])) {
            $api_settings['list'] = $forms_validated['list'];
        }
        $api_settings['form'] = $forms_validated['form_template'] ?? '';
        update_option('messos_woocommerce_api_settings', $api_settings);
        
        wp_send_json_success();
    }
}

// Initialize the forms admin
new MessengerOS_WooCommerce_Admin_Forms();