/**
 * MessengerOS Product View Tracker
 * Lightweight tracking for WooCommerce product views
 */
(function() {
    'use strict';
    
    // Configuration from WordPress
    const config = window.messengerOSTracker || {};
    
    if (!config.eventsUrl || !config.storeId || !config.productId) {
        console.warn('MessengerOS Tracker: Missing configuration');
        return;
    }
    
    // Generate or retrieve session ID
    function getSessionId() {
        let sessionId = sessionStorage.getItem('messos_woocommerce_session_id');
        if (!sessionId) {
            sessionId = generateUUID();
            sessionStorage.setItem('messos_woocommerce_session_id', sessionId);
        }
        return sessionId;
    }
    
    // Simple UUID v4 generator
    function generateUUID() {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
            const r = Math.random() * 16 | 0;
            const v = c === 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    }
    
    // Get viewport dimensions
    function getViewport() {
        return {
            width: window.innerWidth || document.documentElement.clientWidth,
            height: window.innerHeight || document.documentElement.clientHeight
        };
    }
    
    // Track product view
    function trackProductView() {
        const eventData = {
            store_id: config.storeId,
            event: 'product_view',
            timestamp: new Date().toISOString(),
            url: window.location.href,
            session_id: getSessionId(),
            platform: 1, // WooCommerce platform
            data: {
                product_id: config.productId,
                store_uuid: config.storeUuid,
                referrer: document.referrer || '',
                viewport: getViewport(),
                screen: {
                    width: screen.width,
                    height: screen.height
                },
                language: navigator.language
            }
        };
        
        // Add product data if available
        if (config.productData) {
            eventData.data.product = config.productData;
        }
        
        // Send using beacon API for reliability
        if (navigator.sendBeacon) {
            const blob = new Blob([JSON.stringify(eventData)], {
                type: 'application/json'
            });
            navigator.sendBeacon(config.eventsUrl, blob);
        } else {
            // Fallback to fetch
            fetch(config.eventsUrl, {
                method: 'POST',
                body: JSON.stringify(eventData),
                headers: {
                    'Content-Type': 'application/json'
                },
                keepalive: true
            }).catch(function(error) {
                console.error('MessengerOS Tracker error:', error);
            });
        }
    }
    
    // Initialize tracking - only track product view
    trackProductView();
    
})();