<?php
/**
 * This file is part of Media Activity Tracker by ActivityPress.
 *
 * @license GPL-2.0-or-later
 * @copyright 2025 Activitypress LLC
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Adds CPD points when a user logs in for the first time each day.
 *
 * Tracks user logins and awards configurable CPD points once per day
 * when the daily login feature is enabled.
 *
 * @since 1.0.0
 * @param string $user_login The user's login name
 * @param WP_User $user The user object
 */
function mediactr_user_login($user_login, $user)
{

    // only front-end running
    if (is_admin())
    {
        return;
    }

    if (isset($_COOKIE['mediactr_daily_login']) && sanitize_text_field(wp_unslash($_COOKIE['mediactr_daily_login'])) === date_i18n('Y-m-d')) return;

    $login_point = get_option('mediactr_login_points', 0);
    if ($login_point <= 0)
    {
        return;
    }

    global $wpdb;

    $today_start = date_i18n('Y-m-d 00:00:00');
    $today_end = date_i18n('Y-m-d 23:59:59');
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $checkIfExists = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND activity_type = %s AND created_at BETWEEN %s AND %s",
        $user->ID,
        'login',
        $today_start,
        $today_end
    ));
	
    if ($checkIfExists == NULL)
    {
        $data = array(
            'user_id' => $user->ID,
            'activity_type' => 'login',
            'points' => $login_point
        );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result_check = $wpdb->insert($wpdb->prefix . 'gp_tracking_points', $data);
		if ($result_check) {
			if ($mediactr_learndash_is_active == 1) {
				// add Additional Learndash Certicifate if needed
				mediactr_check_alc($user->ID, 'login');
			}
		}	
		
        if (!isset($_COOKIE['mediactr_daily_login'])) mediactr_set_custom_day_cookie();
        else
        {
            $lastday = sanitize_text_field(wp_unslash($_COOKIE['mediactr_daily_login']));
            if ($lastday != date_i18n('Y-m-d')) mediactr_set_custom_day_cookie();
        }
    }
}
add_action('wp_login', 'mediactr_user_login', 10, 2);

/**
 * Handles CPD login tracking for already logged-in users.
 *
 * Ensures login CPD points are awarded to users who remain logged in
 * across day boundaries by checking cookie status.
 *
 * @since 1.0.0
 */
function mediactr_user_loggedin() {

	// Exclude favicon and background requests (WP-Cron, WP-CLI, REST, XML-RPC, AJAX)
	if (
		(isset($_SERVER['REQUEST_URI']) && strpos(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), 'favicon.ico') !== false) || 
		defined('DOING_CRON') && DOING_CRON || 
		defined('WP_CLI') && WP_CLI || 
		defined('REST_REQUEST') && REST_REQUEST || 
		defined('XMLRPC_REQUEST') && XMLRPC_REQUEST || 
		defined('DOING_AJAX') && DOING_AJAX || 
		is_admin() || 
		(isset($_SERVER['REQUEST_URI']) && strpos(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), '/wp-admin/') !== false)
	) {
		return; // Exit if any condition is met
	}
	
	if (isset($_COOKIE['mediactr_daily_login']) && $_COOKIE['mediactr_daily_login'] === date_i18n('Y-m-d')) return;

	if (is_user_logged_in())
	{

		$login_point = get_option('mediactr_login_points', 0);
		if ($login_point <= 0)
		{
			return;
		}

		global $wpdb;

		if (!isset($_COOKIE['mediactr_daily_login']))
		{
			$today_start = date_i18n('Y-m-d 00:00:00');
			$today_end = date_i18n('Y-m-d 23:59:59');
			$user = get_current_user_id();

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$checkIfExists = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND activity_type = %s AND created_at BETWEEN %s AND %s", $user, 'login', $today_start, $today_end));			

			if ($checkIfExists == NULL)
			{
				$data = array(
					'user_id' => get_current_user_id() ,
					'activity_type' => 'login',
					'points' => $login_point
				);
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$result_check = $wpdb->insert($wpdb->prefix . 'gp_tracking_points', $data);
			}
			mediactr_set_custom_day_cookie();
		}
		else
		{
			$lastday = sanitize_text_field(wp_unslash($_COOKIE['mediactr_daily_login']));
			if ($lastday != date_i18n('Y-m-d')) mediactr_set_custom_day_cookie();
		}
	}
}
add_action('wp', 'mediactr_user_loggedin');

/**
 * Sets a daily cookie to track login CPD awards.
 *
 * Creates a browser cookie that expires at midnight to prevent
 * duplicate CPD points from being awarded for multiple logins in one day.
 *
 * @since 1.0.0
 */
function mediactr_set_custom_day_cookie()
{
    // Set the cookie name
    $cookie_name = 'mediactr_daily_login';

    // Set the cookie value to the current day (yyyy-mm-dd format)
    $cookie_value = date_i18n('Y-m-d');

    // Calculate the expiration for the end of the day
    $expiration_time = strtotime('tomorrow midnight') - time();

    // Set the cookie
    setcookie($cookie_name, $cookie_value, time() + $expiration_time, '/');
}

?>