<?php
/**
 * This file is part of Media Activity Tracker by ActivityPress.
 *
 * @license GPL-2.0-or-later
 * @copyright 2025 Activitypress LLC
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Source: https://developers.learndash.com/hook/learndash_course_completed/
 */

/**
 * Awards CPD points when a LearnDash course is completed.
 *
 * Tracks course completions and awards CPD points based on the course's
 * configured point value when LearnDash integration is active.
 *
 * @since 1.0.0
 * @param array $data Course completion data including user and course objects
 */
function mediactr_course_completed($data)
{

	if (is_user_logged_in()) {	
	    $current_post_type = 'sfwd-courses';
	    $allowed_post_types = get_option('mediactr_posttypes', array());
	    if (!in_array($current_post_type, $allowed_post_types))
	    {
	        return;
	    }

	    global $wpdb;

	    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $checkIfExists = $wpdb->get_row($wpdb->prepare("SELECT id FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND course_id = %d AND lesson_id IS NULL", $data['user']->ID, $data['course']->ID));
	    if ($checkIfExists == NULL)
	    {
	    	/* 2DO: default coming from settings */
			$cpd_points = learndash_get_course_points($data['course']->ID, 2) * 3600;

			if ($cpd_points > 0)
			{
				$query_data = array(
					user_id => $data['user']->ID,
					course_id => $data['course']->ID,
					activity_type => 'completion',
					points => $cpd_points
				);
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
				$result_check = $wpdb->insert($wpdb->prefix . 'gp_tracking_points', $query_data);
				if ($result_check) {
					// add Additional Learndash Certicifate if needed
					mediactr_check_alc($data['user']->ID, 'completion');
				}					
			}
	    }
	}
}

add_action('learndash_course_completed', 'mediactr_course_completed', 10, 1);

/***** CERTIFICATES *****/

/**
 * Adds additional LearnDash certificates to user records.
 *
 * Processes certificate awards and creates notification records when
 * a user earns a certificate through CPD activities.
 *
 * @since 1.0.0
 * @param int $user_id The ID of the user
 * @param array $results Array of certificate post objects to award
 */
function mediactr_add_alc($user_id, $results) {
	global $wpdb;
	
	$acl_image = get_option('mediactr_badge_url');
	
	foreach ($results as $result) {
		$cert_id = $result->ID;
		$cert_title = $result->post_title;

		// Data to insert
		$data = array(
			'user_id' => $user_id,
			'alc_id' => $result->ID
		);

		// Construct a SQL query to check if a matching record exists
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$existing_record = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "gp_activity WHERE user_id = %d AND alc_id = %d", $data['user_id'], $data['alc_id']));

		// save the new acl
		if (empty($existing_record)) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$check = $wpdb->insert($wpdb->prefix . 'gp_activity', $data);
			
			wp_cache_flush_group('media-activity-tracker');
			
			if ( $check !== false ) {

				// check if Buddyboss is active or not
				if (is_plugin_active('buddyboss/buddyboss.php')) {

					// push a popup noty 
					$notifications_array = get_user_meta($user_id, 'ld_achievements_notifications', true);
					
					/* translators: %s: Achieved certificate name */
					$message = __('Congratulations! You have achieved the %s award.', 'media-activity-tracker');
					$new_entry = array(
						'title' => $cert_title,
						'message' => sprintf($message, $cert_title),
						'image' => $acl_image
					);
					
					if (!empty($notifications_array)) {
						array_push($notifications_array, $new_entry);
						update_user_meta($user_id, 'ld_achievements_notifications', $notifications_array);
					} else {
						$notifications_array = array($new_entry);
						add_user_meta($user_id, 'ld_achievements_notifications', $notifications_array, true);
					}	
				}
			}
		}
	}
}

/**
 * Checks if user activities qualify for certificate awards.
 *
 * Evaluates the user's CPD activity against certificate thresholds
 * to determine if new certificates should be awarded.
 *
 * @since 1.0.0
 * @param int $user_id The ID of the user
 * @param string $activity_type The type of activity being checked
 */
function mediactr_check_alc($user_id, $activity_type) {
	global $wpdb;
	$table = $wpdb->prefix . 'gp_tracking_points';
	
	/* check the activity type certs */
	// check if the user has more points than any certificate
	$lifeTimePoints = "SELECT SUM(points) FROM $table WHERE user_id = '$user_id' AND activity_type = '$activity_type'";

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results($wpdb->prepare("
    SELECT p.ID, p.post_title 
    FROM {$wpdb->posts} p
    LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
    LEFT JOIN {$wpdb->postmeta} pmv ON p.ID = pmv.post_id
    WHERE p.post_type = %s
    AND p.post_status = %s
    AND pm.meta_key = %s
    AND pm.meta_value = %s
    AND pmv.meta_key = %s
    AND pmv.meta_value <= %d", 'gp_ld_certificates', 'publish', '_gp_ld_cert_activity', $activity_type, '_gp_ld_cert_threshold', $lifeTimePoints));
	if ( !empty($results) ) {
		mediactr_add_alc($user_id, $results);
	}
	
	// check the overall type certs
	$restriction = ( get_option('mediactr_login_points') == 0 ) ? " AND activity_type != 'login'" : "";
	$lifeTimePoints = "SELECT SUM(points) FROM $table WHERE user_id = '$user_id'$restriction";

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results($wpdb->prepare("
    SELECT p.ID, p.post_title 
    FROM {$wpdb->posts} p
    LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
    LEFT JOIN {$wpdb->postmeta} pmv ON p.ID = pmv.post_id
    WHERE p.post_type = %s
    AND p.post_status = %s
    AND pm.meta_key = %s
    AND pm.meta_value = %s
    AND pmv.meta_key = %s
    AND pmv.meta_value <= %d", 'gp_ld_certificates', 'publish', '_gp_ld_cert_activity', 'overall', '_gp_ld_cert_threshold', $lifeTimePoints));
	if ( !empty($results) ) {
		mediactr_add_alc($user_id, $results);
	}
	
	// check the streak type certs
	$dayStreakInRow = mediactr_front_streak_calc($user_id, 0);

	if ( $dayStreakInRow > 0 ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$results = $wpdb->get_results($wpdb->prepare("
    SELECT p.ID, p.post_title 
    FROM {$wpdb->posts} p
    LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
    LEFT JOIN {$wpdb->postmeta} pmv ON p.ID = pmv.post_id
    WHERE p.post_type = %s
    AND p.post_status = %s
    AND pm.meta_key = %s
    AND pm.meta_value = %s
    AND pmv.meta_key = %s
    AND pmv.meta_value <= %d", 'gp_ld_certificates', 'publish', '_gp_ld_cert_activity', 'streak', '_gp_ld_cert_threshold', $dayStreakInRow));
		if ( !empty($results) ) {
			mediactr_add_alc($user_id, $results);
		}
	}
}

/**
 * Generates a certificate download link for a user.
 *
 * Creates a properly formatted URL for accessing a user's earned
 * certificate PDF with appropriate security nonces.
 *
 * @since 1.0.0
 * @param int $alc_id The ID of the certificate post
 * @param int $cert_user_id The ID of the user who earned the certificate
 * @return string URL to download the certificate
 */
function mediactr_alc_get_certificate_link( $alc_id, $cert_user_id ) {
	$cert_user_id = ! empty( $cert_user_id ) ? intval( $cert_user_id ) : get_current_user_id();

	if ( ( empty( $alc_id ) ) || ( empty( $cert_user_id ) ) ) {
		return '';
	}

	$certificate_id = get_post_meta( $alc_id, '_gp_ld_cert_id', true );
	if ( empty( $certificate_id ) ) {
		return '';
	}

	if ( ( learndash_get_post_type_slug( 'certificate' ) !== get_post_type( $certificate_id ) ) ) {
		return '';
	}

	if ( ( learndash_is_admin_user() ) || ( learndash_is_group_leader_user() ) ) {
		$view_user_id = get_current_user_id();
	} else {
		$view_user_id = $cert_user_id;
	}

	$cert_query_args = array(
		'course_id' => $alc_id,
	);	
	
	// We add the user query string key/value if the viewing user is an admin. This
	// allows the admin to view other user's certificated.
	if ( ( $cert_user_id != $view_user_id ) && ( ( learndash_is_admin_user() ) || ( learndash_is_group_leader_user() ) ) ) {
		$cert_query_args['user'] = $cert_user_id;
	}
	$cert_query_args['cert-nonce'] = wp_create_nonce( 'mediactr_' . $alc_id . $cert_user_id . $view_user_id );

	$url = add_query_arg( $cert_query_args, get_permalink( $certificate_id ) );
	
	return $url;
}

/**
 * SHORTCODE: Returns the count of certificates earned by the current user.
 *
 * Shortcode function that displays the number of certificates
 * the logged-in user has earned.
 *
 * @since 1.0.0
 * @return int|bool Number of certificates or FALSE if not logged in
 */
function mediactr_show_alc_num() {
	if (is_user_logged_in()) {	
		global $wpdb;
		$user_id = get_current_user_id();	
		
		// Create a unique cache key for this query
		$cache_key = 'user_certificates_count_' . $user_id;

		// Try to get from cache first
		$record = wp_cache_get($cache_key, 'media-activity-tracker');

		// If not in cache or cache expired, query the database
		if (false === $record) {
			// Execute database query
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$record = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM " . $wpdb->prefix . "gp_activity WHERE user_id = %d", $user_id));

			// Cache the result (for 1 hour = 3600 seconds) 6 hours
			wp_cache_set($cache_key, $record, 'user_activities', 21600);
		}
		
		return $record;
	}
}
add_shortcode('mediactr_show_alc_num', 'mediactr_show_alc_num');

/**
 * SHORTCODE: Renders a list of certificates earned by the current user.
 *
 * Shortcode function that displays the certificates earned by the user
 * with download links and earned dates.
 *
 * @since 1.0.0
 * @param array $atts Shortcode attributes
 * @return string|bool HTML output for certificate list or FALSE if not logged in
 */
function mediactr_show_alc_list($atts) {

	if (is_user_logged_in()) {
		
		$mode = (isset($atts) && !empty($atts)) ? $atts['mode'] : '';
		
		global $wpdb;
		$user_id = get_current_user_id();

		// Create a unique cache key for this query
		$cache_key = 'user_achieved_count_' . $user_id;

		// Try to get from cache first
		$results = wp_cache_get($cache_key, 'media-activity-tracker');

		// If not in cache or cache expired, query the database
		if (false === $results) {
			// Execute database query
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT a.alc_id, a.activity_time, p.post_title FROM " . $wpdb->prefix . "gp_activity a LEFT JOIN wp_posts p ON a.alc_id = p.ID WHERE a.user_id = %d ORDER BY a.activity_time", 
				$user_id
			));

			// Cache the result (for 1 hour = 3600 seconds) 6 hours
			wp_cache_set($cache_key, $results, 'media-activity-tracker', 21600);
		}
		if ( !empty($results) ) {
			$output = '';
			if ( $mode == 'noul' ) $output = '<ul>';
			foreach ($results as $result) {
				$link = mediactr_alc_get_certificate_link( $result->alc_id, $user_id );
				$output.= '<li class="sm-grid-1-1">
						<div class="bb-certificate-wrap">
							<div class="bb-certificate-content">
								<h3 class="bb-certificate-title">
									<span>Certificate in </span>
									<a href="'.$link.'">'.$result->post_title.'</a>
								</h3>
								<div class="bb-certificate-date">
								<span>Earned on</span>
								'.$result->activity_time.'
								</div>
								<p class="bb-certificate-download">
									<a href="'.$link.'"><i class="bb-icon-rl bb-icon-arrow-down" aria-hidden="true"></i>'.__( 'Download PDF', 'media-activity-tracker' ).'</a>
								</p>
							</div>
						</div>
					</li>';

			}
			if ( $mode == 'noul' ) $output.= '</ul>';
		
		return $output;
		}
		else return FALSE;
	}
}
add_shortcode('mediactr_show_alc', 'mediactr_show_alc_list');

/**
 * Handles certificate PDF generation for custom certificates.
 *
 * Extends LearnDash's certificate system to support ActivityPress
 * custom certificate generation and access control.
 *
 * @since 1.0.0
 */
function mediactr_custom_certificate_disallowed_action() {
   // Only process on certificate pages
   if (!is_singular(learndash_get_post_type_slug('certificate'))) {
       return;
   }

   // Early validation - check required parameters
   if (!isset($_GET['cert-nonce']) || empty($_GET['cert-nonce'])) {
       wp_die(esc_html__('Missing security token.', 'media-activity-tracker'));
   }

   if (!isset($_GET['course_id']) || empty($_GET['course_id'])) {
       wp_die(esc_html__('Missing course ID.', 'media-activity-tracker'));
   }

   // Sanitize and validate input
   $nonce = sanitize_text_field(wp_unslash($_GET['cert-nonce']));
   $course_id = absint($_GET['course_id']);
   $certificate_post = get_post(get_the_ID());

   if (!$certificate_post) {
       wp_die(esc_html__('Certificate not found.', 'media-activity-tracker'));
   }

   // Determine user IDs
   $view_user_id = get_current_user_id();
   
   if ((learndash_is_admin_user() || learndash_is_group_leader_user()) && 
       isset($_GET['user']) && !empty($_GET['user'])) {
       $cert_user_id = absint($_GET['user']);
   } else {
       $cert_user_id = get_current_user_id();
   }

   // Verify nonce with correct logic
   if (!wp_verify_nonce($nonce, 'mediactr_' . $course_id . $cert_user_id . $view_user_id)) {
       wp_die(esc_html__('Invalid security token.', 'media-activity-tracker'));
   }

   // Permission checks
   if ($cert_user_id !== $view_user_id) {
       // Admin or Group Leader can view others' certificates
       if (!learndash_is_admin_user() && !learndash_is_group_leader_user()) {
           wp_die(esc_html__('Insufficient permissions to view this certificate.', 'media-activity-tracker'));
       }
   }

   // Verify the certificate exists and is associated with the course
   $certificate_id = get_post_meta($course_id, '_gp_ld_cert_id', true);
   if (empty($certificate_id)) {
       wp_die(esc_html__('No certificate associated with this course.', 'media-activity-tracker'));
   }

   if (learndash_get_post_type_slug('certificate') !== get_post_type($certificate_id)) {
       wp_die(esc_html__('Invalid certificate type.', 'media-activity-tracker'));
   }

   // Verify user has earned this certificate
   global $wpdb;
   // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
   $has_certificate = $wpdb->get_var($wpdb->prepare(
       "SELECT COUNT(*) FROM {$wpdb->prefix}gp_activity WHERE user_id = %d AND alc_id = %d",
       $cert_user_id,
       $course_id
   ));

   if (!$has_certificate) {
       wp_die(esc_html__('Certificate not earned by this user.', 'media-activity-tracker'));
   }

   // Generate certificate PDF - NO user switching
   if (has_action('learndash_tcpdf_init')) {
       do_action(
           'learndash_tcpdf_init',
           array(
               'cert_id' => $certificate_id,
               'user_id' => $cert_user_id,
               'post_id' => $course_id,
           )
       );
   } else {
       // Fallback to direct PDF generation
       if (file_exists(WP_PLUGIN_DIR . '/sfwd-lms/includes/ld-convert-post-pdf.php')) {
           require_once WP_PLUGIN_DIR . '/sfwd-lms/includes/ld-convert-post-pdf.php';
           learndash_certificate_post_shortcode(
               array(
                   'cert_id' => $certificate_id,
                   'user_id' => $cert_user_id,
                   'post_id' => $course_id,
               )
           );
       } else {
           wp_die(esc_html__('Certificate generation system not available.', 'media-activity-tracker'));
       }
   }

   // Terminate execution after certificate generation
   exit;
}
add_action('learndash_certificate_disallowed', 'mediactr_custom_certificate_disallowed_action');

/**
 * SHORTCODE: Provides shortcode functionality for certificate data.
 *
 * Allows dynamic insertion of certificate details into certificate templates,
 * such as completion dates.
 *
 * @since 1.0.0
 * @param array $mode_atts Shortcode attributes
 * @return string Certificate data based on requested field
 */
function mediactr_alc_shortcode_function( $mode_atts ) {
	global $wpdb;
	$user_id = get_current_user_id();
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
	$course_id = isset( $_GET['course_id'] ) ? absint( $_GET['course_id'] ) : 0;
	
    $default_atts = array(
        'show' => 'completed_on',
    );

    $atts = shortcode_atts( $default_atts, $mode_atts, 'cpd' );
	
	switch ($atts['show']) {
    case 'completed_on':
		// Create a unique cache key for this query
		$cache_key = 'user_completed_date_' . $user_id;
		// Try to get from cache first
		$results = wp_cache_get($cache_key, 'media-activity-tracker');
		// If not in cache or cache expired, query the database
		if (false === $results) {
			// Execute database query
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT a.alc_id, a.activity_time, p.post_title FROM " . $wpdb->prefix . "gp_activity a LEFT JOIN wp_posts p ON a.alc_id = p.ID WHERE a.user_id = %d ORDER BY a.activity_time", 
				$user_id
			));

			// Cache the result (for 1 hour = 3600 seconds) 1 week
			wp_cache_set($cache_key, $results, 'media-activity-tracker', 604800);
		}
        $return = date_i18n('F j, Y', strtotime($date));
        break;
	default:
		$return = '';
	}
	
    return $return;
	
}
add_shortcode('mediactr_cpd', 'mediactr_alc_shortcode_function');

/**
 * Cleans up certificate records when certificates are deleted.
 *
 * Ensures data consistency by removing associated activity records
 * when certificate posts are deleted.
 *
 * @since 1.0.0
 * @param int $postid The ID of the post being deleted
 * @param WP_Post $post The post being deleted
 */
function mediactr_before_certificate_delete( $postid, $post ) {
	global $wpdb;
	
	if ( 'gp_ld_certificates' == $post->post_type ) {
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->query( $wpdb->prepare("DELETE FROM " . $wpdb->prefix . "gp_activity WHERE alc_id = %d", $postid) );
		
	}
	elseif ( 'sfwd-certificates' == $post->post_type ) {
		/*$args = array(
			'post_type' => 'gp_ld_certificates',
			'posts_per_page' => -1,
			'fields' => 'ids', // This ensures that only post IDs are returned
			'meta_query' => array(
				array(
					'key' => '_gp_ld_cert_id',
					'value' => $postid,
					'compare' => '='
				)
			)
		);

		$posts_with_meta = new WP_Query($args);*/
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$posts_with_meta = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT p.ID FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE p.post_type = 'gp_ld_certificates'
        AND pm.meta_key = '_gp_ld_cert_id' 
        AND pm.meta_value = %s",
				$postid
			)
		);		

		if ($posts_with_meta->have_posts()) {
			$post_ids = $posts_with_meta->posts; // Get an array of post IDs

			$ids_to_delete = array();
			foreach ($post_ids as $post_id) {
				$ids_to_delete[] = $post_id;
			}

			// delete the CPT
			foreach ($ids_to_delete as $post_id) {
				wp_delete_post($post_id, true);
			}

			// delete the generated certificates			
			$safe_ids = array_map('intval', $ids_to_delete);
			foreach ($safe_ids as $id) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->prefix}gp_activity WHERE alc_id = %d", $id));
			}
		} 	
	}
}

add_action( 'before_delete_post', 'mediactr_before_certificate_delete', 99, 2 );

?>