<?php
/**
 * Plugin Name: Media Activity Tracker
 * Description: Adds CPD functionality with a video/podcast/reading tracker for WordPress.
 * Author: ActivityPress
 * Version: 1.0.2
 * Author URI: https://activitypress.io/
 *
 * Text Domain: media-activity-tracker
 * Function Prefix: mediactr_  
 * 
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * @copyright 2025 ActivityPress LLC
*/

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

/********** INIT FUNCTIONS **********/

// Check if Learndash is active on the WP

if( !function_exists('is_plugin_active') ) {
    include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
}

global $mediactr_learndash_is_active;
$mediactr_learndash_is_active = (is_plugin_active('sfwd-lms/sfwd_lms.php')) ? 1 : 0;

// CONFIG DB hook
register_activation_hook( __FILE__, 'mediactr_activation_function' );

/**
 * Sets up database tables on plugin activation.
 *
 * Creates the necessary database tables for storing CPD activity tracking
 * data when the plugin is activated.
 *
 * @since 1.0.0
 */
function mediactr_activation_function() {
    // create DB-s if necessary
    global $wpdb;
  
    if (!mediactr_table_exists('gp_activity')) {

        // Table doesn't exist, so we create it
        
        $sql = "CREATE TABLE {$wpdb->prefix}gp_activity (
            id int(10) unsigned NOT NULL AUTO_INCREMENT,
            user_id int(10) unsigned NOT NULL,
            alc_id int(10) unsigned NOT NULL,
            activity_time timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
            PRIMARY KEY (id),
            KEY user_id (user_id)
        )";

        // Include WordPress database upgrade script
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        // Execute the SQL query
        dbDelta($sql);
		
		wp_cache_flush_group('media-activity-tracker');

    }

	if (!mediactr_table_exists('gp_tracking_points')) {
		
        // Table doesn't exist, so we create it
        $sql = "CREATE TABLE {$wpdb->prefix}gp_tracking_points (
            id int(10) unsigned NOT NULL AUTO_INCREMENT,
            user_id int(10) unsigned NOT NULL,
            post_id int(10) unsigned DEFAULT NULL,
            course_id int(10) unsigned DEFAULT NULL,
            lesson_id int(10) unsigned DEFAULT NULL,
            topic_id int(10) unsigned DEFAULT NULL,
            activity_type enum('reading','watching','listening','completion','writing','login') COLLATE utf8mb4_unicode_ci NOT NULL,
            progress varchar(300) COLLATE utf8mb4_unicode_ci DEFAULT NULL,
            lastpercent smallint(6) DEFAULT NULL,
            points smallint(6) NOT NULL DEFAULT 0,
            created_at timestamp NULL DEFAULT current_timestamp(),
            finished_at timestamp NULL DEFAULT NULL,
            reflection text COLLATE utf8mb4_unicode_ci NOT NULL,
            PRIMARY KEY (id),
            KEY user_id (user_id)
        )";

        // Include WordPress database upgrade script
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        // Execute the SQL query
        dbDelta($sql);
		
		wp_cache_flush_group('media-activity-tracker');

    }

}

/**
 * Logs a message when the plugin is deactivated.
 *
 * Creates a log entry to record the deactivation of the plugin.
 * Useful for tracking plugin status changes for maintenance or auditing purposes.
 *
 * @since 1.0.0
 */

function mediactr_deactivation( $plugin ) {
    // Get the current plugin's basename (e.g., 'myplugin/myplugin.php')
    $plugin_slug = plugin_basename( __FILE__ );

    // Check if the deactivated plugin matches this plugin
    if ( $plugin === $plugin_slug ) wp_cache_flush_group('media-activity-tracker');
}
add_action( 'deactivated_plugin', 'mediactr_deactivation' );

/**
	 * Check if a table exists in the database with caching
	 *
	 * Uses caching to reduce database calls and improve performance.
	 * Direct database query is used because WordPress does not provide
	 * a native API for checking table existence.
	 *
	 * @param string $table_name The table name (without prefix)
	 * @return bool Whether the table exists
	 */
function mediactr_table_exists($table_name) {
	global $wpdb;

	// Create cache key
    $cache_key = 'table_exists_' . $table_name;
    
    // Try to get from cache
    $exists = wp_cache_get($cache_key, 'media-activity-tracker');
    
    // If not in cache, query and cache the result
	if ($exists === false) {
		$full_table_name = $wpdb->prefix . $table_name;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$exists = ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $full_table_name)) === $full_table_name);

		// Cache the result (for 1 hour)
		wp_cache_set($cache_key, $exists, 'database_checks', 3600);
	}

	return $exists;
}

/********** BACK-END & AJAX FUNCTIONS **********/

if( is_admin() ) {

    require_once('backend.php');

}

/********** FRONT-END FUNCTIONS **********/

require_once('ajax.php');

require_once('frontend.php');