<?php
/**
 * This file is part of Media Activity Tracker by ActivityPress.
 *
 * @license GPL-2.0-or-later
 * @copyright 2025 Activitypress LLC
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

require_once('login.php');

if ( $mediactr_learndash_is_active == 1 ) require_once('learndash.php');

/***** CONTENT ALTERING *****/

/**
 * Determines the activity type for a given post.
 *
 * Checks both global and post-specific settings to determine the appropriate
 * activity type (reading, watching, listening) for CPD tracking.
 *
 * @since 1.0.0
 * @param int $post_id The ID of the post to check
 * @return string|bool The activity type or FALSE if tracking is disabled
 */
function mediactr_get_activity( $post_id ) {

	$current_post_type = get_post_type( $post_id );

	$allowed_post_types = get_option('mediactr_posttypes', array());

	if (!in_array($current_post_type, $allowed_post_types)) {
		return FALSE;
	}
	
	// get the global cpt activity
	$post_type_activities = get_option('mediactr_posttype_activities', array());
	$activity = $post_type_activities[$current_post_type];

	// check the override
	$cpd_activity = get_post_meta( $post_id, 'cpd_activity', true);
	
	if ( $cpd_activity == 'no_tracking' ) return FALSE;
	if ( $cpd_activity != '' && $cpd_activity != $activity ) $activity = $cpd_activity;
	
	return $activity;
}

/**
 * Adds tracking identifiers to media elements in content.
 *
 * Modifies post content to add tracking IDs to video players, audio elements,
 * and reading materials for CPD monitoring.
 *
 * @since 1.0.0
 * @param string $content The post content
 * @return string Modified content with tracking elements
 */
function mediactr_video_autoid($content) {

  // only front-end running
  if( is_admin() ) {
    return;
  }

  $post_id = get_the_ID();
  $cpd_activity = mediactr_get_activity( $post_id );
  if ( $cpd_activity ) {
        if ( $cpd_activity == 'watching' ) {

            $patterns = [
                '/<iframe[^>]*src=["\'](https?:\/\/www\.youtube\.com\/embed\/[^\s"\']+)["\'][^>]*><\/iframe>/is',
                '/<iframe[^>]*src=["\'](https?:\/\/player\.vimeo\.com\/video\/[^\s"\']+)["\'][^>]*><\/iframe>/is'
            ];

            // add ID for figure->iframes
            $content = preg_replace_callback($patterns, 'mediactr_addIframeId', $content);
			
            // add API enabling for youtube
            if (strpos($content, 'youtubeplayer') !== false) $content = preg_replace('/(<iframe id="youtubeplayer[^>]+src="[^"]+)/', '$1' . '&amp;enablejsapi=1', $content);
			
        }
        elseif ( $cpd_activity == 'reading' ) {
			$wpm = 225;
			$text = wp_strip_all_tags($content);
			$time = ceil(str_word_count($text) / $wpm);
			
			$style = ( get_post_meta($post_id, 'cpd_reading_time', true) == 1 ) ? '' : ' class="hide_the_reader"';
			
			$content = '<p'.$style.'><span id="reading_time">'.$time.'</span> minute read</p>'.$content;
            
        }

        //memory indicator
        if ( $cpd_activity == 'watching' || $cpd_activity == 'listening' ){
                $patterns = array(
					'/<video(.*?)<\/video>/i',
					'/<audio(.*?)<\/audio>/i',
					'/.mp3<\/div><\/figure><!-- \/wp:embed -->/i',
					'/<iframe\s+id="youtubeplayer(.*?)(<\/iframe>)/i',
					'/<iframe\s+id="vimeoplayer(.*?)(<\/iframe>)/i'                    
                );
				//'/<div[^>]*class="[^"]*mejs-audio[^"]*"[^>]*>.*?<\/div>/i',

                foreach ($patterns as $pattern) {
                    $content = preg_replace($pattern, '$0<div class="cpd_memory"></div>', $content);
                }
        }       
  }
  return $content;

}
add_action('the_content', 'mediactr_video_autoid', 20);

	/**
	 * Adds iframe IDs to media embeds for tracking.
	 *
	 * Helper function that adds unique identifiers to YouTube and Vimeo
	 * iframes for JavaScript tracking capabilities.
	 *
	 * @since 1.0.0
	 * @param array $matches Regex match data for iframes
	 * @return string Modified iframe HTML with tracking ID
	 */
    function mediactr_addIframeId($matches) {
        $provider = strpos($matches[1], 'vimeo.com') !== false ? 'vimeoplayer' : 'youtubeplayer';
        return preg_replace('/<iframe/', '<iframe id="' . $provider . '"', $matches[0]);
    }

/**
 * Enqueues necessary scripts for media tracking.
 *
 * Loads JavaScript libraries required for tracking media consumption,
 * including specialized scripts for different media types (video, audio, reading).
 * 
 * Source: https://www.urbaninsight.com/article/tracking-progress-embedded-vimeo-videos + https://github.com/vimeo/player.js 
 * 
 * @since 1.0.0
 */
function mediactr_reading_scripts() {

  // only front-end running
  if( is_admin() ) {
    return;
  }

  if (is_user_logged_in()) {	
	  $post_id = get_the_ID();
	  $cpd_activity = mediactr_get_activity( $post_id );
		
	  if ( $cpd_activity ) {  
		global $wpdb;
		$current_user = wp_get_current_user();
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching  
		$checkIfExists = $wpdb->get_row($wpdb->prepare(
			  "SELECT id, progress, lastpercent FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND post_id = %d AND activity_type = %s",
			  $current_user->ID,
			  $post_id,
			  $cpd_activity
		));

		wp_enqueue_style('tracker-style', plugins_url('/assets/css/tracker.css', __FILE__), array(), '1.0', 'all');		  
		  
	    if ( $checkIfExists == NULL || $checkIfExists->progress != '' ) {

	        $tracker_dependencies = array();
	        if ( $cpd_activity == 'reading' ) {
	            wp_enqueue_script( 'mediactr_scrollgress', plugins_url('/assets/js/scrollgress.js', __FILE__), array('jquery'), filemtime(plugin_dir_path(__FILE__) . '/assets/js/scrollgress.js'), true );
	            $tracker_dependencies[] = 'mediactr_scrollgress';
	        }
	        
	        if ( $cpd_activity == 'watching' ) {
	            wp_enqueue_script( 'mediactr_vimeo', 'https://player.vimeo.com/api/player.js', array(), '1.0', true );
	            wp_enqueue_script( 'mediactr_ytapi', 'https://www.youtube.com/iframe_api', array(), '1.0', true );
	            array_push($tracker_dependencies, 'jquery', 'mediactr_ytapi', 'mediactr_vimeo');
	        }	        

	        // encode the progress array to send to the JS
	        $progress = ($checkIfExists && isset($checkIfExists->progress) && $checkIfExists->progress !== '') ? json_encode($checkIfExists->progress) : '';
			$lastpercent = ($checkIfExists && isset($checkIfExists->lastpercent) && $checkIfExists->lastpercent !== '') ? $checkIfExists->lastpercent : '';
	        
			$current_post_type = get_post_type( $post_id );
			
	        // localize
			// for avoid caching
			$filepath = plugin_dir_path(__FILE__) . 'assets/js/tracker.js';
			$file_version = file_exists($filepath) ? filemtime($filepath) : false;
	        wp_register_script( 'mediactr_tracker', plugins_url('/assets/js/tracker.js', __FILE__), $tracker_dependencies, $file_version, true );
	        $translation_array = array(
	          'id' => $post_id,
	          'ajax_url' => admin_url('admin-ajax.php'),
	          'type' => $cpd_activity,
			  'cpt' => $current_post_type,
	          'progress' => $progress,
			  'lastpercent' => $lastpercent,
	          'nonce' => wp_create_nonce('mediactr-tracking')
	        );
	        wp_localize_script( 'mediactr_tracker', 'tracking', $translation_array );
	        wp_enqueue_script( 'mediactr_tracker' );
	        
	    }

	  }
	}
}

add_action('wp_enqueue_scripts', 'mediactr_reading_scripts');


/***** SHORTCODES for VIEWS *****/

/**
 * Calculates consecutive days of user activity for streak tracking.
 *
 * Determines the user's activity streak by analyzing daily CPD completion records.
 * Can return both the streak count and an array of streak data for visualization.
 *
 * @since 1.0.0
 * @param int $user_id The ID of the user
 * @param int $mode Mode 1 returns array data, mode 0 returns only count
 * @return array|int Array of streak data or streak count depending on mode
 */
function mediactr_front_streak_calc($user, $mode) {
	global $wpdb;
	
	$streakLevel = get_option('mediactr_streaklevel', 1200);
	$dayStreakInRow = 0;
	$dayStreakInRowCounter = 0;
	$yesterday = new DateTime('yesterday');
	$today = new DateTime('today');
	$i = 0;
	$prev = '';
	
	$streakDays = array();
	$previousMonthLastDay = new DateTime('last day of previous month');
	$dateSixDaysBefore = $previousMonthLastDay->sub(new DateInterval('P5D'))->format('Y-m-d');
	
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$results = $wpdb->get_results(
	   $wpdb->prepare(
		   "SELECT 
		   CASE WHEN finished_at IS NOT NULL AND finished_at != '' THEN DATE(finished_at) 
		   ELSE DATE(created_at) 
		   END AS day, 
		   SUM(points) AS daily_points 
		   FROM " . $wpdb->prefix . "gp_tracking_points
		   WHERE user_id = %s 
		   GROUP BY 
		   CASE WHEN finished_at IS NOT NULL AND finished_at != '' THEN DATE(finished_at)
		   ELSE DATE(created_at)
		   END 
		   ORDER BY day DESC",
		   $user
	   )
	); // AND activity_type != 'login' - if we need without login
	if ($results) {
		foreach ($results as $result) {
			
			// if the first line is older than yesterday streak is failed
			if ( $i == 0 ) {
				$firstDayInQuery = new DateTime($result->day);
				if ( $firstDayInQuery < $yesterday ) $dayStreakInRowCounter = 1;
				elseif ( $today == $firstDayInQuery && $result->daily_points >= $streakLevel ) $dayStreakInRow++; // if the actual day is already streaked, than +1
			}
			
			// if the streak is broken than it is failed
			if ( $prev != '' ) {
				if ( $dayStreakInRowCounter == 0 ) {
					$actualDayInQuery = new DateTime($result->day);
					$prevDayInQuery = new DateTime($prev);
					if ( $actualDayInQuery->diff($prevDayInQuery)->days === 1  ) {
						if ( $result->daily_points >= $streakLevel ) $dayStreakInRow++;
						else $dayStreakInRowCounter = 1;
					}
					else $dayStreakInRowCounter = 1;
				}
			}
			
			// fill the array with stars
			if ( $mode == 1 ) {
				if ( $dateSixDaysBefore <= $result->day ) {
					if ( $result->daily_points >= $streakLevel ) $star = 5;
					elseif ( $result->daily_points > 0 ) $star = 3;
					else $star = 1;
					$streakDays[] = array(intval(date_i18n('d', strtotime($result->day))), intval(date_i18n('n', strtotime($result->day))), $star);
				}
			}
			
			// break the loop
			if ( $dateSixDaysBefore > $result->day && $dayStreakInRowCounter == 1 ) break;
			
			$prev = $result->day;
			$i = 1;
		}
	}

    if ( $mode == 1 ) return [$streakDays, $dayStreakInRow];
	else return $dayStreakInRow;
}

/**
 * SHORTCODE: Renders the daily streak calendar shortcode output.
 *
 * Creates a visual calendar showing the user's activity streak and consistency
 * with interactive month selection capabilities.
 *
 * @since 1.0.0
 * @param array $atts Shortcode attributes
 * @return string HTML output for streak calendar
 */
function mediactr_front_dailystreak($atts) {

	if (is_user_logged_in()) {	
		// get the first activity date for the Select field
		$user = get_current_user_id();
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$result = $wpdb->get_var($wpdb->prepare("SELECT created_at FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d ORDER BY created_at ASC LIMIT 1", $user));
		
		$lastDayOfCurrentMonth = strtotime('last day of this month', time());
		$lastSecondOfCurrentMonth = strtotime('23:59:59', $lastDayOfCurrentMonth);	

		$firstActivity = ( $result == NULL ) ? $lastSecondOfCurrentMonth : strtotime($result);
		
		// get the dailystreak before today + stars
		list($streakDays, $dayStreakInRow) = mediactr_front_streak_calc($user, 1);

		$output = '<div class="calendar-container">
	<h3>' .__( 'Day streak statistics', 'media-activity-tracker' ). '</h3>	
	  <div class="calendar-header">
	    <h4>
	      <span class="icon">'.$dayStreakInRow.'</span> ' .__( 'Day streak!', 'media-activity-tracker' ). '
	    </h4>
		<select id="actualMonth">';

		// generate options
		while ($firstActivity <= $lastSecondOfCurrentMonth) {
			$year = date_i18n('Y', $firstActivity);
			$month = date_i18n('m', $firstActivity);
			$month_latin = date_i18n('F', $firstActivity);
		    $isCurrentMonth = $firstActivity >= strtotime('first day of this month', $lastSecondOfCurrentMonth) && 
	                      $firstActivity <= strtotime('last day of this month', $lastSecondOfCurrentMonth);

			$selected = ( $isCurrentMonth ) ? ' selected' : '';
			$output.= '<option value="'.$month.'" data-year="'.$year.'"'.$selected.'>'.$month_latin.' '.$year.'</option>';

			// Move to the next month
			$firstActivity = strtotime('+1 month', $firstActivity);
		}	
	     $output.= '</select>
	  </div>
	  <div class="calendar" id="calendar"></div>
	</div>';
		
		wp_enqueue_style('tracker-style', plugins_url('/assets/css/streakday.css', __FILE__), array(), '1.0', 'all');
		// localize
		wp_register_script( 'mediactr_streakcalendar', plugins_url('/assets/js/streakday.js', __FILE__), array('jquery'), '1.0', true );
		$translation_array = array(
			'ajax_url' => admin_url('admin-ajax.php'),
			'stars' => json_encode($streakDays),
			'nonce' => wp_create_nonce('mediactr-streak')
		);
		wp_localize_script( 'mediactr_streakcalendar', 'streakday', $translation_array );
		wp_enqueue_script( 'mediactr_streakcalendar' );	
		
	    return $output; 
	}
}
add_shortcode('mediactr_dailystreak', 'mediactr_front_dailystreak');

/**
 * SHORTCODE: Renders the CPD hours graph shortcode output.
 *
 * Creates a visual chart showing the breakdown of CPD hours by activity type
 * with percentage distribution.
 *
 * @since 1.0.0
 * @param array $atts Shortcode attributes
 * @return string HTML output for CPD graph
 */
function mediactr_front_graph($atts) {

	if (is_user_logged_in()) {	
		$output = '<div class="chartjs-container">
			<div class="chartjs-header">
				<h3>' .__( 'CPD Hours', 'media-activity-tracker' ). '</h3>
			</div>
			<canvas id="barChart"></canvas>
			<ul>';	
		
	    global $wpdb;
	    $user_id = get_current_user_id();
		$stat = array();

		// If login is disabled ban this from the calculation
		if ( get_option('mediactr_login_points') == 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$lifeTimePoints = $wpdb->get_var($wpdb->prepare("SELECT SUM(points) FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND activity_type != %s", $user_id, 'login'));
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$results = $wpdb->get_results(
				$wpdb->prepare("SELECT activity_type, SUM(points) AS full_points FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND activity_type != %s GROUP BY activity_type", $user_id, 'login'));
			
		} else {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$lifeTimePoints = $wpdb->get_var($wpdb->prepare("SELECT SUM(points) FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d", $user_id));
			
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$results = $wpdb->get_results($wpdb->prepare("SELECT activity_type, SUM(points) AS full_points FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d GROUP BY activity_type", $user_id));
		}
		
		if ($results) {
			foreach ($results as $result) {		
				switch ($result->activity_type) {
					case "reading":
						$text = __( 'Reading', 'media-activity-tracker' );
						break;
					case "watching":
						$text = __( 'Watching', 'media-activity-tracker' );
						break;
					case "listening":
						$text = __( 'Podcasts', 'media-activity-tracker' );
						break;
					case "completion":
						$text = __( 'Completing Courses', 'media-activity-tracker' );
						break;
					case "login":
						$text = __( 'Daily Logins', 'media-activity-tracker' );
						break;
					default:
						break;
				}
				
				$output.= '<li class="'.$result->activity_type.'"><strong>'.round($result->full_points / 3600, 2).' hours</strong> '.$text.'</li>';

				$activityPercentage = round($result->full_points / $lifeTimePoints * 100, 2);			
				
				$stat[] = array($result->activity_type, $activityPercentage);
			}
			$stat = json_encode($stat);
		}
		else {
			$stat = '';
			$output = str_replace('<canvas id="barChart"></canvas>', '<p>'.__( 'There is no any data to show. Start to learn on our portal and come back daily to check your progress.', 'media-activity-tracker' ).'</p>', $output);
		}
		
		$output.='</ul>
		</div>';	
		
		wp_enqueue_script( 'mediactr_chart', plugins_url('/assets/js/chart.js', __FILE__), array(), '4.4', true );
		wp_enqueue_script( 'mediactr_chartlabel', plugins_url('/assets/js/chartjs-plugin-datalabels.js', __FILE__), array( 'mediactr_chart' ), '2.2', true );
		wp_enqueue_style('chart-style', plugins_url('/assets/css/chartjs.css', __FILE__), array(), '1.0', 'all');
		
		// localize 
	    wp_register_script('mediactr_horizontalbar', plugins_url('/assets/js/horizontalbar.js', __FILE__), array( 'mediactr_chart', 'mediactr_chartlabel' ), '1.0', true);
	    $translation_array = array(
		  'stat' => $stat
	    );
	    wp_localize_script( 'mediactr_horizontalbar', 'chartdata', $translation_array );	  
		wp_enqueue_script( 'mediactr_horizontalbar' );	
		
		return $output;
	}
}

add_shortcode('mediactr_graph', 'mediactr_front_graph');

/**
 * SHORTCODE: Renders the activity log table shortcode output.
 *
 * Creates an interactive table showing the user's CPD activities with
 * filtering, reflection capabilities, and export options.
 * 
 * Source: https://github.com/DataTables/DataTablesSrc
 * 
 * @since 1.0.0
 * @param array $atts Shortcode attributes
 * @return string HTML output for activity log table
 */
function mediactr_front_log($atts) {
	if (is_user_logged_in()) {	
		$output = '<div class="datatable-container">
			<h3>'.__( 'Activity log', 'media-activity-tracker' ).'</h3>
			<div class="datatable-header">
			<label for="start_date">'.__( 'Date:', 'media-activity-tracker' ).'</label>
			<input type="date" id="date" />
			<label for="activity">'.__( 'Activity:', 'media-activity-tracker' ).'</label>
			<select id="activity">
				<option value="">'.__( 'All CPD activities', 'media-activity-tracker' ).'</option>
				<option value="completion">'.__( 'Course', 'media-activity-tracker' ).'</option>
				<option value="reading">'.__( 'Article', 'media-activity-tracker' ).'</option>
				<option value="listening">'.__( 'Podcast', 'media-activity-tracker' ).'</option>
				<option value="watching">'.__( 'Watching', 'media-activity-tracker' ).'</option>
				<option value="login">'.__( 'Login', 'media-activity-tracker' ).'</option>
			</select>
			<button type="button" id="exportCSV"><i class="_mi _before bb-icon-l bb-icon-file-download" aria-hidden="true"></i> '.__( 'Download', 'media-activity-tracker' ).'</button>
			</div>
		<table id="datatable" class="display" style="width:100%">
	    <thead>
	      <tr>
	        <th>'.__( 'Date', 'media-activity-tracker' ).'</th>
	        <th>'.__( 'CPD Type', 'media-activity-tracker' ).'</th>
	        <th>'.__( 'CPD Details', 'media-activity-tracker' ).'</th>
	        <th>'.__( 'CPD Hours', 'media-activity-tracker' ).'</th>
	        <th>'.__( 'Reflection', 'media-activity-tracker' ).'</th>
	      </tr>
	    </thead>
	    <tbody>
	    </tbody>
	  </table>';

		// modal code
		$output.= '<div id="reflection" class="modal">
	  <div class="modal-content">
	    <h3></h3>
		<label for="note">'.__( 'Your comment:', 'media-activity-tracker' ).'</label>
	    <textarea id="note"></textarea>
		<span class="msg"></span>
		<div class="modal-footer">
			<button type="button" class="cancel">'.__( 'Cancel', 'media-activity-tracker' ).'</button>
			<button type="button" class="save">'.__( 'Save', 'media-activity-tracker' ).'</button>
		</div>
	  </div>
	</div>
	</div>';
		
		global $wpdb;
	    $user_id = get_current_user_id();
		$rows = array();

		// If login is disabled ban this from the calculation
		if ( get_option('mediactr_login_points') == 0 ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT id, 
        CASE 
            WHEN finished_at IS NOT NULL AND finished_at != '' THEN DATE_FORMAT(finished_at, '%%d.%%m.%%y')
            ELSE DATE_FORMAT(created_at, '%%d.%%m.%%y')
        END AS f_date,
        activity_type, post_id, course_id, lesson_id, topic_id, points, reflection
        FROM " . $wpdb->prefix . "gp_tracking_points
        WHERE user_id = %d AND activity_type != %s", $user_id, 'login'));
		} else {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$results = $wpdb->get_results($wpdb->prepare(
				"SELECT id, 
        CASE 
            WHEN finished_at IS NOT NULL AND finished_at != '' THEN DATE_FORMAT(finished_at, '%%d.%%m.%%y')
            ELSE DATE_FORMAT(created_at, '%%d.%%m.%%y')
        END AS f_date,
        activity_type, post_id, course_id, lesson_id, topic_id, points, reflection
        FROM " . $wpdb->prefix . "gp_tracking_points
        WHERE user_id = %d", $user_id));
		}	
		
		if ($results) {
			foreach ($results as $result) {
				
				// build the name of the content
				if ( $result->topic_id != 0 ) $title = get_the_title($result->topic_id);
				elseif ( $result->lesson_id != 0 ) $title = get_the_title($result->lesson_id);
				elseif ( $result->course_id != 0 ) $title = get_the_title($result->course_id);
				elseif ( $result->post_id != 0 ) $title = get_the_title($result->post_id);
				else $title = "Daily Login";
				
				// build the reflection link
				$reflection = '<a class="reflect" href="#" data-rid="'.$result->id.'" data-title="'.$title.'" data-note="';
				$reflection.= ( $result->reflection != '' ) ? $result->reflection.'"><i class="_mi _before bb-icon-l bb-icon-pencil" aria-hidden="true"></i> Edit' : '"><i class="_mi _before bb-icon-l bb-icon-pencil" aria-hidden="true"></i> Add new';
				$reflection.= '</a>';
				
				switch ($result->activity_type) {
					case 'completion':
						$act_renamed = 'Course';
						break;
					case 'reading':
						$act_renamed = 'Article';
						break;
					case 'listening':
						$act_renamed = 'Podcast';
						break;
					case 'watching':
						$act_renamed = 'Watching';
						break;
					case 'login':
						$act_renamed = 'Daily Login';
						break;						
					}

				$rows[] = array($result->f_date, $act_renamed, $title, round($result->points / 3600, 2), $reflection);
			}
			$rows = json_encode($rows);
		}
		else $rows = '';

		wp_enqueue_script( 'mediactr_datatables', plugins_url('/assets/js/jquery.dataTables.min.js', __FILE__), array( 'jquery' ), '1.10.24', true );
		wp_enqueue_style('table-style', plugins_url('/assets/css/tabledata.css', __FILE__), array(), '1.0', 'all');
		
		// localize 
	    wp_register_script('mediactr_tabledata', plugins_url('/assets/js/tabledata.js', __FILE__), array( 'mediactr_datatables' ), '1.0', true);
	    $translation_array = array(
	      'ajax_url' => admin_url('admin-ajax.php'),
		  'rows' => $rows,
		  'nonce' => wp_create_nonce('mediactr-table')
	    );
	    wp_localize_script( 'mediactr_tabledata', 'mediactr_tabledata', $translation_array );	  
		wp_enqueue_script( 'mediactr_tabledata' );
		
	    return $output;
	}
}
add_shortcode('mediactr_log', 'mediactr_front_log');

?>