<?php
/**
 * This file is part of Media Activity Tracker by ActivityPress.
 *
 * @license GPL-2.0-or-later
 * @copyright 2025 Activitypress LLC
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Adds the CPD menu and submenu items to the WordPress admin panel.
 *
 * Creates the main CPD menu and relevant submenu pages based on whether LearnDash is active.
 * Different menu structures are displayed depending on the learning management system integration.
 *
 * @since 1.0.0
 */
function mediactr_add_cpd_menu()
{
    global $mediactr_learndash_is_active;
       
    // Add CPD main menu
    add_menu_page(
        'CPD Activity Log',
        'CPD',
        'manage_options',
        'mediactr-activity-log',
        'mediactr_display_activity_log_table',
        'dashicons-awards'
    );

    // Add CPD Settings submenu
    add_submenu_page(
        'mediactr-activity-log',
        'CPD Settings',
        'Settings',
        'manage_options',
        'mediactr-settings',
        'mediactr_settings_page'
    );

}
add_action('admin_menu', 'mediactr_add_cpd_menu');

/**
 * Handles user record revocation in the admin interface.
 *
 * Processes requests to remove all activity records for a specific user when the
 * revoke action is confirmed through the admin interface.
 *
 * @since 1.0.0
 */
function mediactr_revoke_user_records()
{
    global $wpdb;

    // Handle the 'revoke_confirm' action
    if (
        isset($_GET['action'], $_GET['user_id'], $_GET['nonce_revoke']) &&
        $_GET['action'] === 'revoke_confirm'
    ) {
        $user_id = intval($_GET['user_id']);
        $nonce = sanitize_text_field(wp_unslash($_GET['nonce_revoke']));

        if (wp_verify_nonce($nonce, 'mediactr_revoke_user_' . $user_id)) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->delete($wpdb->prefix . 'gp_tracking_points', ['user_id' => $user_id]);

            // Add admin notice for feedback
            add_action('admin_notices', function () {
                echo '<div class="notice notice-success is-dismissible"><p>' .
                    esc_html__('All of this user\'s records have been successfully revoked.', 'media-activity-tracker') .
                    '</p></div>';
            });
        }
    }
}
add_action('admin_init', 'mediactr_revoke_user_records');

/**
* Enqueues admin styles and scripts for CPD pages only.
*
* Loads the admin CSS and JavaScript files specifically for CPD-related
* admin pages (activity log, settings) to avoid unnecessary loading on
* other WordPress admin pages.
*
* @since 1.0.0
* @param string $hook The current admin page hook suffix.
*/
function mediactr_enqueue_admin_assets($hook) {
    wp_enqueue_style(
       'mediactr_admin_styles', 
       plugins_url('assets/css/admin.css', __FILE__), 
       array(), 
       '1.0.0'
    );

    wp_enqueue_script(
       'mediactr_admin_scripts', 
       plugins_url('assets/js/admin.js', __FILE__), 
       array('jquery'), 
       '1.0.0', 
       true
    );
}
add_action('admin_enqueue_scripts', 'mediactr_enqueue_admin_assets');

/**
 * Displays the activity log table in the admin interface.
 *
 * Renders a table showing all CPD activities across users with pagination support.
 * Includes functionality for filtering, sorting, and deleting records.
 *
 * @since 1.0.0
 */
function mediactr_display_activity_log_table()
{
    global $wpdb, $mediactr_learndash_is_active;

    // Check if the table exists
	if (!mediactr_table_exists('gp_tracking_points')) {
		echo '<div class="wrap"><h1>Activity Log</h1><p>No records found.</p></div>';
		return;
	}	

    // Handle record deletion
	if (
		isset($_GET['action'], $_GET['id'], $_GET['nonce_delete']) &&
		$_GET['action'] === 'delete'
	) {
		$id = intval($_GET['id']);
		$nonce = sanitize_text_field(wp_unslash($_GET['nonce_delete']));

		if (wp_verify_nonce($nonce, 'mediactr_delete_' . $id)) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->delete($wpdb->prefix . 'gp_tracking_points', ['id' => $id]);
			
            // Add admin notice for feedback
            add_action('admin_notices', function () {
                echo '<div class="notice notice-success is-dismissible"><p>' .
                    esc_html__('Once user record have been successfully revoked.', 'media-activity-tracker') .
                    '</p></div>';
            });			
		}
	}

    // Set the number of records to display per page
    $records_per_page = 20;

    // Get the current page number
    $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;

    // Calculate the offset for the SQL query
    $offset = ($current_page - 1) * $records_per_page;

    // Fetch records with pagination
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$records = $wpdb->get_results($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "gp_tracking_points ORDER BY created_at DESC LIMIT %d OFFSET %d", $records_per_page, $offset), ARRAY_A);	

    // Fetch total number of records
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $total_records = $wpdb->get_var("SELECT COUNT(*) FROM " . $wpdb->prefix . "gp_tracking_points");

    // Calculate total number of pages
    $total_pages = ceil($total_records / $records_per_page);

    // Display table
    echo '<div class="wrap">';
    echo '<h1>Activity Log</h1>';
    echo '<table class="wp-list-table widefat fixed striped">';
    echo '<thead><tr><th>User</th><th>Post</th>';
    if ( $mediactr_learndash_is_active == 1 ) echo '<th>Course</th><th>Lesson</th><th>Topic</th>';
    echo '<th>Activity Type</th><th>Hours</th><th>Created At</th><th>Finished At</th><th>Action</th></tr></thead>';
    echo '<tbody>';

    // Iterate records and display them
    foreach ($records as $record) { 
        echo '<tr>';
    
        $user = get_userdata($record['user_id']);
        if ($user && is_a($user, 'WP_User')) $user_login = $user->user_login;   
        else $user_login = 'deleted user';
    
        echo '<td>' .
            esc_html( $user_login ) .
            ' <a href="' .
            esc_html( admin_url('user-edit.php?user_id=' . $record['user_id']) ) .
            '">(' .
            esc_html( $record['user_id'] ) .
            ')</a></td>';
        echo '<td>' .
            (!empty($record['post_id'])
                ? esc_html( get_the_title($record['post_id']) ) .
                    ' <a href="' .
                    esc_html( get_permalink($record['post_id']) ) .
                    '">(' .
                    esc_html( $record['post_id'] ) .
                    ')</a>'
                : '') .
            '</td>';
        if ( $mediactr_learndash_is_active == 1 ) echo '<td>' .
            (!empty($record['course_id'])
                ? esc_html( get_the_title($record['course_id']) ) .
                    ' <a href="' .
                    esc_html( get_permalink($record['course_id']) ) .
                    '">(' .
                    esc_html( $record['course_id'] ) .
                    ')</a>'
                : '') .
            '</td>';
        if ( $mediactr_learndash_is_active == 1 ) echo '<td>' .
            (!empty($record['lesson_id'])
                ? esc_html( get_the_title($record['lesson_id']) ) .
                    ' <a href="' .
                    esc_html( get_permalink($record['lesson_id']) ) .
                    '">(' .
                    esc_html( $record['lesson_id'] ) .
                    ')</a>'
                : '') .
            '</td>';
        if ( $mediactr_learndash_is_active == 1 ) echo '<td>' .
            (!empty($record['topic_id'])
                ? esc_html( get_the_title($record['topic_id']) ) .
                    ' <a href="' .
                    esc_html( get_permalink($record['topic_id']) ) .
                    '">(' .
                    esc_html( $record['topic_id'] ) .
                    ')</a>'
                : '') .
            '</td>';
        echo '<td>' . esc_html( $record['activity_type'] ) . '</td>';
        echo '<td>' .
            esc_html( round($record['points'] / 3600, 2) ) .
            (!empty($record['progress']) ? ' (in progress)' : '') .
            '</td>';
        echo '<td>' . esc_html( $record['created_at'] ) . '</td>';
        echo '<td>' . esc_html( $record['finished_at'] ) . '</td>';
		
		$delete_url = add_query_arg([
			'page' => 'mediactr-activity-log',
			'action' => 'delete',
			'id' => $record['id'],
			'nonce_delete' => wp_create_nonce('mediactr_delete_' . $record['id'])
		], admin_url('admin.php?page=mediactr-activity-log'));
		
        echo '<td><a href="' . esc_url($delete_url) . '">Delete</a> | ';

		$revoke_url = add_query_arg([
			'page' => 'mediactr-activity-log',
			'action' => 'revoke_confirm',
			'user_id' => $record['user_id'],
			'nonce_revoke' => wp_create_nonce('mediactr_revoke_user_' . $record['user_id'])
		], admin_url('admin.php?page=mediactr-activity-log'));
			
        echo '<a class="revoker" href="' . esc_url($revoke_url) . '">' . esc_html__('Revoke user', 'media-activity-tracker') . '</a></td>';
        echo '</tr>';
    }

    echo '</tbody>';
    echo '</table>';

    // Display pagination
    echo '<div class="tablenav">';
    echo '<div class="tablenav-pages">';
    echo esc_html( paginate_links(
        array(
            'base'      => add_query_arg('paged', '%#%'),
            'format'    => '',
            'prev_text' => __('&laquo;','media-activity-tracker'),
            'next_text' => __('&raquo;','media-activity-tracker'),
            'total'     => $total_pages,
            'current'   => $current_page,
        )
    ) );
    echo '</div>';
    echo '</div>';
    
    echo '</div>';
    
    echo '<div id="id_confrmdiv"><p>Are you sure you want to delete all the records for this user?</p>
    <button id="id_truebtn">Yes</button>
    <button id="id_falsebtn">No</button>
</div>'; 
}

/**
 * Renders the CPD settings page in the admin area.
 *
 * Creates the interface for configuring plugin settings including license key,
 * media thresholds, post type activity associations, and badge images.
 *
 * @since 1.0.0
 */
function mediactr_settings_page() {
    // Save
	$nonce = isset($_POST['cpd_settings_nonce']) ? sanitize_text_field(wp_unslash($_POST['cpd_settings_nonce'])) : '';
	if (wp_verify_nonce($nonce, 'mediactr_save_settings')) {
        $threshold = isset($_POST['mediactr_treshold']) ? absint($_POST['mediactr_treshold']) : 0;
        update_option('mediactr_treshold', $threshold);

		$posttypes = isset($_POST['mediactr_posttypes']) ? array_map('sanitize_text_field', wp_unslash($_POST['mediactr_posttypes'])) : [];
        update_option('mediactr_posttypes', $posttypes);

        $activities = [];
        foreach ($posttypes as $pt) {
            $key = $pt . '_activity';
            if (isset($_POST[$key])) {
                $activities[$pt] = sanitize_text_field(wp_unslash($_POST[$key]));
            }
        }
        update_option('mediactr_posttype_activities', $activities);

        $login_points = isset($_POST['mediactr_login_points']) ? floatval($_POST['mediactr_login_points']) * 60 : 0;
        update_option('mediactr_login_points', $login_points);

        $streak = isset($_POST['mediactr_streaklevel']) ? floatval($_POST['mediactr_streaklevel']) * 60 : 0;
        update_option('mediactr_streaklevel', $streak);

		echo '<div class="updated notice"><p><strong>Settings saved.</strong></p></div>';
    }

    // Load variables
    $threshold = get_option('mediactr_treshold', 0);
    $selected_post_types = get_option('mediactr_posttypes', []);
    $selected_post_type_activities = get_option('mediactr_posttype_activities', []);
    $login_points = get_option('mediactr_login_points', 0) / 60;
    $streak = get_option('mediactr_streaklevel', 0) / 60;

    $post_types = get_post_types(['public' => true], 'objects');
    $not_allowed_post_types = ['attachment', 'revision', 'sfwd-quiz', 'sfwd-question'];

    ?>
    <div class="wrap">
        <h1>CPD Settings</h1>
        <form method="post">
            <?php wp_nonce_field('mediactr_save_settings', 'cpd_settings_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="mediactr_treshold">Media Threshold (%)</label></th>
                    <td><input type="number" min="80" max="99" step="1" id="mediactr_treshold" name="mediactr_treshold" value="<?php echo esc_attr(
                        $threshold,
                    ); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row">Active post types (the default activities able to individually overridden)</th>
                    <td id="cpt_list"><dl>
                        <?php foreach (
                            $post_types
                            as $post_type => $post_type_obj
                        ): ?>
							<?php if ( !in_array($post_type, ['post', 'page']) ) continue; ?>
							<?php //if ( ( $serial == 'invalid' || $serial == '' ) && $post_type != 'page' && $post_type != 'post' ) continue; ?>
                            <?php if ( !in_array($post_type, $not_allowed_post_types) && strpos($post_type, 'elementor') !== 0 ): ?>
                                <dd>
                                    <label>
                                        <input type="checkbox" name="mediactr_posttypes[]" value="<?php echo esc_attr(
                                            $post_type,
                                        ); ?>" <?php checked(
        in_array($post_type, $selected_post_types),
    ); ?>>
                                        <?php echo esc_html(
                                            $post_type_obj->labels->name,
                                        ); ?>
                                    </label>
                                </dd>
                                <dt>
                                    <?php if ( $post_type == 'sfwd-courses' ):
                                        echo 'Completion';
                                    else : ?>
                                    <label>Default activity:
                                        <select name="<?php echo esc_html( $post_type ); ?>_activity"'; <?php disabled( !in_array($post_type, $selected_post_types) ); 
                                                    ?>>
                                            <option value="reading" <?php if (isset($selected_post_type_activities[$post_type])) selected( $selected_post_type_activities[$post_type] == 'reading' ); ?>
                                                >Reading</option>
                                            <option value="watching" <?php if (isset($selected_post_type_activities[$post_type])) selected( $selected_post_type_activities[$post_type] == 'watching' ); ?>
                                                >Watching</option>
                                            <option value="listening"  <?php if (isset($selected_post_type_activities[$post_type])) selected( $selected_post_type_activities[$post_type] == 'listening' ); ?>
                                                >Listening</option>
                                        </select>
                                    </label>
                                </dt>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        </dl></td>
                </tr>
                <tr>
                    <th scope="row"><label for="mediactr_login_points">Daily CPD for login (in minutes | 0 = disable)</label></th>
                    <td><input type="number" id="mediactr_login_points" min="0" step="1" name="mediactr_login_points" value="<?php echo esc_attr(
         get_option('mediactr_login_points', 0) / 60,
     ); ?>" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="mediactr_streaklevel">Daily Streak Level</label></th>
                    <td><input type="number" id="mediactr_streaklevel" min="1" step="1" name="mediactr_streaklevel" value="<?php echo esc_attr(
         get_option('mediactr_streaklevel', 0) / 60,
     ); ?>" /></td>
                </tr>           
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * Sanitizes the post types array for CPD settings.
 *
 * Ensures that the input is a valid array of sanitized post type slugs.
 * This helps prevent any invalid or malicious data from being saved.
 *
 * @since 1.0.0
 * @param array $input The array of selected post types from the settings form.
 * @return array Sanitized array of post type slugs.
 */
function mediactr_sanitize_posttypes( $input ) {
    // Make sure we get an array of sanitized post type slugs
    if ( ! is_array( $input ) ) {
        return [];
    }

    return array_map( 'sanitize_key', $input );
}

/**
 * Enqueues the WordPress media uploader scripts.
 *
 * Makes the WordPress media library available for selecting badge images
 * in the CPD settings page.
 *
 * @since 1.0.0
 */
function mediactr_enqueue_media_uploader() {
    wp_enqueue_media();
}

add_action('admin_enqueue_scripts', 'mediactr_enqueue_media_uploader');

/**
 * Converts minutes to seconds for CPD point calculations.
 *
 * Helper function that transforms time values from minutes to seconds for
 * consistent storage of time-based CPD values in the database.
 *
 * @since 1.0.0
 * @param int $value Time value in minutes
 * @return int Time value in seconds
 */
function mediactr_points_min_to_sec($value)
{
    return $value * 60;
}

/**
 * Cleans up user data when a user is deleted.
 *
 * Removes all CPD tracking records and activity logs associated with a user
 * when that user is deleted from the WordPress system.
 *
 * @since 1.0.0
 * @param int $user_id The ID of the deleted user
 */
function mediactr_after_user_deleted($user_id) {
	global $wpdb;
	
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->delete("{$wpdb->prefix}gp_activity", ['user_id' => $user_id], ['%d']); 
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->delete("{$wpdb->prefix}gp_tracking_points", ['user_id' => $user_id], ['%d']);

}

add_action('delete_user', 'mediactr_after_user_deleted');

/***** ONLY WITH LEARNDASH: extend the certificate system *****/

if ( $mediactr_learndash_is_active == 1 ) { // if Learndash is active

    /**
     * Renders the certificate meta box fields.
     *
     * Displays the UI for configuring certificate settings including thresholds
     * and trigger activities when LearnDash is active.
     *
     * @since 1.0.0
     * @param WP_Post $post The current post object
     */
    function mediactr_certificate_callback($post) {

        // Query Learndash Certificates
        $certificate_args = array(
            'post_type' => 'sfwd-certificates', // Replace with the actual post type of Learndash Certificates
            'posts_per_page' => -1,
        );

        $certificate_query = new WP_Query($certificate_args);

        // Get the currently saved value (if any)
        $selected_certificate = get_post_meta($post->ID, '_gp_ld_cert_id', true);       
        
        // Define the list of options for your select dropdown
        $certificate_options = array(
            'overall' => 'Overall time',
            'streak' => 'Streak days',
            'reading' => 'Category time: reading ',
            'watching' => 'Category time: watching',
            'listening' => 'Category time: listening',
            'completion' => 'Category time: completion',
            /*'writing' => 'Category time: writing',
            'login' => 'Category time: login',*/
        );

        // Get the currently saved value (if any)
        $selected_option = get_post_meta($post->ID, '_gp_ld_cert_activity', true);      
        $threshold = ( $selected_option != 'streak' ) ? intval( get_post_meta($post->ID, '_gp_ld_cert_threshold', true) ) / 60 : intval( get_post_meta($post->ID, '_gp_ld_cert_threshold', true) );

        // certificate select
        echo '<div class="field-row">';
        echo '<label for="gp_ld_cert_id">Select Learndash Certificate<sup>*</sup>:</label> ';
        echo '<select id="gp_ld_cert_id" name="gp_ld_cert_id" required>';
        echo '<option value="">Select a Certificate</option>';

        if ($certificate_query->have_posts()) {
            while ($certificate_query->have_posts()) {
                $certificate_query->the_post();
                $certificate_id = get_the_ID();
                $certificate_title = get_the_title();
                $selected = selected($selected_certificate, $certificate_id, false);

                echo '<option value="' . esc_attr($certificate_id) . '" ' . esc_html( $selected ). '>' . esc_html($certificate_title) . '</option>';
            }
        }

        echo '</select>';
        echo '</div>';
        wp_reset_postdata(); // Restore the global post object.     
        
        // activity select
        echo '<div class="field-row">';
        echo '<label for="gp_ld_cert_activity">Trigger Activity<sup>*</sup>:</label> ';
        echo '<select id="gp_ld_cert_activity" name="gp_ld_cert_activity" required>';
        echo '<option value="">Select an Activity</option>';
        foreach ($certificate_options as $value => $label) {
            echo '<option value="' . esc_attr($value) . '" ' . selected($selected_option, $value, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '</div>';
        
        // threshold
        echo '<div class="field-row">';
        echo '<label for="gp_ld_cert_threshold">Threshold (in min)<sup>*</sup>:</label> ';
        echo '<input type="number" min="0" id="gp_ld_cert_threshold" name="gp_ld_cert_threshold" value="' . esc_attr($threshold) . '" required />';
        echo '</div>';
    }

	/**
	 * Saves certificate meta data when a certificate post is saved.
	 *
	 * Processes and stores the custom meta fields for certificates including
	 * thresholds and associated activities.
	 *
	 * @since 1.0.0
	 * @param int $post_id The ID of the post being saved
	 * @return int The post ID
	 */
	function mediactr_save_certificate_meta($post_id) {
		// Skip if doing autosave
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return $post_id;
		}

		// Verify nonce for security
		if (!isset($_POST['cpd_nonce'])) {
			return $post_id;
		}

		$nonce = sanitize_text_field(wp_unslash($_POST['cpd_nonce']));
		if (!wp_verify_nonce($nonce, 'mediactr_save_settings')) {
			return $post_id;
		}

		// Sanitize and process gp_ld_cert_threshold field
		if (isset($_POST['gp_ld_cert_threshold'])) {
			$threshold = sanitize_text_field(wp_unslash($_POST['gp_ld_cert_threshold']));
			$threshold = intval($threshold); // Ensure it is an integer

			if (isset($_POST['gp_ld_cert_activity'])) {
				$activity = sanitize_text_field(wp_unslash($_POST['gp_ld_cert_activity']));

				if ($activity == 'streak') {
					update_post_meta($post_id, '_gp_ld_cert_threshold', $threshold);
				} else {
					update_post_meta($post_id, '_gp_ld_cert_threshold', $threshold * 60);
				}
			}
		}

		// Sanitize and process gp_ld_cert_activity field
		if (isset($_POST['gp_ld_cert_activity'])) {
			$activity = sanitize_text_field(wp_unslash($_POST['gp_ld_cert_activity']));
			update_post_meta($post_id, '_gp_ld_cert_activity', $activity);
		}

		// Sanitize and process gp_ld_cert_id field
		if (isset($_POST['gp_ld_cert_id'])) {
			$cert_id = sanitize_text_field(wp_unslash($_POST['gp_ld_cert_id']));
			update_post_meta($post_id, '_gp_ld_cert_id', $cert_id);
		}

		return $post_id;
	}
	add_action('save_post', 'mediactr_save_certificate_meta');

}	
	
	/***** Add custom fields to the exact post types *****/

	/**
	 * Adds CPD points fields to relevant post types.
	 *
	 * Attaches custom meta boxes to enabled post types for configuring
	 * CPD tracking settings per content item.
	 *
	 * @since 1.0.0
	 */
	function mediactr_cpd_points_field() {
		$enabled_post_types = get_option('mediactr_posttypes', array());
		
		foreach ($enabled_post_types as $post_type) {
			// only for 1.0.0
			if ( !in_array($post_type, ['post', 'page']) ) exit;
			
			if ($post_type == 'sfwd-courses' || $post_type == 'sfwd-quiz') continue;
			add_meta_box(
				'cpd_points_meta_box',
				'CPD Points',
				'mediactr_cpd_points_meta_box',
				$post_type,
				'normal',
				'high'
			);
		}
	}

	/**
	 * Renders the CPD points meta box interface.
	 *
	 * Creates the UI for selecting activity type and reading time settings
	 * for individual content items.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The current post object
	 */
	function mediactr_cpd_points_meta_box($post) {
		wp_nonce_field('mediactr_cpd_points_save', 'activity_cpd_points_nonce');
		
		$cpd_activity = get_post_meta( $post->ID, 'cpd_activity', true );

		$current_post_type = get_post_type( $post->ID );
		$post_type_activity = get_option('mediactr_posttype_activities', array());
		$global_activity = $post_type_activity[$current_post_type];

		if ( !isset($cpd_activity) ||  $cpd_activity == '' ) $cpd_activity = $global_activity;

		/* translators: %s: Default activity value */
		$title = sprintf( __('Activity (default is %s ):', 'media-activity-tracker'), $global_activity );
		?>
		<label for="cpd_activity"><?php echo esc_html( $title ); ?></label>
		<select id="cpd_activity" name="cpd_activity">
				<option value="no_tracking" <?php selected($cpd_activity, 'no_tracking'); ?>><?php esc_html_e( 'No Tracking', 'media-activity-tracker' ); ?></option>
				<option value="reading" <?php selected($cpd_activity, 'reading'); ?>><?php esc_html_e( 'Reading', 'media-activity-tracker' ); ?></option>
				<option value="listening" <?php selected($cpd_activity, 'listening'); ?>><?php esc_html_e( 'Listening', 'media-activity-tracker' ); ?></option>
				<option value="watching" <?php selected($cpd_activity, 'watching'); ?>><?php esc_html_e( 'Watching', 'media-activity-tracker' ); ?></option>
		</select>
		<br /><br />
		<div id="cpd-reader-container" <?php echo ($cpd_activity == 'reading') ? '' : 'style="display: none;"'; ?>>
			<label for="cpd_reading_time"><?php esc_html_e( 'Reading Time:', 'media-activity-tracker' ); ?></label>
			<select id="cpd_reading_time" name="cpd_reading_time">
				<option value="1" <?php selected(get_post_meta($post->ID, 'cpd_reading_time', true) , '1'); ?>><?php esc_html_e( 'Enable', 'media-activity-tracker' ); ?></option>
				<option value="0" <?php selected(get_post_meta($post->ID, 'cpd_reading_time', true) , '0'); ?>><?php esc_html_e( 'Disable', 'media-activity-tracker' ); ?></option>
			</select>
		</div>
		<?php
	}

	/**
	 * Saves CPD points settings for individual posts.
	 *
	 * Processes and stores the activity type and reading time configuration
	 * when content is saved.
	 *
	 * @since 1.0.0
	 * @param int $post_id The ID of the post being saved
	 */
	function mediactr_cpd_points_save($post_id)
	{
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return;
		}

		// Verify nonce.
		if (!isset($_POST['activity_cpd_points_nonce'])) {
			return;
		}
		
		$nonce = sanitize_text_field(wp_unslash($_POST['activity_cpd_points_nonce']));
		if (!wp_verify_nonce($nonce, 'mediactr_cpd_points_save')) {
			return;
		}
		
		if (array_key_exists('cpd_activity', $_POST)) {
			update_post_meta($post_id, 'cpd_activity', sanitize_text_field(wp_unslash($_POST['cpd_activity'])));
		}

		if (array_key_exists('cpd_reading_time', $_POST)) {
			update_post_meta($post_id, 'cpd_reading_time', sanitize_text_field(wp_unslash($_POST['cpd_reading_time'])));
		}
	}
	add_action('add_meta_boxes', 'mediactr_cpd_points_field');
	add_action('save_post', 'mediactr_cpd_points_save');	

?>