<?php
/**
 * This file is part of Media Activity Tracker by ActivityPress.
 *
 * @license GPL-2.0-or-later
 * @copyright 2025 Activitypress LLC
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Handles AJAX requests for saving CPD tracking points.
 *
 * This function processes user media consumption tracking data and saves it to the database.
 * It validates the nonce, checks user login status, and manages both new records and updates
 * to existing tracking data.
 *
 * @since 1.0.0
 */
function mediactr_tracking_text_ajax_handler()
{

	$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
	if (!wp_verify_nonce($nonce, 'mediactr-tracking')) {
		wp_send_json_error('Invalid nonce.');
	}	

    if (is_user_logged_in()) {
		$post_id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    	$cpd_activity = mediactr_get_activity( $post_id );

        if ( $cpd_activity )
        {
            global $wpdb, $mediactr_learndash_is_active;
			$progress = [];

            // Custom sanitization for JSON progress data
			if (isset($_POST['progress'])) {
                // Note: We immediately decode and re-validate since sanitize_text_field strips brackets which breaks JSON format
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $raw_progress = wp_unslash($_POST['progress']);
                $decoded_progress = json_decode($raw_progress, true);

                if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded_progress)) {
                    wp_send_json_error('Invalid progress format.');
                }

                // Filter and sanitize in one step
                $progress = array_filter(
                    array_map('absint', $decoded_progress),
                    function($item) { return $item >= 0 && $item <= 100; }
                );

                if (empty($progress) || count($progress) > 100) {
                    wp_send_json_error('Invalid progress data.');
                }
			}	
			
            $current_user = wp_get_current_user();
            // Time validation
			$points = isset($_POST['time']) ? intval(wp_unslash($_POST['time'])) : 0;
            if ($points < 0 || $points > 21600) { // Max 6 hours
                wp_send_json_error('Invalid time value.');
            }            
            // Lastpercent validation  
			$lastpercent = isset($_POST['lastpercent']) ? intval(wp_unslash($_POST['lastpercent'])) : 0;
            if ($lastpercent < 0 || $lastpercent > 100) {
                wp_send_json_error('Invalid percentage value.');
            }            
    		
    	    $threshold = get_option('mediactr_treshold', 100);
			
    		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$checkIfExists = $wpdb->get_row($wpdb->prepare("SELECT id, progress FROM " . $wpdb->prefix . "gp_tracking_points WHERE user_id = %d AND post_id = %d AND activity_type = %s", $current_user->ID,  $post_id, $cpd_activity));
            if ($checkIfExists == NULL) { // insert
                $course_id = (function_exists('learndash_get_course_id')) ? learndash_get_course_id($post_id) : '';
                $lesson_id = (function_exists('learndash_get_lesson_id')) ? learndash_get_lesson_id($post_id) : '';
                if (get_post_type($post_id) == 'sfwd-topic') $topic_id = $post_id;
                else $topic_id = NULL;          

                if ($cpd_activity == 'reading') $point = $points;
                else
                {
                    // remove 0 if part of the array
                    $progress = array_filter($progress, 'mediactr_arrayfilter_zero');
                    
                    $progress = array_unique($progress);
                    sort($progress);
                    
                    // if the media is completed (whether or not longer than 100s)
                    if ((count($progress) > round($points * $threshold / 100) && $points < 100) || (count($progress) > $threshold && $points >= 100)) {
                        $progress = '';
                        $point = $points;
                    }
                    else { // save progress
                        $point = ($points < 100) ? count($progress) : round(count($progress) / 100 * $points);
                    }
                    $progress = ($progress == '') ? '' : wp_json_encode($progress);
                }

                $data = array(
                    'user_id' => $current_user->ID,
                    'post_id' => $post_id,
                    'course_id' => $course_id,
                    'lesson_id' => $lesson_id,
                    'topic_id' => $topic_id,
                    'activity_type' => $cpd_activity,
                    'progress' => $progress,
                    'points' => $point,
    				'lastpercent' => $lastpercent
                );
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
                $result_check = $wpdb->insert($wpdb->prefix . 'gp_tracking_points', $data);
                if ($result_check) {
    				if ( $cpd_activity == 'reading' || $progress == '' ) echo wp_json_encode('complete');
    				else echo wp_json_encode('progress registered');
    				// add Additional Learndash Certicifate if needed
    				if ( $mediactr_learndash_is_active == 1 ) mediactr_check_alc($current_user->ID, $cpd_activity);
    			}

            }
            elseif ($checkIfExists->progress != '' && $cpd_activity != 'reading') { // update
                // merge the progress arrays and calculate the new points
                $new_progress = array_merge($progress, json_decode($checkIfExists->progress, true));
                $new_progress = array_unique($new_progress);

                // remove 0 if part of the array
                $new_progress = array_filter($new_progress, 'mediactr_arrayfilter_zero');     

                sort($new_progress);
                
                // if the media is completed (whether or not longer than 100s)
                if ((count($new_progress) > round($points * $threshold / 100) && $points < 100) || (count($new_progress) > $threshold && $points >= 100)) {
                    $new_progress = '';
                    $point = $points;
                }
                else { // save progress
                    $point = ($points < 100) ? count($new_progress) : round(count($new_progress) / 100 * $points);
                    $new_progress = wp_json_encode($new_progress);
                }
    			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $result_check = $wpdb->query($wpdb->prepare("UPDATE " . $wpdb->prefix . "gp_tracking_points SET finished_at = NOW(), progress = %s, points = %d, lastpercent = %d WHERE id = %d", $new_progress, $point, $lastpercent, $checkIfExists->id));
                if ($result_check) {
    				if ( $progress == '' ) echo wp_json_encode('complete');
    				else echo wp_json_encode('progress updated');
    				// add Additional Learndash Certicifate if needed
    				if ( $mediactr_learndash_is_active == 1 ) mediactr_check_alc($current_user->ID, $cpd_activity);
    			}
            }
            else echo wp_json_encode('already registered content');

            wp_die();
        }
    }
}

add_action('wp_ajax_mediactr_savepoints', 'mediactr_tracking_text_ajax_handler');
add_action('wp_ajax_nopriv_mediactr_savepoints', 'mediactr_tracking_text_ajax_handler');

/**
 * Determines the activity type for a given post.
 *
 * Checks both global and post-specific settings to determine the appropriate
 * activity type (reading, watching, listening) for CPD tracking.
 *
 * @since 1.0.0
 * @param int $post_id The ID of the post to check
 * @return string|bool The activity type or FALSE if tracking is disabled
 */
function mediactr_arrayfilter_zero($value) {
    return $value !== 0;
}

/**
 * Handles AJAX requests for saving user reflections on activity logs.
 *
 * Processes and saves user notes/reflections related to their CPD activities.
 * Validates the request through a nonce check before storing the reflection text.
 *
 * @since 1.0.0
 */
function mediactr_savenote_ajax_handler()
{

	$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
	if (!wp_verify_nonce($nonce, 'mediactr-table')) {
		wp_send_json_error('Invalid nonce.');
	}		

    global $wpdb;
	$id = isset($_POST['record_id']) ? intval(wp_unslash($_POST['record_id'])) : 0;

	// strip the note
	$reflection = isset($_POST['note']) ? wp_strip_all_tags(wp_unslash($_POST['note'])) : '';
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$result = $wpdb->update($wpdb->prefix . 'gp_tracking_points', array('reflection' => $reflection), array('id' => $id));

	if ($result !== false) echo wp_json_encode($reflection);
	else echo wp_json_encode($wpdb->last_error);

    wp_die();
}

add_action('wp_ajax_mediactr_savenote', 'mediactr_savenote_ajax_handler');
add_action('wp_ajax_nopriv_mediactr_savenote', 'mediactr_savenote_ajax_handler');

/**
 * Handles AJAX requests for retrieving user streak data.
 *
 * Retrieves the user's activity streak data for calendar visualization in the frontend.
 * The function calculates consecutive days of CPD activity and returns data for displaying
 * in the activity calendar.
 *
 * @since 1.0.0
 */
function mediactr_getstreak_ajax_handler()
{

	$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
	if (!wp_verify_nonce($nonce, 'mediactr-streak')) {
		wp_send_json_error('Invalid nonce.');
	}			

    $user = get_current_user_id();

    list($streakDays, $dayStreakInRow) = mediactr_front_streak_calc($user, 1);

	echo wp_json_encode($streakDays);

    wp_die();
}

add_action('wp_ajax_mediactr_getstreak', 'mediactr_getstreak_ajax_handler');
add_action('wp_ajax_nopriv_mediactr_getstreak', 'mediactr_getstreak_ajax_handler');

?>