<?php
/**
 * Plugin Name:       MDI Persist Query String
 * Description:       Persist query string parameters across page visits for tracking and analytics.
 * Version:           1.0.0
 * Author:            Medium Interactive
 * Author URI:        https://mediuminteractive.com
 * Text Domain:       mdi-persist-query-string
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.5
 * Requires PHP:      7.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Parameters to look for and persist
 *
 * Enter 1 param per array element
 */
if (!defined('MDIPQS_PERSIST_PARAMETERS')) {
    define('MDIPQS_PERSIST_PARAMETERS', array(
        'campaign',
        'network',
        'keyword',
        'device',
        'placement',
        'utm_source',
        'utm_medium',
        'utm_campaign',
    ));
}

/**
 * Main Plugin Class
 */
class MDIPQS_Persist_Query_String_Plugin
{
    const VERSION = '1.0.0';
    const COOKIE_NAME = 'mdipqs_pqsp';

    /**
     * Provider instance
     *
     * @var MDIPQS_Persist_Query_String_Plugin_Provider
     */
    protected $provider;

    /**
     * Instance
     *
     * @var MDIPQS_Persist_Query_String_Plugin
     */
    private static $_instance = null;

    /**
     * Get instance
     *
     * @return MDIPQS_Persist_Query_String_Plugin
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor
     */
    public function __construct()
    {
        // You can switch between HTTP referrer or HTTP cookie as providers
        $this->provider = new MDIPQS_Persist_Query_String_Plugin_Provider_Http_Referrer(
            self::COOKIE_NAME,
            MDIPQS_PERSIST_PARAMETERS
        );
    }

    /**
     * Initialize the plugin
     */
    public function mdipqs_init()
    {
        // If no referrer unset existing value
        if (!isset($_SERVER['HTTP_REFERER'])) {
            $this->provider->mdipqs_set(null);
        }

        if (is_admin() || (defined('DOING_AJAX') && DOING_AJAX)) {
            return;
        }

        // Skip form submission
        if (isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD']) {
            return;
        }

        // Append query string
        $qs = $this->provider->mdipqs_get();
        if ($qs) {
            $this->provider->mdipqs_set(null);
            wp_safe_redirect(add_query_arg($qs));
            exit;
        }

        // Persist to cookie
        $this->provider->mdipqs_persist();
    }
}

/**
 * Abstract Provider Class
 */
abstract class MDIPQS_Persist_Query_String_Plugin_Provider
{
    /**
     * Arguments
     *
     * @var array
     */
    protected $args;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->args = func_get_args();
    }

    /**
     * Get query string
     *
     * @return array
     */
    public function mdipqs_get()
    {
        return array();
    }

    /**
     * Set value
     *
     * @param mixed $value Value to set.
     */
    public function mdipqs_set($value)
    {
    }

    /**
     * Persist to storage
     */
    public function mdipqs_persist()
    {
    }

    /**
     * Get cookie value
     *
     * @param string $name Cookie name.
     * @return string|null
     */
    protected function mdipqs_get_cookie($name)
    {
        if (isset($_COOKIE[$name])) {
            return sanitize_text_field(wp_unslash($_COOKIE[$name]));
        }
        return null;
    }

    /**
     * Set cookie
     *
     * @param string   $name    Cookie name.
     * @param mixed    $value   Cookie value.
     * @param int|null $expires Expiration time.
     */
    protected function mdipqs_set_cookie($name, $value, $expires = null)
    {
        if (is_null($value)) {
            unset($_COOKIE[$name]);
            setcookie($name, ' ', time() - YEAR_IN_SECONDS, SITECOOKIEPATH);
        } else {
            setcookie($name, $value, $expires, SITECOOKIEPATH);
        }
    }
}

/**
 * HTTP Cookie Provider
 */
class MDIPQS_Persist_Query_String_Plugin_Provider_Http_Cookie extends MDIPQS_Persist_Query_String_Plugin_Provider
{
    /**
     * Get query string from cookie
     *
     * @return array
     */
    public function mdipqs_get()
    {
        $cookie_value = $this->mdipqs_get_cookie($this->args[0]);
        if ($cookie_value) {
            wp_parse_str($cookie_value, $qs);
            return is_array($qs) ? $qs : array();
        }

        return array();
    }

    /**
     * Set cookie value
     *
     * @param mixed $value Value to set.
     */
    public function mdipqs_set($value)
    {
        $this->mdipqs_set_cookie($this->args[0], $value, null);
    }

    /**
     * Persist query string to cookie
     */
    public function mdipqs_persist()
    {
        $qs = array();

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameters for persistence, not form processing.
        foreach ($this->args[1] as $param) {
            if (isset($_GET[$param])) {
                $qs[$param] = sanitize_text_field(wp_unslash($_GET[$param]));
            }
        }

        // Save to cookie
        if (!empty($qs)) {
            $this->mdipqs_set(http_build_query($qs));
        }
    }
}

/**
 * HTTP Referrer Provider
 */
class MDIPQS_Persist_Query_String_Plugin_Provider_Http_Referrer extends MDIPQS_Persist_Query_String_Plugin_Provider
{
    const COOKIE_IDENTIFIER_LENGTH = 6;

    /**
     * Get query string from referrer
     *
     * @return array
     */
    public function mdipqs_get()
    {
        if (isset($_SERVER['HTTP_REFERER'])) {
            $url = esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER']));
            $parsed_url = wp_parse_url($url);
            $query_string = isset($parsed_url['query']) ? $parsed_url['query'] : '';

            wp_parse_str($query_string, $qs);

            $cookie_key = substr(md5($url), 0, self::COOKIE_IDENTIFIER_LENGTH);

            if ($this->mdipqs_get_cookie($cookie_key)) {
                $this->mdipqs_set_cookie($cookie_key, null);
                return array();
            }

            if (isset($qs['_i']) && $qs['_i'] === md5($url)) {
                return array();
            }

            foreach ($qs as $prop => $value) {
                if (!in_array($prop, $this->args[1], true)) {
                    unset($qs[$prop]);
                }
            }

            if (!empty($qs)) {
                $this->mdipqs_set_cookie($cookie_key, '1', time() + 60);
            }

            return $qs;
        }

        return array();
    }
}

/**
 * Initialize plugin
 */
function mdipqs_initialize_plugin()
{
    $plugin = MDIPQS_Persist_Query_String_Plugin::instance();
    $plugin->mdipqs_init();
}
add_action('plugins_loaded', 'mdipqs_initialize_plugin');