<?php

/**
 * Plugin Name:       Smart Traffic Source Filter for WordPress
 * Description:       Hide elements or redirect users when arriving from configured referrer domains. Supports PHP-side and client-side checks and optional notification.
 * Version:           1.0
 * Author:            Medium Interactive
 * Author URI:        https://mediuminteractive.com/
 * License:           GPL-2.0+
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       mdi-http-referer-block
 */

if (!defined('ABSPATH')) {
    exit;
}


/**
 * Default options
 */
function mdihtreb_get_default_options()
{
    return array(
        'show_notification' => 0,
        'enable_js_check' => 0,
        'syndicated_domains' => '',
        'element_selector' => '',
        'redirect_url' => '',
        'cookie_lifetime' => 1, // days
    );
}

/**
 * Check referrer and set cookie early (PHP-side)
 */
function mdihtreb_check_and_set_cookie()
{
    // Only run on front-end requests
    if (is_admin()) {
        return;
    }

    $referrer = isset($_SERVER['HTTP_REFERER']) ? esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
    $referrer = trim(sanitize_text_field($referrer));
    if (empty($referrer)) {
        return;
    }

    $options = wp_parse_args(get_option('mdihtreb_settings', array()), mdihtreb_get_default_options());

    $domains_raw = $options['syndicated_domains'];
    if (empty($domains_raw)) {
        return;
    }

    $domains = array_filter(array_map('trim', explode(',', $domains_raw)));
    if (empty($domains)) {
        return;
    }

    foreach ($domains as $domain) {
        if ('' === $domain) {
            continue;
        }

        if (stripos($referrer, $domain) !== false) {
            $lifetime = max(1, (int) $options['cookie_lifetime']) * DAY_IN_SECONDS;
            $cookie_path = defined('COOKIEPATH') ? COOKIEPATH : '/';
            $cookie_domain = defined('COOKIE_DOMAIN') ? COOKIE_DOMAIN : '';

            // secure flag - only set true if site serves over https
            $secure = is_ssl();

            // httponly true to reduce JS access; but JS check may want to set cookie itself
            setcookie('mdihtreb_syndicated_ref', '1', time() + $lifetime, $cookie_path, $cookie_domain, $secure, true);

            // Also set in global for immediate PHP checks
            $_COOKIE['mdihtreb_syndicated_ref'] = '1';
            break;
        }
    }
}
add_action('init', 'mdihtreb_check_and_set_cookie', 1);

/**
 * Output notice, hide elements, redirect
 */
function mdihtreb_show_syndicated_notice()
{
    if (is_admin()) {
        return;
    }

    $options = wp_parse_args(get_option('mdihtreb_settings', array()), mdihtreb_get_default_options());

    // Show notification — sanitize output and escape.
    if (!empty($options['show_notification']) && !empty($_SERVER['HTTP_REFERER'])) {
        $referrer = sanitize_text_field(esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])));

        // Display only host + path, avoid showing query params or credentials
        $ref_host = wp_parse_url($referrer, PHP_URL_HOST) ? wp_parse_url($referrer, PHP_URL_HOST) : $referrer;
        $ref_path = wp_parse_url($referrer, PHP_URL_PATH) ? wp_parse_url($referrer, PHP_URL_PATH) : '';

        printf(
            '<div class="mdihtreb-referrer-notice" role="status" aria-live="polite" style="position:fixed;bottom:0;left:0;right:0;background:#f1c40f;color:#000;padding:10px;text-align:center;z-index:9999;">%s %s%s</div>',
            esc_html__('You came from:', 'mdi-http-referer-block'),
            esc_html($ref_host),
            $ref_path ? esc_html($ref_path) : ''
        );
    }

    // If PHP-side cookie set, output immediate CSS or redirect (safe)
    if (isset($_COOKIE['mdihtreb_syndicated_ref']) && '1' === $_COOKIE['mdihtreb_syndicated_ref']) {
        if (!empty($options['element_selector'])) {

            // Only allow simple selectors
            $selector = wp_strip_all_tags($options['element_selector']);

            wp_register_style('mdihtreb-hide-elements', false, array(), '1.0');
            wp_enqueue_style('mdihtreb-hide-elements');
            wp_add_inline_style('mdihtreb-hide-elements', esc_html($selector) . ' { display:none !important; }');
        }

        if (!empty($options['redirect_url'])) {
            $redirect = esc_url_raw($options['redirect_url']);

            if (!empty($redirect)) {
                wp_register_script('mdihtreb-redirect', false, array(), '1.0', true);
                wp_enqueue_script('mdihtreb-redirect');
                wp_add_inline_script('mdihtreb-redirect', 'window.location.href = ' . wp_json_encode(esc_url($redirect)) . ';');
            }
        }
    }

    // Enqueue client-side script if enabled
    if (!empty($options['enable_js_check'])) {
        $domains = array_filter(array_map('trim', explode(',', $options['syndicated_domains'])));
        $data = array(
            'domains' => array_values($domains),
            'element_selector' => $options['element_selector'],
            'redirect_url' => $options['redirect_url'],
            'cookie_lifetime' => max(1, (int) $options['cookie_lifetime']),
            'cookie_path' => defined('COOKIEPATH') ? COOKIEPATH : '/',
            'cookie_name' => 'mdihtreb_syndicated_ref'
        );

        // Register and enqueue the script
        wp_register_script('mdihtreb-referrer-tracker', plugins_url('assets/js/mdi-referrer-tracker.js', __FILE__), array(), '1.0', true);
        wp_enqueue_script('mdihtreb-referrer-tracker');
        wp_localize_script('mdihtreb-referrer-tracker', 'mdihtrebReferrerData', $data);
    }
}
add_action('wp_head', 'mdihtreb_show_syndicated_notice', 20);

/**
 * Settings registration
 */
function mdihtreb_register_settings()
{
    register_setting(
        'mdihtreb_settings_group',
        'mdihtreb_settings',
        array('sanitize_callback' => 'mdihtreb_sanitize_settings')
    );

    add_settings_section(
        'mdihtreb_main_section',
        __('MDI HTTP Referer Settings', 'mdi-http-referer-block'),
        '__return_false',
        'mdihtreb-referer-settings'
    );

    add_settings_field(
        'mdihtreb_show_notification',
        __('Show Notification', 'mdi-http-referer-block'),
        'mdihtreb_field_show_notification_cb',
        'mdihtreb-referer-settings',
        'mdihtreb_main_section'
    );

    add_settings_field(
        'mdihtreb_enable_js_check',
        __('Enable JavaScript Referer Check', 'mdi-http-referer-block'),
        'mdihtreb_field_enable_js_cb',
        'mdihtreb-referer-settings',
        'mdihtreb_main_section'
    );
}
add_action('admin_init', 'mdihtreb_register_settings');

/* Settings field callbacks */
function mdihtreb_field_show_notification_cb()
{
    $options = wp_parse_args(get_option('mdihtreb_settings', array()), mdihtreb_get_default_options());
    printf(
        '<input type="checkbox" name="mdihtreb_settings[show_notification]" value="1" %s /> <span class="description">%s</span>',
        checked(1, $options['show_notification'], false),
        esc_html__('Show HTTP Referer source in a small notification at the bottom of the page.', 'mdi-http-referer-block')
    );
}

function mdihtreb_field_enable_js_cb()
{
    $options = wp_parse_args(get_option('mdihtreb_settings', array()), mdihtreb_get_default_options());
    printf(
        '<input type="checkbox" name="mdihtreb_settings[enable_js_check]" value="1" %s /> <span class="description">%s</span>',
        checked(1, $options['enable_js_check'], false),
        esc_html__('Enable client-side referrer checking (useful when referrer not present during server-side init).', 'mdi-http-referer-block')
    );
}

/**
 * Sanitize settings
 */
function mdihtreb_sanitize_settings($input)
{
    $defaults = mdihtreb_get_default_options();

    $output = array();
    // capability check
    if (!current_user_can('manage_options')) {
        return get_option('mdihtreb_settings', $defaults);
    }

    // boolean flags
    $output['show_notification'] = !empty($input['show_notification']) ? 1 : 0;
    $output['enable_js_check'] = !empty($input['enable_js_check']) ? 1 : 0;

    // syndicated domains - keep as comma-separated list of domains/strings
    $domains_raw = isset($input['syndicated_domains']) ? sanitize_text_field(wp_unslash($input['syndicated_domains'])) : '';
    // normalize multiple commas / whitespace
    $parts = array_filter(array_map('trim', explode(',', $domains_raw)));
    $output['syndicated_domains'] = implode(', ', $parts);

    // element selector - limit length and sanitize
    $selector = isset($input['element_selector']) ? wp_strip_all_tags(wp_unslash($input['element_selector'])) : '';
    $output['element_selector'] = substr($selector, 0, 255);

    // redirect url - validate
    $redirect = isset($input['redirect_url']) ? esc_url_raw($input['redirect_url']) : '';
    $output['redirect_url'] = $redirect;

    // cookie lifetime
    $lifetime = isset($input['cookie_lifetime']) ? absint($input['cookie_lifetime']) : $defaults['cookie_lifetime'];
    $output['cookie_lifetime'] = max(1, $lifetime);

    return $output;
}

/**
 * Settings page HTML
 */
function mdihtreb_settings_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $options = wp_parse_args(get_option('mdihtreb_settings', array()), mdihtreb_get_default_options());
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('HTTP Referer Settings', 'mdi-http-referer-block'); ?></h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('mdihtreb_settings_group');
            do_settings_sections('mdihtreb-referer-settings');
            ?>
            <table class="form-table">
                <tr>
                    <th><?php esc_html_e('Blocked Referer Domains', 'mdi-http-referer-block'); ?></th>
                    <td>
                        <input type="text" name="mdihtreb_settings[syndicated_domains]"
                            value="<?php echo esc_attr($options['syndicated_domains']); ?>" class="regular-text" />
                        <p class="description">
                            <?php esc_html_e('Comma-separated list. Partial domain matches are supported (e.g. example.com, partner.example).', 'mdi-http-referer-block'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Element Selector to Hide', 'mdi-http-referer-block'); ?></th>
                    <td>
                        <input type="text" name="mdihtreb_settings[element_selector]"
                            value="<?php echo esc_attr($options['element_selector']); ?>" class="regular-text" />
                        <p class="description">
                            <?php esc_html_e('CSS selector to hide for flagged visitors. Example: ".price, #buy-button"', 'mdi-http-referer-block'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Redirect URL', 'mdi-http-referer-block'); ?></th>
                    <td>
                        <input type="url" name="mdihtreb_settings[redirect_url]"
                            value="<?php echo esc_attr($options['redirect_url']); ?>" class="regular-text" />
                        <p class="description">
                            <?php esc_html_e('Optional. When set, flagged visitors will be redirected to this URL.', 'mdi-http-referer-block'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e('Cookie Lifetime (Days)', 'mdi-http-referer-block'); ?></th>
                    <td>
                        <input type="number" name="mdihtreb_settings[cookie_lifetime]"
                            value="<?php echo esc_attr($options['cookie_lifetime']); ?>" min="1" class="small-text" />
                        <p class="description">
                            <?php esc_html_e('Number of days the cookie remains valid.', 'mdi-http-referer-block'); ?></p>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/**
 * Register admin menu
 */
function mdihtreb_register_settings_page()
{
    add_options_page(
        __('HTTP Referer Settings', 'mdi-http-referer-block'),
        __('HTTP Referer', 'mdi-http-referer-block'),
        'manage_options',
        'mdihtreb-referer-settings',
        'mdihtreb_settings_page'
    );
}
add_action('admin_menu', 'mdihtreb_register_settings_page');
