<?php
/**
 * Pawfect Wishlist Shortcode
 * 
 * This file contains the wishlist shortcode functionality for the Pawfect theme.
 * Moved from theme to plugin to comply with WordPress theme review guidelines.
 * 
 * @package Matcha_Extra
 * @since 1.0.0
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register the wishlist shortcode
 * 
 * Usage: [pawfect_wishlist]
 * 
 * This shortcode displays the user's wishlist items.
 * Requires WooCommerce to be active.
 */
add_shortcode('pawfect_wishlist', 'matcha_extra_pawfect_wishlist_shortcode');

/**
 * Wishlist Shortcode Callback Function
 * 
 * Displays the wishlist items for the current user.
 * Falls back gracefully if WooCommerce is not active or user is not logged in.
 * 
 * @param array $atts Shortcode attributes (currently unused, reserved for future use)
 * @return string HTML output of the wishlist
 */
function matcha_extra_pawfect_wishlist_shortcode($atts) {
    // Start output buffering
    ob_start();

    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        echo '<div class="pawfect-wishlist-notice">';
        echo '<p>' . esc_html__('WooCommerce is required for the wishlist feature.', 'matcha-extra') . '</p>';
        echo '</div>';
        return ob_get_clean();
    }

    // Main wishlist container
    echo '<div class="pawfect-wishlist-container">';

    // Get current user ID
    $user_id = get_current_user_id();

    // Check if user is logged in
    if (!$user_id) {
        echo '<div class="pawfect-wishlist-login-notice">';
        echo '<p>' . esc_html__('Please log in to view your wishlist.', 'matcha-extra') . '</p>';
        echo '<a href="' . esc_url(wp_login_url(get_permalink())) . '" class="pawfect-btn pawfect-btn-primary">';
        echo esc_html__('Log In', 'matcha-extra');
        echo '</a>';
        echo '</div>';
        echo '</div>';
        return ob_get_clean();
    }

    // Get wishlist items for current user
    // Check if theme function exists (for compatibility)
    if (function_exists('pawfect_get_user_wishlist')) {
        $wishlist_items = pawfect_get_user_wishlist($user_id);
    } else {
        // Fallback: try to get from database directly
        global $wpdb;
        $table_name = $wpdb->prefix . 'pawfect_wishlist';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") == $table_name) {
            $wishlist_items = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM $table_name WHERE user_id = %d ORDER BY added_date DESC",
                    $user_id
                ),
                ARRAY_A
            );
        } else {
            $wishlist_items = array();
        }
    }

    // Display wishlist items or empty message
    if (!empty($wishlist_items)) {
        echo '<div class="pawfect-wishlist-items">';
        
        foreach ($wishlist_items as $item) {
            $product_id = isset($item['product_id']) ? $item['product_id'] : 0;
            $product = wc_get_product($product_id);
            
            if ($product) {
                echo '<div class="pawfect-wishlist-item" data-product-id="' . esc_attr($product_id) . '">';
                
                // Product image
                echo '<div class="pawfect-wishlist-item-image">';
                echo $product->get_image('thumbnail');
                echo '</div>';
                
                // Product details
                echo '<div class="pawfect-wishlist-item-details">';
                echo '<h3 class="pawfect-wishlist-item-title">' . esc_html($product->get_name()) . '</h3>';
                echo '<div class="pawfect-wishlist-item-price">' . wp_kses_post($product->get_price_html()) . '</div>';
                
                // Stock status
                if ($product->is_in_stock()) {
                    echo '<p class="pawfect-wishlist-stock in-stock">' . esc_html__('In Stock', 'matcha-extra') . '</p>';
                } else {
                    echo '<p class="pawfect-wishlist-stock out-of-stock">' . esc_html__('Out of Stock', 'matcha-extra') . '</p>';
                }
                
                echo '</div>';
                
                // Action buttons
                echo '<div class="pawfect-wishlist-item-actions">';
                
                // Add to cart button (if in stock)
                if ($product->is_in_stock()) {
                    echo '<a href="' . esc_url($product->add_to_cart_url()) . '" class="pawfect-btn pawfect-btn-primary add-to-cart-btn">';
                    echo esc_html__('Add to Cart', 'matcha-extra');
                    echo '</a>';
                }
                
                // Remove from wishlist button
                echo '<button class="pawfect-btn pawfect-btn-secondary remove-from-wishlist" data-product-id="' . esc_attr($product_id) . '">';
                echo esc_html__('Remove', 'matcha-extra');
                echo '</button>';
                
                echo '</div>';
                
                echo '</div>'; // .pawfect-wishlist-item
            }
        }
        
        echo '</div>'; // .pawfect-wishlist-items
    } else {
        // Empty wishlist message
        echo '<div class="pawfect-wishlist-empty">';
        echo '<p>' . esc_html__('Your wishlist is empty.', 'matcha-extra') . '</p>';
        echo '<a href="' . esc_url(get_permalink(wc_get_page_id('shop'))) . '" class="pawfect-btn pawfect-btn-primary">';
        echo esc_html__('Continue Shopping', 'matcha-extra');
        echo '</a>';
        echo '</div>';
    }

    echo '</div>'; // .pawfect-wishlist-container

    // Return the buffered output
    return ob_get_clean();
}

