<?php
/**
 * Pawfect Wishlist Database Functions
 * 
 * This file contains the wishlist database operations for the Pawfect theme.
 * These functions were moved from the theme to the plugin to comply with
 * WordPress.org theme review guidelines (themes should not create database tables).
 * 
 * @package Matcha_Extra
 * @since 1.0.2
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Create wishlist database table on plugin activation
 */
function matcha_create_wishlist_table() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'pawfect_wishlist';
    
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        product_id bigint(20) NOT NULL,
        date_added datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY (id),
        UNIQUE KEY user_product (user_id, product_id)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

// Create table on init if it doesn't exist (for users who already have the plugin)
add_action('init', 'matcha_maybe_create_wishlist_table');
function matcha_maybe_create_wishlist_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'pawfect_wishlist';
    
    // Only check once per session
    if (get_transient('matcha_wishlist_table_checked')) {
        return;
    }
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
    if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) !== $table_name) {
        matcha_create_wishlist_table();
    }
    
    set_transient('matcha_wishlist_table_checked', true, HOUR_IN_SECONDS);
}

/**
 * Get user's wishlist items
 * 
 * @param int|null $user_id User ID (defaults to current user)
 * @return array Array of product IDs in wishlist
 */
function matcha_get_wishlist_items($user_id = null) {
    global $wpdb;
    
    if (!$user_id) {
        $user_id = get_current_user_id();
    }
    
    if (!$user_id) {
        return array();
    }
    
    $table_name = $wpdb->prefix . 'pawfect_wishlist';
    
    // Check if table exists
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
    if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) !== $table_name) {
        return array();
    }
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $results = $wpdb->get_results($wpdb->prepare(
        "SELECT product_id FROM {$table_name} WHERE user_id = %d ORDER BY date_added DESC",
        $user_id
    ));
    
    return array_map(function($item) {
        return (int) $item->product_id;
    }, $results);
}

/**
 * Add product to user's wishlist
 * 
 * @param int $product_id Product ID to add
 * @param int|null $user_id User ID (defaults to current user)
 * @return bool True on success, false on failure
 */
function matcha_add_to_wishlist($product_id, $user_id = null) {
    global $wpdb;
    
    if (!$user_id) {
        $user_id = get_current_user_id();
    }
    
    if (!$user_id || !$product_id) {
        return false;
    }
    
    $table_name = $wpdb->prefix . 'pawfect_wishlist';
    
    // Ensure table exists
    matcha_maybe_create_wishlist_table();
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
    $result = $wpdb->insert(
        $table_name,
        array(
            'user_id' => absint($user_id),
            'product_id' => absint($product_id),
            'date_added' => current_time('mysql')
        ),
        array('%d', '%d', '%s')
    );
    
    return $result !== false;
}

/**
 * Remove product from user's wishlist
 * 
 * @param int $product_id Product ID to remove
 * @param int|null $user_id User ID (defaults to current user)
 * @return bool True on success, false on failure
 */
function matcha_remove_from_wishlist($product_id, $user_id = null) {
    global $wpdb;
    
    if (!$user_id) {
        $user_id = get_current_user_id();
    }
    
    if (!$user_id || !$product_id) {
        return false;
    }
    
    $table_name = $wpdb->prefix . 'pawfect_wishlist';
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
    $result = $wpdb->delete(
        $table_name,
        array(
            'user_id' => absint($user_id),
            'product_id' => absint($product_id)
        ),
        array('%d', '%d')
    );
    
    return $result !== false;
}

/**
 * Get wishlist count for user
 * 
 * @param int|null $user_id User ID (defaults to current user)
 * @return int Number of items in wishlist
 */
function matcha_get_wishlist_count($user_id = null) {
    $items = matcha_get_wishlist_items($user_id);
    return count($items);
}

/**
 * Check if product is in user's wishlist
 * 
 * @param int $product_id Product ID to check
 * @param int|null $user_id User ID (defaults to current user)
 * @return bool True if in wishlist, false otherwise
 */
function matcha_is_in_wishlist($product_id, $user_id = null) {
    $wishlist_items = matcha_get_wishlist_items($user_id);
    return in_array((int) $product_id, $wishlist_items, true);
}

/**
 * Get wishlist page URL
 * 
 * @return string Wishlist page URL
 */
function matcha_get_wishlist_url() {
    // First try to get the actual page
    $wishlist_page = get_page_by_path('wishlist');
    if ($wishlist_page) {
        return get_permalink($wishlist_page->ID);
    }
    
    // Fallback
    return home_url('/wishlist/');
}

/**
 * AJAX handler for toggling wishlist items
 */
add_action('wp_ajax_matcha_toggle_wishlist', 'matcha_toggle_wishlist_ajax');
add_action('wp_ajax_nopriv_matcha_toggle_wishlist', 'matcha_toggle_wishlist_ajax_nopriv');

function matcha_toggle_wishlist_ajax() {
    check_ajax_referer('pawfect_wishlist_nonce', 'nonce');
    
    $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
    $action = isset($_POST['wishlist_action']) ? sanitize_text_field($_POST['wishlist_action']) : '';
    
    if (!$product_id) {
        wp_send_json_error(__('Invalid product ID', 'matcha-extra'));
        return;
    }
    
    $success = false;
    $message = '';
    
    if ($action === 'add') {
        $success = matcha_add_to_wishlist($product_id);
        $message = $success ? __('Added to wishlist', 'matcha-extra') : __('Failed to add to wishlist', 'matcha-extra');
    } elseif ($action === 'remove') {
        $success = matcha_remove_from_wishlist($product_id);
        $message = $success ? __('Removed from wishlist', 'matcha-extra') : __('Failed to remove from wishlist', 'matcha-extra');
    }
    
    if ($success) {
        wp_send_json_success(array(
            'message' => $message,
            'wishlist_count' => matcha_get_wishlist_count(),
            'is_in_wishlist' => matcha_is_in_wishlist($product_id)
        ));
    } else {
        wp_send_json_error($message);
    }
}

function matcha_toggle_wishlist_ajax_nopriv() {
    wp_send_json_error(__('Please log in to manage your wishlist', 'matcha-extra'));
}
