<?php
/**
 * Button Colours Settings Template
 *
 * Displays button colour settings for customizing frontend button gradients.
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/templates/admin
 * @since      1.0.71
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Get current settings values with defaults (same as trial booking button).
$macm_button_gradient_start = get_option( 'macm_button_gradient_start', '#B11226' );
$macm_button_gradient_end   = get_option( 'macm_button_gradient_end', '#8F0E1E' );
?>

<div class="kcm-button-colours-settings">
	<form method="post" action="">
		<?php wp_nonce_field( 'macm_save_settings', 'macm_settings_nonce' ); ?>

		<table class="form-table" role="presentation">
			<tbody>
				<tr>
					<th scope="row">
						<label for="macm_button_gradient_start"><?php esc_html_e( 'Button Gradient', 'martial-arts-club-manager' ); ?></label>
					</th>
					<td>
						<fieldset>
							<legend class="screen-reader-text">
								<span><?php esc_html_e( 'Button Gradient Colors', 'martial-arts-club-manager' ); ?></span>
							</legend>

							<p>
								<label for="macm_button_gradient_start" style="display: inline-block; width: 120px;">
									<?php esc_html_e( 'Start Color:', 'martial-arts-club-manager' ); ?>
								</label>
								<input
									type="color"
									name="macm_button_gradient_start"
									id="macm_button_gradient_start"
									value="<?php echo esc_attr( $macm_button_gradient_start ); ?>"
									class="kcm-color-picker"
								/>
								<input
									type="text"
									name="macm_button_gradient_start_text"
									id="macm_button_gradient_start_text"
									value="<?php echo esc_attr( $macm_button_gradient_start ); ?>"
									class="regular-text"
									pattern="^#[0-9A-Fa-f]{6}$"
									style="width: 100px; margin-left: 10px;"
								/>
							</p>

							<p>
								<label for="macm_button_gradient_end" style="display: inline-block; width: 120px;">
									<?php esc_html_e( 'End Color:', 'martial-arts-club-manager' ); ?>
								</label>
								<input
									type="color"
									name="macm_button_gradient_end"
									id="macm_button_gradient_end"
									value="<?php echo esc_attr( $macm_button_gradient_end ); ?>"
									class="kcm-color-picker"
								/>
								<input
									type="text"
									name="macm_button_gradient_end_text"
									id="macm_button_gradient_end_text"
									value="<?php echo esc_attr( $macm_button_gradient_end ); ?>"
									class="regular-text"
									pattern="^#[0-9A-Fa-f]{6}$"
									style="width: 100px; margin-left: 10px;"
								/>
							</p>

							<p class="description">
								<?php esc_html_e( 'Choose the gradient colors for all frontend buttons. The gradient goes from left to right.', 'martial-arts-club-manager' ); ?>
							</p>
						</fieldset>
					</td>
				</tr>

				<tr>
					<th scope="row">
						<label><?php esc_html_e( 'Preview', 'martial-arts-club-manager' ); ?></label>
					</th>
					<td>
						<div id="kcm-button-preview-container" style="padding: 20px; background: #f5f5f5; border-radius: 8px; max-width: 600px;">
							<p style="margin: 0 0 15px 0; color: #6B7280; font-size: 14px;">
								<?php esc_html_e( 'Button preview:', 'martial-arts-club-manager' ); ?>
							</p>
							<button
								type="button"
								id="kcm-button-preview"
								style="background: linear-gradient(135deg, <?php echo esc_attr( $macm_button_gradient_start ); ?> 0%, <?php echo esc_attr( $macm_button_gradient_end ); ?> 100%); color: #ffffff; border: none; padding: 15px 40px; font-size: 16px; font-weight: 600; border-radius: 4px; cursor: pointer; transition: transform 0.2s, box-shadow 0.2s;"
								onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 4px 12px rgba(177, 18, 38, 0.4)';"
								onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none';"
							>
								<?php esc_html_e( 'Submit Button', 'martial-arts-club-manager' ); ?>
							</button>
						</div>
						<p class="description" style="margin-top: 10px;">
							<?php esc_html_e( 'The preview updates as you change the colors. This gradient will be applied to all frontend buttons.', 'martial-arts-club-manager' ); ?>
						</p>
					</td>
				</tr>

				<tr>
					<th scope="row">
						<label><?php esc_html_e( 'Reset to Default', 'martial-arts-club-manager' ); ?></label>
					</th>
					<td>
						<button type="button" class="button" id="kcm-reset-button-gradient">
							<?php esc_html_e( 'Reset to Default Gradient', 'martial-arts-club-manager' ); ?>
						</button>
						<p class="description">
							<?php esc_html_e( 'Reset button gradient colors to the default red gradient (#B11226 to #8F0E1E).', 'martial-arts-club-manager' ); ?>
						</p>
					</td>
				</tr>
			</tbody>
		</table>

		<?php
		/**
		 * Action hook for adding custom button colour settings fields.
		 *
		 * @since 1.0.71
		 */
		do_action( 'macm_button_colours_settings_fields' );
		?>

		<?php submit_button( __( 'Save Settings', 'martial-arts-club-manager' ), 'primary', 'macm_settings_submit' ); ?>
	</form>
</div>

<?php
// Enqueue inline script for button colour settings using WordPress-compliant method.
$macm_button_colours_script = "
jQuery(document).ready(function($) {
	// Sync color picker with text input.
	function syncColorInputs() {
		$('#macm_button_gradient_start').on('input', function() {
			$('#macm_button_gradient_start_text').val($(this).val());
			updatePreview();
		});

		$('#macm_button_gradient_start_text').on('input', function() {
			var color = $(this).val();
			if (/^#[0-9A-Fa-f]{6}$/.test(color)) {
				$('#macm_button_gradient_start').val(color);
				updatePreview();
			}
		});

		$('#macm_button_gradient_end').on('input', function() {
			$('#macm_button_gradient_end_text').val($(this).val());
			updatePreview();
		});

		$('#macm_button_gradient_end_text').on('input', function() {
			var color = $(this).val();
			if (/^#[0-9A-Fa-f]{6}$/.test(color)) {
				$('#macm_button_gradient_end').val(color);
				updatePreview();
			}
		});
	}

	// Update preview.
	function updatePreview() {
		var startColor = $('#macm_button_gradient_start').val();
		var endColor = $('#macm_button_gradient_end').val();
		var gradient = 'linear-gradient(135deg, ' + startColor + ' 0%, ' + endColor + ' 100%)';
		$('#kcm-button-preview').css('background', gradient);
	}

	// Reset to default.
	$('#kcm-reset-button-gradient').on('click', function() {
		$('#macm_button_gradient_start').val('#B11226');
		$('#macm_button_gradient_start_text').val('#B11226');
		$('#macm_button_gradient_end').val('#8F0E1E');
		$('#macm_button_gradient_end_text').val('#8F0E1E');
		updatePreview();
	});

	// Initialize.
	syncColorInputs();
});
";
wp_add_inline_script( 'kcm-admin-settings', $macm_button_colours_script );

// Enqueue inline styles for button colour settings using WordPress-compliant method.
$macm_button_colours_styles = '
.kcm-button-colours-settings fieldset {
	margin-bottom: 15px;
}

.kcm-button-colours-settings .kcm-color-picker {
	width: 60px;
	height: 35px;
	border: 1px solid #D1D5DB;
	border-radius: 4px;
	cursor: pointer;
	vertical-align: middle;
}

.kcm-button-colours-settings input[type="text"] {
	vertical-align: middle;
}
';
wp_add_inline_style( 'kcm-admin-settings', $macm_button_colours_styles );
?>
