<?php
/**
 * Admin Data Import Tab Template
 *
 * Import interface with file upload, user mapping, and conflict resolution options.
 * Variable $entities is passed from the parent template (data-page.php).
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/templates/admin
 * @since      1.0.265
 * @since      1.0.272 Added success/error notices from URL parameters.
 * @since      1.0.280 Redesigned UI with card-based layout for better clarity.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Variable is passed from parent template.
if ( ! isset( $entities ) || ! is_array( $entities ) ) {
	wp_die( esc_html__( 'Invalid template usage: $entities array required.', 'martial-arts-club-manager' ) );
}

// Check for import success/error messages from URL parameters.
$macm_import_success = filter_input( INPUT_GET, 'import_success', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
$macm_import_error   = filter_input( INPUT_GET, 'import_error', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
?>

<div class="macm-import-tab">
	<?php if ( ! empty( $macm_import_success ) ) : ?>
		<div class="notice notice-success is-dismissible">
			<p><?php echo esc_html( rawurldecode( $macm_import_success ) ); ?></p>
		</div>
	<?php endif; ?>

	<?php if ( ! empty( $macm_import_error ) ) : ?>
		<div class="notice notice-error is-dismissible">
			<p><?php echo esc_html( rawurldecode( $macm_import_error ) ); ?></p>
		</div>
	<?php endif; ?>

	<?php
	$macm_premium_files_exist = file_exists( MACM_PLUGIN_DIR . 'includes/classes/class-macm-location.php' );
	$macm_can_use_premium     = function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() && $macm_premium_files_exist;
	?>

	<div class="macm-data-description">
		<h2><?php esc_html_e( 'Import Data', 'martial-arts-club-manager' ); ?></h2>
		<?php if ( $macm_can_use_premium ) : ?>
			<p><?php esc_html_e( 'Import club data from a backup file or spreadsheet. Supported formats: JSON (full backup), ZIP (with photos), or multiple CSV files (one per data type).', 'martial-arts-club-manager' ); ?></p>
		<?php else : ?>
			<p><?php esc_html_e( 'Import club data from a backup file or spreadsheet. Supported formats: JSON (full backup) or multiple CSV files (one per data type).', 'martial-arts-club-manager' ); ?></p>
		<?php endif; ?>
	</div>

	<?php if ( $macm_can_use_premium ) : ?>
	<div class="notice notice-info inline macm-import-notice">
		<p>
			<span class="dashicons dashicons-info"></span>
			<strong><?php esc_html_e( 'Tip:', 'martial-arts-club-manager' ); ?></strong>
			<?php esc_html_e( 'ZIP files exported with the "Include Photos" option will automatically import photos. Plain JSON and CSV imports do not include photos.', 'martial-arts-club-manager' ); ?>
		</p>
	</div>
	<?php endif; ?>

	<form method="post" enctype="multipart/form-data" class="macm-import-form" id="macm-import-form">
		<?php wp_nonce_field( 'macm_import_data', 'macm_import_nonce' ); ?>
		<input type="hidden" name="action" value="macm_import_data">
		<input type="hidden" name="import_id" id="macm-import-id" value="">

		<!-- File Upload -->
		<div class="macm-form-section">
			<h3><?php esc_html_e( 'Select Files', 'martial-arts-club-manager' ); ?></h3>
			<div class="macm-file-upload">
				<input type="file"
					name="import_file[]"
					id="macm-import-file"
					accept=".json,.csv,.zip"
					multiple>
				<label for="macm-import-file" class="macm-file-label">
					<span class="dashicons dashicons-upload"></span>
					<span class="macm-file-text"><?php esc_html_e( 'Choose files or drag and drop', 'martial-arts-club-manager' ); ?></span>
				</label>
				<p class="macm-file-info">
					<?php
					printf(
						/* translators: %s: maximum file size */
						esc_html__( 'Maximum file size: %s per file (100MB for ZIP). Formats: JSON, ZIP (with photos), or CSV files.', 'martial-arts-club-manager' ),
						esc_html( size_format( wp_max_upload_size() ) )
					);
					?>
				</p>
			</div>

			<!-- Selected Files List (populated via JS) -->
			<div class="macm-selected-files" id="macm-selected-files" style="display: none;">
				<h4><?php esc_html_e( 'Selected Files', 'martial-arts-club-manager' ); ?></h4>
				<ul class="macm-file-list"></ul>
				<button type="button" class="button macm-clear-files" id="macm-clear-files">
					<span class="dashicons dashicons-no-alt"></span>
					<?php esc_html_e( 'Clear All', 'martial-arts-club-manager' ); ?>
				</button>
			</div>

			<!-- File Preview (populated via JS) -->
			<div class="macm-file-preview" id="macm-file-preview" style="display: none;">
				<h4><?php esc_html_e( 'File Preview', 'martial-arts-club-manager' ); ?></h4>
				<div class="macm-preview-content"></div>
			</div>
		</div>

		<!-- Import Options: Two Column Layout -->
		<div class="macm-import-options-grid">
			<!-- User Mapping Options -->
			<div class="macm-form-section macm-options-card">
				<div class="macm-options-header">
					<span class="dashicons dashicons-admin-users"></span>
					<div>
						<h3><?php esc_html_e( 'User Mapping', 'martial-arts-club-manager' ); ?></h3>
						<p class="description"><?php esc_html_e( 'How to link members to WordPress users', 'martial-arts-club-manager' ); ?></p>
					</div>
				</div>

				<div class="macm-option-cards">
					<label class="macm-option-card">
						<input type="radio" name="user_mapping" value="match_email" checked>
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-email"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Match by Email', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'Link to users with matching email. Best for migrations.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>

					<label class="macm-option-card">
						<input type="radio" name="user_mapping" value="assign_current">
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-admin-users"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Assign to Current User', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'All members assigned to your account.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>

					<label class="macm-option-card">
						<input type="radio" name="user_mapping" value="skip_unmatched">
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-dismiss"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Skip Unmatched', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'Only import if email matches existing user.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>
				</div>
			</div>

			<!-- Conflict Resolution Options -->
			<div class="macm-form-section macm-options-card">
				<div class="macm-options-header">
					<span class="dashicons dashicons-update"></span>
					<div>
						<h3><?php esc_html_e( 'Conflict Resolution', 'martial-arts-club-manager' ); ?></h3>
						<p class="description"><?php esc_html_e( 'How to handle records that already exist', 'martial-arts-club-manager' ); ?></p>
					</div>
				</div>

				<div class="macm-option-cards">
					<label class="macm-option-card">
						<input type="radio" name="conflict_resolution" value="skip" checked>
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-migrate"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Skip Existing', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'Keep existing, only import new records.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>

					<label class="macm-option-card">
						<input type="radio" name="conflict_resolution" value="update">
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-update-alt"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Update Existing', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'Overwrite existing with imported data.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>

					<label class="macm-option-card">
						<input type="radio" name="conflict_resolution" value="create_new">
						<span class="macm-option-radio"></span>
						<span class="macm-option-icon"><span class="dashicons dashicons-plus-alt2"></span></span>
						<span class="macm-option-content">
							<strong><?php esc_html_e( 'Create Duplicates', 'martial-arts-club-manager' ); ?></strong>
							<span class="macm-option-desc"><?php esc_html_e( 'Always create new, even if similar exist.', 'martial-arts-club-manager' ); ?></span>
						</span>
					</label>
				</div>
			</div>
		</div>

		<!-- Import Progress -->
		<div class="macm-form-section macm-import-progress" id="macm-import-progress" style="display: none;">
			<h3><?php esc_html_e( 'Import Progress', 'martial-arts-club-manager' ); ?></h3>
			<div class="macm-progress-bar">
				<div class="macm-progress-fill" style="width: 0%;"></div>
			</div>
			<div class="macm-progress-status">
				<span class="macm-progress-text"><?php esc_html_e( 'Preparing...', 'martial-arts-club-manager' ); ?></span>
				<span class="macm-progress-percent">0%</span>
			</div>
			<div class="macm-progress-log"></div>
		</div>

		<!-- Import Results -->
		<div class="macm-form-section macm-import-results" id="macm-import-results" style="display: none;">
			<h3><?php esc_html_e( 'Import Results', 'martial-arts-club-manager' ); ?></h3>
			<div class="macm-results-content"></div>
		</div>

		<!-- Import Button -->
		<div class="macm-form-actions">
			<button type="submit" class="button button-primary button-hero macm-import-btn" id="macm-import-btn">
				<span class="dashicons dashicons-upload"></span>
				<?php esc_html_e( 'Start Import', 'martial-arts-club-manager' ); ?>
			</button>

			<p class="macm-import-warning">
				<span class="dashicons dashicons-warning"></span>
				<?php esc_html_e( 'Warning: Importing data may modify existing records. It is recommended to backup your database before proceeding.', 'martial-arts-club-manager' ); ?>
			</p>
		</div>
	</form>
</div>
