<?php
/**
 * Admin Data Export Tab Template
 *
 * Export interface with format selection and entity checkboxes.
 * Variable $entities is passed from the parent template (data-page.php).
 *
 * @package    Karate_Club_Manager
 * @subpackage Karate_Club_Manager/templates/admin
 * @since      1.0.265
 * @since      1.0.280 Redesigned UI for better clarity and visual appeal
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Variable is passed from parent template.
if ( ! isset( $entities ) || ! is_array( $entities ) ) {
	wp_die( esc_html__( 'Invalid template usage: $entities array required.', 'martial-arts-club-manager' ) );
}

// Icon mapping for each entity type.
$macm_entity_icons = array(
	'settings'            => 'dashicons-admin-settings',
	'belt_colors'         => 'dashicons-awards',
	'membership_types'    => 'dashicons-id-alt',
	'members'             => 'dashicons-groups',
	'trial_bookings'      => 'dashicons-calendar-alt',
	'locations'           => 'dashicons-location',
	'groups'              => 'dashicons-networking',
	'clubs'               => 'dashicons-building',
	'instructors'         => 'dashicons-businessman',
	'classes'             => 'dashicons-schedule',
	'class_enrollments'   => 'dashicons-clipboard',
	'attendance_records'  => 'dashicons-yes-alt',
	'events'              => 'dashicons-calendar',
	'event_registrations' => 'dashicons-tickets-alt',
	'training_videos'     => 'dashicons-video-alt3',
	'grading_history'     => 'dashicons-chart-bar',
);
?>

<div class="macm-export-tab">
	<div class="macm-data-description">
		<h2><?php esc_html_e( 'Export Data', 'martial-arts-club-manager' ); ?></h2>
		<p><?php esc_html_e( 'Export your club data for backup or migration to another site. Select the data types you want to export and choose a format.', 'martial-arts-club-manager' ); ?></p>
	</div>

	<?php
	$macm_premium_files_exist = file_exists( MACM_PLUGIN_DIR . 'includes/classes/class-macm-location.php' );
	$macm_can_use_premium     = function_exists( 'macm_fs' ) && macm_fs()->can_use_premium_code() && $macm_premium_files_exist;
	$macm_has_zip_archive     = class_exists( 'ZipArchive' );
	$macm_can_export_photos   = $macm_can_use_premium && $macm_has_zip_archive;
	?>

	<?php if ( $macm_can_use_premium && ! $macm_has_zip_archive ) : ?>
	<div class="notice notice-info inline macm-export-notice">
		<p>
			<span class="dashicons dashicons-info"></span>
			<strong><?php esc_html_e( 'Note:', 'martial-arts-club-manager' ); ?></strong>
			<?php esc_html_e( 'Photo export requires the PHP ZipArchive extension. Please contact your hosting provider to enable it.', 'martial-arts-club-manager' ); ?>
		</p>
	</div>
	<?php endif; ?>

	<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="macm-export-form">
		<?php wp_nonce_field( 'macm_export_data', 'macm_export_nonce' ); ?>
		<input type="hidden" name="action" value="macm_export_data">

		<!-- Export Format Selection -->
		<div class="macm-form-section">
			<h3><?php esc_html_e( 'Export Format', 'martial-arts-club-manager' ); ?></h3>
			<div class="macm-format-options">
				<label class="macm-format-option">
					<input type="radio" name="export_format" value="json" checked>
					<span class="macm-format-icon"><span class="dashicons dashicons-media-code"></span></span>
					<span class="macm-format-label">
						<strong><?php esc_html_e( 'JSON', 'martial-arts-club-manager' ); ?></strong>
						<span class="macm-format-desc"><?php esc_html_e( 'Full backup with all relationships. Best for site migration.', 'martial-arts-club-manager' ); ?></span>
					</span>
				</label>
				<label class="macm-format-option">
					<input type="radio" name="export_format" value="csv">
					<span class="macm-format-icon"><span class="dashicons dashicons-media-spreadsheet"></span></span>
					<span class="macm-format-label">
						<strong><?php esc_html_e( 'CSV', 'martial-arts-club-manager' ); ?></strong>
						<span class="macm-format-desc"><?php esc_html_e( 'Spreadsheet format. Multiple selections download as ZIP. Best for Excel/Google Sheets.', 'martial-arts-club-manager' ); ?></span>
					</span>
				</label>
			</div>
		</div>

		<?php if ( $macm_can_use_premium ) : ?>
		<!-- Include Photos Option (JSON format only, premium + ZipArchive required) -->
		<div class="macm-form-section macm-photos-section" id="macm-photos-section" style="display: none;">
			<h3><?php esc_html_e( 'Photo Options', 'martial-arts-club-manager' ); ?></h3>
			<label for="macm-include-photos">
				<input type="checkbox"
					name="include_photos"
					value="1"
					id="macm-include-photos"
					<?php echo $macm_can_export_photos ? '' : 'disabled'; ?>>
				<strong><?php esc_html_e( 'Include Photos', 'martial-arts-club-manager' ); ?></strong>
			</label>
			<p class="description"><?php esc_html_e( 'Bundle member, instructor, and event photos in a ZIP file. This will export a .zip instead of .json.', 'martial-arts-club-manager' ); ?></p>
		</div>
		<?php endif; ?>

		<!-- Entity Selection -->
		<div class="macm-form-section macm-entity-section">
			<div class="macm-section-header">
				<h3><?php esc_html_e( 'Select Data to Export', 'martial-arts-club-manager' ); ?></h3>
				<div class="macm-select-actions">
					<button type="button" class="button button-small macm-select-all">
						<span class="dashicons dashicons-yes"></span>
						<?php esc_html_e( 'Select All', 'martial-arts-club-manager' ); ?>
					</button>
					<button type="button" class="button button-small macm-select-none">
						<span class="dashicons dashicons-no-alt"></span>
						<?php esc_html_e( 'Select None', 'martial-arts-club-manager' ); ?>
					</button>
				</div>
			</div>

			<?php
			// Check if this is the premium build (not just if user has a license).
			// is_premium() checks the build type, can_use_premium_code() checks license.
			// Free build should never show premium options, even if user has a license.
			$macm_is_premium_build = function_exists( 'macm_fs' ) && macm_fs()->is_premium();
			?>
			<?php foreach ( $entities as $macm_group_key => $macm_group ) : ?>
				<?php
				// Skip premium group entirely for free build.
				if ( 'premium' === $macm_group_key && ! $macm_is_premium_build ) {
					continue;
				}
				?>
				<div class="macm-entity-group <?php echo esc_attr( $macm_group_key ); ?>">
					<div class="macm-group-header">
						<h4>
							<?php if ( 'premium' === $macm_group_key ) : ?>
								<span class="dashicons dashicons-star-filled macm-premium-star"></span>
							<?php else : ?>
								<span class="dashicons dashicons-portfolio"></span>
							<?php endif; ?>
							<?php echo esc_html( $macm_group['label'] ); ?>
						</h4>
						<span class="macm-group-count">
							<?php
							/* translators: %d: number of data types in group */
							printf( esc_html__( '%d data types', 'martial-arts-club-manager' ), count( $macm_group['entities'] ) );
							?>
						</span>
					</div>
					<div class="macm-entity-grid">
						<?php foreach ( $macm_group['entities'] as $macm_entity_key => $macm_entity ) : ?>
							<?php
							$macm_is_premium    = ! empty( $macm_entity['premium'] );
							$macm_is_available  = ! $macm_is_premium || ( isset( $macm_entity['available'] ) && $macm_entity['available'] );
							$macm_disabled_attr = $macm_is_available ? '' : 'disabled';
							$macm_locked_class  = $macm_is_available ? '' : 'macm-entity-locked';
							$macm_icon_class    = isset( $macm_entity_icons[ $macm_entity_key ] ) ? $macm_entity_icons[ $macm_entity_key ] : 'dashicons-database';
							?>
							<label class="macm-entity-card <?php echo esc_attr( $macm_locked_class ); ?>" for="export-entity-<?php echo esc_attr( $macm_entity_key ); ?>">
								<div class="macm-card-checkbox">
									<input type="checkbox"
										name="export_entities[]"
										value="<?php echo esc_attr( $macm_entity_key ); ?>"
										id="export-entity-<?php echo esc_attr( $macm_entity_key ); ?>"
										aria-describedby="entity-desc-<?php echo esc_attr( $macm_entity_key ); ?>"
										<?php echo esc_attr( $macm_disabled_attr ); ?>
										<?php checked( $macm_is_available ); ?>>
									<span class="macm-checkbox-custom" aria-hidden="true"></span>
								</div>
								<div class="macm-card-icon">
									<span class="dashicons <?php echo esc_attr( $macm_icon_class ); ?>"></span>
								</div>
								<div class="macm-card-content">
									<span class="macm-card-title">
										<?php echo esc_html( $macm_entity['label'] ); ?>
										<?php if ( $macm_is_premium && ! $macm_is_available ) : ?>
											<span class="dashicons dashicons-lock macm-lock-icon" title="<?php esc_attr_e( 'Premium feature', 'martial-arts-club-manager' ); ?>"></span>
										<?php endif; ?>
									</span>
									<span class="macm-card-desc" id="entity-desc-<?php echo esc_attr( $macm_entity_key ); ?>"><?php echo esc_html( $macm_entity['description'] ); ?></span>
								</div>
							</label>
						<?php endforeach; ?>
					</div>
				</div>
			<?php endforeach; ?>
		</div>

		<!-- Export Button -->
		<div class="macm-form-actions">
			<button type="submit" class="button button-primary button-hero macm-export-btn">
				<span class="dashicons dashicons-download"></span>
				<?php esc_html_e( 'Download Export', 'martial-arts-club-manager' ); ?>
			</button>
		</div>
	</form>
</div>
